/* tcl_usr.c -- Tcl interface to libusr, the Userspace Routing library.

   The "rtable" and "route" Tcl commands, and the "rtable" Tcl object type
   are implemented here.

   Copyright (C) 2007, 2008 Eloy Paris

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"
#include "util-tcl.h"

/********************************************************************
 *		     The "rtable" Tcl object type                   *
 ********************************************************************/

/* Forward declarations */
static void update_string(Tcl_Obj *);
static void free_int_rep(Tcl_Obj *);
static int set_from_any(Tcl_Interp *interp, Tcl_Obj *objPtr);
static void dup_int_rep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr);

Tcl_ObjType rtable_obj = {
    .name = TCL_RTABLE_TYPE_NAME,
    .freeIntRepProc = &free_int_rep,
    .dupIntRepProc = &dup_int_rep,
    .updateStringProc = &update_string,
    .setFromAnyProc = &set_from_any
};

static int
set_from_any(Tcl_Interp *interp _U_, Tcl_Obj *objPtr _U_)
{
    return TCL_OK;
}

static void
dup_int_rep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr)
{
    struct rtable *old, *new;

    old = srcPtr->internalRep.otherValuePtr;
    new = (struct rtable *) ckalloc(sizeof(struct rtable) );
    new->type = old->type;
    new->nentries = old->nentries;
    new->entries = (struct route_entry *) ckalloc(new->nentries
					    *sizeof(struct route_entry) );
    memcpy(new->entries, old->entries,
	   new->nentries*sizeof(struct route_entry) );

    dupPtr->typePtr = &rtable_obj;
    dupPtr->internalRep.otherValuePtr = new;
}

static void
update_string(Tcl_Obj *rtable_obj)
{
    char buffer[256];
    size_t len;
    struct rtable *rtable;
    static const char *rtable_types[] = {
	[RTABLE_IPV4] = "IPv4",
	[RTABLE_IPV6] = "IPv6"
    };

    rtable = rtable_obj->internalRep.otherValuePtr;

    snprintf(buffer, sizeof(buffer), "%s routing table; %d %s",
	     rtable_types[rtable->type], rtable->nentries,
	     rtable->nentries == 1 ? "entry" : "entries");
    len = strlen(buffer);

    rtable_obj->bytes = ckalloc(len + 1);
    if (!rtable_obj->bytes)
	return;

    rtable_obj->length = len;
    strlcpy(rtable_obj->bytes, buffer, len + 1);
}

static void
free_int_rep(Tcl_Obj *obj)
{
    struct rtable *rtable;

    rtable = obj->internalRep.otherValuePtr;
    rtable_destroy(rtable);
}

Tcl_Obj *
Tcl_NewRTableObj(void)
{
    Tcl_Obj *obj;
    struct rtable *rtable;

    obj = Tcl_NewObj();

    rtable = rtable_new(RTABLE_IPV4, 1);
    if (rtable == NULL)
	return NULL;

    obj->bytes = NULL;
    obj->typePtr = &rtable_obj;
    obj->internalRep.otherValuePtr = rtable;

    return obj;
}

/********************************************************************
 *			 rtable sub-commands                        *
 ********************************************************************/

static int
tcl_rtable_new(Tcl_Interp *interp, int argc, Tcl_Obj * const *objv _U_)
{
    Tcl_Obj *rtable_obj;

    if (argc != 1) {
	nexp_error(interp, "usage: rtable new");
	return TCL_ERROR;
    }

    rtable_obj = Tcl_NewRTableObj();
    if (rtable_obj == NULL)
	return TCL_ERROR;

    Tcl_SetObjResult(interp, rtable_obj);

    return TCL_OK;
}

/*
 * XXX - is there a need for a "rtable delete" command? Why not just do
 * a regular Tcl "unset <varname>"??? Perhaps it's just for consistency?
 */
static int
tcl_rtable_delete(Tcl_Interp *interp, int argc, Tcl_Obj * const *objv)
{
    Tcl_Obj *obj;

    if (argc != 2) {
	nexp_error(interp, "usage: rtable delete <rtable>");
	return TCL_ERROR;
    }

    obj = Tcl_ObjGetVar2(interp, objv[1], NULL, TCL_LEAVE_ERR_MSG);
    if (!obj)
	return TCL_ERROR;

    if (!obj->typePtr || strcmp(TCL_RTABLE_TYPE_NAME, obj->typePtr->name) ) {
	nexp_error(interp, "\"%s\" is not an rtable object",
		   Tcl_GetString(objv[1]) );
	return TCL_ERROR;
    }

    return Tcl_UnsetVar(interp, Tcl_GetString(objv[1]), TCL_LEAVE_ERR_MSG);
}

/********************************************************************
 *				rtable                              *
 ********************************************************************/

static int
NExp_RTableObjCmd(ClientData clientData _U_, Tcl_Interp *interp, int argc,
		  Tcl_Obj * const *objv)
{
    int index, retval;
    static const char *subcmds[] = {
	"new", "delete", NULL
    };
    enum subcmds {
	SUBCMD_NEW, SUBCMD_DELETE
    };

    if (Tcl_GetIndexFromObj(interp, objv[1], subcmds, "subcmd", 0, &index)
	!= TCL_OK)
	return TCL_ERROR;

    switch ( (enum subcmds) index) {
    case SUBCMD_NEW:
	retval = tcl_rtable_new(interp, argc - 1, &objv[1]);
	break;
    case SUBCMD_DELETE:
	retval = tcl_rtable_delete(interp, argc - 1, &objv[1]);
	break;
    }

    return retval;
}

/********************************************************************
 *			   ip sub-commands                          *
 ********************************************************************/

static int
tcl_ip_link(Tcl_Interp *interp, int argc, Tcl_Obj * const *objv)
{
    if (argc == 1) {
	/* Just print the interface table */
	iftable_print(NULL);
	return TCL_OK;
    }

    return TCL_OK;
}

static int
tcl_ip_address(Tcl_Interp *interp, int argc, Tcl_Obj * const *objv)
{
    if (argc == 1) {
	/* Just print the interface table */
	iftable_print(NULL);
	return TCL_OK;
    }

    return TCL_OK;
}

static int
tcl_ip_route(Tcl_Interp *interp, int argc, Tcl_Obj * const *objv)
{
    if (argc == 1) {
	 /* If no arguments then just print the system routing table. */
	 rtable_print(NULL);
	 return TCL_OK;
    }

    return TCL_OK;
}

/********************************************************************
 *				  ip                                *
 ********************************************************************/

static int
NExp_IPObjCmd(ClientData clientData _U_, Tcl_Interp *interp, int argc,
	      Tcl_Obj * const *objv)
{
    int index, retval;
    static const char *subcmds[] = {
	"address", "link", "route", NULL
    };
    enum subcmds {
	SUBCMD_LINK, SUBCMD_ADDRESS, SUBCMD_ROUTE
    };

    if (argc == 1) {
	/* XXX - need to print out usage information */
	nexp_error(interp, "wrong # args");
	return TCL_ERROR;
    }

    if (Tcl_GetIndexFromObj(interp, objv[1], subcmds, "subcmd", 0, &index)
	!= TCL_OK)
	return TCL_ERROR;

    switch ( (enum subcmds) index) {
    case SUBCMD_LINK:
	retval = tcl_ip_link(interp, argc - 1, &objv[1]);
	break;
    case SUBCMD_ADDRESS:
	retval = tcl_ip_address(interp, argc - 1, &objv[1]);
	break;
    case SUBCMD_ROUTE:
	retval = tcl_ip_route(interp, argc - 1, &objv[1]);
	break;
    }

    return retval;
}

static struct nexp_cmd_data cmd_data[] = {
    {"rtable", NExp_RTableObjCmd, NULL, 0, 0},
    {"ip", NExp_IPObjCmd, NULL, 0, 0},

    {NULL, NULL, NULL, 0, 0}
};

void
nexp_init_usr_cmds(Tcl_Interp *interp)
{
    Tcl_RegisterObjType(&rtable_obj);

    usr_init();

    nexp_create_commands(interp, cmd_data);
}
