#include "netwibtest.h"

/*-------------------------------------------------------------*/
static netwib_err netwib_uint64_init_uint32_test(void)
{
  netwib_uint64 ui64;

  puts("++++----\nTest of netwib_uint64_init_uint32");

  netwib_er(netwib_uint64_init_uint32(0, &ui64));
  netwib_er(netwibtest_uint64_content(ui64, "0000000000000000"));
  netwib_er(netwib_uint64_init_uint32(0xFFFFFFFFu, &ui64));
  netwib_er(netwibtest_uint64_content(ui64, "00000000FFFFFFFF"));

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_uint64_init_uintmax_test(void)
{
  netwib_uint64 ui64;

  puts("++++----\nTest of netwib_uint64_init_uintmax");

  netwib_er(netwib_uint64_init_uintmax(0, &ui64));
  netwib_er(netwibtest_uint64_content(ui64, "0000000000000000"));
  netwib_er(netwib_uint64_init_uintmax(0xFFFFFFFFu, &ui64));
  netwib_er(netwibtest_uint64_content(ui64, "00000000FFFFFFFF"));
#if NETWIB_INTMAX_BITS == 64
  netwib_er(netwib_uint64_init_uintmax(NETWIB_UINT_LL(0xFFFFFFFFFFFFFFFF),
                                       &ui64));
  netwib_er(netwibtest_uint64_content(ui64, "FFFFFFFFFFFFFFFF"));
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_uint64_init_uintptr_test(void)
{
  netwib_uint64 ui64;

  puts("++++----\nTest of netwib_uint64_init_uintptr");

  netwib_er(netwib_uint64_init_uintptr(0, &ui64));
  netwib_er(netwibtest_uint64_content(ui64, "0000000000000000"));
  netwib_er(netwib_uint64_init_uintptr(0xFFFFFFFFu, &ui64));
  netwib_er(netwibtest_uint64_content(ui64, "00000000FFFFFFFF"));
#if NETWIB_INTPTR_BITS == 64
  netwib_er(netwib_uint64_init_uintptr(NETWIB_UINT_LL(0xFFFFFFFFFFFFFFFF),
                                       &ui64));
  netwib_er(netwibtest_uint64_content(ui64, "FFFFFFFFFFFFFFFF"));
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_int64_init_int32_test(void)
{
  netwib_int64 i64;

  puts("++++----\nTest of netwib_int64_init_int32");

  netwib_er(netwib_int64_init_int32(0, &i64));
  netwib_er(netwibtest_int64_content(i64, "0000000000000000"));
  netwib_er(netwib_int64_init_int32(0xFFFFFFFFu, &i64));
  netwib_er(netwibtest_int64_content(i64, "FFFFFFFFFFFFFFFF"));
  netwib_er(netwib_int64_init_int32(-3, &i64));
  netwib_er(netwibtest_int64_content(i64, "FFFFFFFFFFFFFFFD"));

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_int64_init_intmax_test(void)
{
  netwib_int64 i64;

  puts("++++----\nTest of netwib_int64_init_intmax");

  netwib_er(netwib_int64_init_intmax(0, &i64));
  netwib_er(netwibtest_int64_content(i64, "0000000000000000"));
  netwib_er(netwib_int64_init_intmax(0x7FFFFFFF, &i64));
  netwib_er(netwibtest_int64_content(i64, "000000007FFFFFFF"));
  netwib_er(netwib_int64_init_intmax(-3, &i64));
  netwib_er(netwibtest_int64_content(i64, "FFFFFFFFFFFFFFFD"));
#if NETWIB_INTMAX_BITS == 64
  netwib_er(netwib_int64_init_intmax(NETWIB_UINT_LL(0xFFFFFFFFFF), &i64));
  netwib_er(netwibtest_int64_content(i64, "000000FFFFFFFFFF"));
#endif
  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_int64_init_intptr_test(void)
{
  netwib_int64 i64;

  puts("++++----\nTest of netwib_int64_init_intptr");

  netwib_er(netwib_int64_init_intptr(0, &i64));
  netwib_er(netwibtest_int64_content(i64, "0000000000000000"));
  netwib_er(netwib_int64_init_intptr(0x7FFFFFFF, &i64));
  netwib_er(netwibtest_int64_content(i64, "000000007FFFFFFF"));
  netwib_er(netwib_int64_init_intptr(-3, &i64));
  netwib_er(netwibtest_int64_content(i64, "FFFFFFFFFFFFFFFD"));
#if NETWIB_INTPTR_BITS == 64
  netwib_er(netwib_int64_init_intptr(NETWIB_UINT_LL(0xFFFFFFFFFF), &i64));
  netwib_er(netwibtest_int64_content(i64, "000000FFFFFFFFFF"));
#endif
  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_uint32_init_uint64_test(void)
{
  netwib_uint64 ui64;
  netwib_uint32 ui32;

  puts("++++----\nTest of netwib_uint32_init_uint64");

  netwib_er(netwibtest_uint64_init("0000000000000000", &ui64));
  netwib_er(netwib_uint32_init_uint64(ui64, &ui32));
  if (ui32 != 0) {
    return(NETWIBTEST_ERR_1);
  }
  netwib_er(netwibtest_uint64_init("00000000FFFFFFFF", &ui64));
  netwib_er(netwib_uint32_init_uint64(ui64, &ui32));
  if (ui32 != 0xFFFFFFFFu) {
    return(NETWIBTEST_ERR_1);
  }
  netwib_er(netwibtest_uint64_init("00010000FFFFFFFF", &ui64));
  netwib_eir(netwib_uint32_init_uint64(ui64, &ui32), NETWIB_ERR_PATOOHIGH);

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_uintmax_init_uint64_test(void)
{
  netwib_uint64 ui64;
  netwib_uintmax uimax;

  puts("++++----\nTest of netwib_uintmax_init_uint64");

  netwib_er(netwibtest_uint64_init("0000000000000000", &ui64));
  netwib_er(netwib_uintmax_init_uint64(ui64, &uimax));
  if (uimax != 0) {
    return(NETWIBTEST_ERR_1);
  }
  netwib_er(netwibtest_uint64_init("00000000FFFFFFFF", &ui64));
  netwib_er(netwib_uintmax_init_uint64(ui64, &uimax));
  if (uimax != 0xFFFFFFFFu) {
    return(NETWIBTEST_ERR_1);
  }
  netwib_er(netwibtest_uint64_init("00010000FFFFFFFF", &ui64));
#if NETWIB_INTMAX_BITS == 64
  netwib_er(netwib_uintmax_init_uint64(ui64, &uimax));
  if (uimax != NETWIB_UINT_LL(0x10000FFFFFFFF)) {
    return(NETWIBTEST_ERR_1);
  }
#else
  netwib_eir(netwib_uintmax_init_uint64(ui64, &uimax), NETWIB_ERR_PATOOHIGH);
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_uintptr_init_uint64_test(void)
{
  netwib_uint64 ui64;
  netwib_uintptr uiptr;

  puts("++++----\nTest of netwib_uintptr_init_uint64");

  netwib_er(netwibtest_uint64_init("0000000000000000", &ui64));
  netwib_er(netwib_uintptr_init_uint64(ui64, &uiptr));
  if (uiptr != 0) {
    return(NETWIBTEST_ERR_1);
  }
  netwib_er(netwibtest_uint64_init("00000000FFFFFFFF", &ui64));
  netwib_er(netwib_uintptr_init_uint64(ui64, &uiptr));
  if (uiptr != 0xFFFFFFFFu) {
    return(NETWIBTEST_ERR_1);
  }
  netwib_er(netwibtest_uint64_init("00010000FFFFFFFF", &ui64));
#if NETWIB_INTPTR_BITS == 64
  netwib_er(netwib_uintptr_init_uint64(ui64, &uiptr));
  if (uiptr != NETWIB_UINT_LL(0x10000FFFFFFFF)) {
    return(NETWIBTEST_ERR_1);
  }
#else
  netwib_eir(netwib_uintptr_init_uint64(ui64, &uiptr), NETWIB_ERR_PATOOHIGH);
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_int32_init_int64_test(void)
{
  netwib_int64 i64;
  netwib_int32 i32;

  puts("++++----\nTest of netwib_int32_init_int64");

  netwib_er(netwibtest_int64_init("0000000000000000", &i64));
  netwib_er(netwib_int32_init_int64(i64, &i32));
  if (i32 != 0) {
    return(NETWIBTEST_ERR_1);
  }
  netwib_er(netwibtest_int64_init("000000007FFFFFFF", &i64));
  netwib_er(netwib_int32_init_int64(i64, &i32));
  if (i32 != 0x7FFFFFFF) {
    return(NETWIBTEST_ERR_1);
  }
  netwib_er(netwibtest_int64_init("FFFFFFFFFFFFFFFF", &i64));
  netwib_er(netwib_int32_init_int64(i64, &i32));
  if (i32 != -1) {
    return(NETWIBTEST_ERR_1);
  }
  netwib_er(netwibtest_int64_init("FFFFFFFF80000000", &i64));
  netwib_er(netwib_int32_init_int64(i64, &i32));
  if (i32 != -0x7FFFFFFF-1) {
    return(NETWIBTEST_ERR_1);
  }
  netwib_er(netwibtest_int64_init("00010000FFFFFFFF", &i64));
  netwib_eir(netwib_int32_init_int64(i64, &i32), NETWIB_ERR_PATOOHIGH);
  netwib_er(netwibtest_int64_init("80010000FFFFFFFF", &i64));
  netwib_eir(netwib_int32_init_int64(i64, &i32), NETWIB_ERR_PATOOLOW);

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_intmax_init_int64_test(void)
{
  netwib_int64 i64;
  netwib_intmax imax;

  puts("++++----\nTest of netwib_intmax_init_int64");

  netwib_er(netwibtest_int64_init("0000000000000000", &i64));
  netwib_er(netwib_intmax_init_int64(i64, &imax));
  if (imax != 0) {
    return(NETWIBTEST_ERR_1);
  }
  netwib_er(netwibtest_int64_init("000000007FFFFFFF", &i64));
  netwib_er(netwib_intmax_init_int64(i64, &imax));
  if (imax != 0x7FFFFFFF) {
    return(NETWIBTEST_ERR_1);
  }
  netwib_er(netwibtest_int64_init("FFFFFFFFFFFFFFFF", &i64));
  netwib_er(netwib_intmax_init_int64(i64, &imax));
  if (imax != -1) {
    return(NETWIBTEST_ERR_1);
  }
  netwib_er(netwibtest_int64_init("FFFFFFFF80000000", &i64));
  netwib_er(netwib_intmax_init_int64(i64, &imax));
  if (imax != -0x7FFFFFFF-1) {
    return(NETWIBTEST_ERR_1);
  }
  netwib_er(netwibtest_int64_init("00010000FFFFFFFF", &i64));
#if NETWIB_INTMAX_BITS == 64
  netwib_er(netwib_intmax_init_int64(i64, &imax));
  if (imax != NETWIB_INT_LL(0x10000FFFFFFFF)) {
    return(NETWIBTEST_ERR_1);
  }
#else
  netwib_eir(netwib_intmax_init_int64(i64, &imax), NETWIB_ERR_PATOOHIGH);
#endif
  netwib_er(netwibtest_int64_init("80010000FFFFFFFF", &i64));
#if NETWIB_INTMAX_BITS == 64
  netwib_er(netwib_intmax_init_int64(i64, &imax));
  if (imax != (netwib_intmax)NETWIB_INT_LL(0x80010000FFFFFFFF)) {
    return(NETWIBTEST_ERR_1);
  }
#else
  netwib_eir(netwib_intmax_init_int64(i64, &imax), NETWIB_ERR_PATOOLOW);
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_intptr_init_int64_test(void)
{
  netwib_int64 i64;
  netwib_intptr iptr;

  puts("++++----\nTest of netwib_intptr_init_int64");

  netwib_er(netwibtest_int64_init("0000000000000000", &i64));
  netwib_er(netwib_intptr_init_int64(i64, &iptr));
  if (iptr != 0) {
    return(NETWIBTEST_ERR_1);
  }
  netwib_er(netwibtest_int64_init("000000007FFFFFFF", &i64));
  netwib_er(netwib_intptr_init_int64(i64, &iptr));
  if (iptr != 0x7FFFFFFF) {
    return(NETWIBTEST_ERR_1);
  }
  netwib_er(netwibtest_int64_init("FFFFFFFFFFFFFFFF", &i64));
  netwib_er(netwib_intptr_init_int64(i64, &iptr));
  if (iptr != -1) {
    return(NETWIBTEST_ERR_1);
  }
  netwib_er(netwibtest_int64_init("FFFFFFFF80000000", &i64));
  netwib_er(netwib_intptr_init_int64(i64, &iptr));
  if (iptr != -0x7FFFFFFF-1) {
    return(NETWIBTEST_ERR_1);
  }
  netwib_er(netwibtest_int64_init("00010000FFFFFFFF", &i64));
#if NETWIB_INTPTR_BITS == 64
  netwib_er(netwib_intptr_init_int64(i64, &iptr));
  if (iptr != NETWIB_INT_LL(0x10000FFFFFFFF)) {
    return(NETWIBTEST_ERR_1);
  }
#else
  netwib_eir(netwib_intptr_init_int64(i64, &iptr), NETWIB_ERR_PATOOHIGH);
#endif
  netwib_er(netwibtest_int64_init("80010000FFFFFFFF", &i64));
#if NETWIB_INTPTR_BITS == 64
  netwib_er(netwib_intptr_init_int64(i64, &iptr));
  if (iptr != (netwib_intptr)NETWIB_INT_LL(0x80010000FFFFFFFF)) {
    return(NETWIBTEST_ERR_1);
  }
#else
  netwib_eir(netwib_intptr_init_int64(i64, &iptr), NETWIB_ERR_PATOOLOW);
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_uint64_init_int64_test(void)
{
  netwib_uint64 ui64;
  netwib_int64 i64;

  puts("++++----\nTest of netwib_uint64_init_int64");

  netwib_er(netwibtest_int64_init("0000000000000000", &i64));
  netwib_er(netwib_uint64_init_int64(i64, &ui64));
  netwib_er(netwibtest_uint64_content(ui64, "0000000000000000"));
  netwib_er(netwibtest_int64_init("000000007FFFFFFF", &i64));
  netwib_er(netwib_uint64_init_int64(i64, &ui64));
  netwib_er(netwibtest_uint64_content(ui64, "000000007FFFFFFF"));
  netwib_er(netwibtest_int64_init("FFFFFFFFFFFFFFFF", &i64));
  netwib_eir(netwib_uint64_init_int64(i64, &ui64), NETWIB_ERR_PATOOLOW);

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_int64_init_uint64_test(void)
{
  netwib_uint64 ui64;
  netwib_int64 i64;

  puts("++++----\nTest of netwib_int64_init_uint64");

  netwib_er(netwibtest_uint64_init("0000000000000000", &ui64));
  netwib_er(netwib_int64_init_uint64(ui64, &i64));
  netwib_er(netwibtest_int64_content(i64, "0000000000000000"));
  netwib_er(netwibtest_uint64_init("000000007FFFFFFF", &ui64));
  netwib_er(netwib_int64_init_uint64(ui64, &i64));
  netwib_er(netwibtest_int64_content(i64, "000000007FFFFFFF"));
  netwib_er(netwibtest_uint64_init("FFFFFFFFFFFFFFFF", &ui64));
  netwib_eir(netwib_int64_init_uint64(ui64, &i64), NETWIB_ERR_PATOOHIGH);

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_uint64_add_test(void)
{
  netwib_uint64 ui64a, ui64b, ui64c;

  puts("++++----\nTest of netwib_uint64_add");

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwibtest_uint64_init("1111111111111111", &ui64b));
  netwib_er(netwib_uint64_add(ui64a, ui64b, &ui64c));
  netwib_er(netwibtest_uint64_content(ui64c, "2345678923456789"));

  netwib_er(netwibtest_uint64_init("1234567882345678", &ui64a));
  netwib_er(netwibtest_uint64_init("11111111A1111111", &ui64b));
  netwib_er(netwib_uint64_add(ui64a, ui64b, &ui64c));
  netwib_er(netwibtest_uint64_content(ui64c, "2345678A23456789"));

  netwib_er(netwibtest_uint64_init("A234567882345678", &ui64a));
  netwib_er(netwibtest_uint64_init("B1111111A1111111", &ui64b));
  netwib_er(netwib_uint64_add(ui64a, ui64b, &ui64c));
  netwib_er(netwibtest_uint64_content(ui64c, "5345678A23456789"));

  netwib_er(netwibtest_uint64_init("FFFFFFFFFFFFFFFF", &ui64a));
  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64b));
  netwib_er(netwib_uint64_add(ui64a, ui64b, &ui64c));
  netwib_er(netwibtest_uint64_content(ui64c, "1234567812345677"));

  netwib_er(netwibtest_uint64_init("FFFFFFFFFFFFFFFF", &ui64a));
  netwib_er(netwibtest_uint64_init("FFFFFFFFFFFFFFFF", &ui64b));
  netwib_er(netwib_uint64_add(ui64a, ui64b, &ui64c));
  netwib_er(netwibtest_uint64_content(ui64c, "FFFFFFFFFFFFFFFE"));

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_uint64_sub_test(void)
{
  netwib_uint64 ui64a, ui64b, ui64c;

  puts("++++----\nTest of netwib_uint64_sub");

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwibtest_uint64_init("1111111111111111", &ui64b));
  netwib_er(netwib_uint64_sub(ui64a, ui64b, &ui64c));
  netwib_er(netwibtest_uint64_content(ui64c, "0123456701234567"));

  netwib_er(netwibtest_uint64_init("1234567882345678", &ui64a));
  netwib_er(netwibtest_uint64_init("11111111A1111111", &ui64b));
  netwib_er(netwib_uint64_sub(ui64a, ui64b, &ui64c));
  netwib_er(netwibtest_uint64_content(ui64c, "01234566E1234567"));

  netwib_er(netwibtest_uint64_init("A234567882345678", &ui64a));
  netwib_er(netwibtest_uint64_init("B1111111A1111111", &ui64b));
  netwib_er(netwib_uint64_sub(ui64a, ui64b, &ui64c));
  netwib_er(netwibtest_uint64_content(ui64c, "F1234566E1234567"));

  netwib_er(netwibtest_uint64_init("FFFFFFFFFFFFFFFF", &ui64a));
  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64b));
  netwib_er(netwib_uint64_sub(ui64a, ui64b, &ui64c));
  netwib_er(netwibtest_uint64_content(ui64c, "EDCBA987EDCBA987"));

  netwib_er(netwibtest_uint64_init("FFFFFFFFFFFFFFFF", &ui64a));
  netwib_er(netwibtest_uint64_init("FFFFFFFFFFFFFFFF", &ui64b));
  netwib_er(netwib_uint64_sub(ui64a, ui64b, &ui64c));
  netwib_er(netwibtest_uint64_content(ui64c, "0000000000000000"));

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_uint64_mul_test(void)
{
  netwib_uint64 ui64a, ui64b, ui64c;

  puts("++++----\nTest of netwib_uint64_mul");

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwibtest_uint64_init("0000000000000001", &ui64b));
  netwib_er(netwib_uint64_mul(ui64a, ui64b, &ui64c));
  netwib_er(netwibtest_uint64_content(ui64c, "1234567812345678"));

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwibtest_uint64_init("0000000000000002", &ui64b));
  netwib_er(netwib_uint64_mul(ui64a, ui64b, &ui64c));
  netwib_er(netwibtest_uint64_content(ui64c, "2468ACF02468ACF0"));

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwibtest_uint64_init("1111111111111111", &ui64b));
  netwib_er(netwib_uint64_mul(ui64a, ui64b, &ui64c));
  netwib_er(netwibtest_uint64_content(ui64c, "CB961C5E652FB5F8"));

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwibtest_uint64_init("FFFFFFFFFFFFFFFF", &ui64b));
  netwib_er(netwib_uint64_mul(ui64a, ui64b, &ui64c));
  netwib_er(netwibtest_uint64_content(ui64c, "EDCBA987EDCBA988"));

  netwib_er(netwibtest_uint64_init("FFFFFFFFFFFFFFFE", &ui64a));
  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64b));
  netwib_er(netwib_uint64_mul(ui64a, ui64b, &ui64c));
  netwib_er(netwibtest_uint64_content(ui64c, "DB97530FDB975310"));

  netwib_er(netwibtest_uint64_init("FFFFFFFFFFFFFFFF", &ui64a));
  netwib_er(netwibtest_uint64_init("FFFFFFFFFFFFFFFF", &ui64b));
  netwib_er(netwib_uint64_mul(ui64a, ui64b, &ui64c));
  netwib_er(netwibtest_uint64_content(ui64c, "0000000000000001"));

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_uint64_div_test(void)
{
  netwib_uint64 ui64a, ui64b, ui64c, ui64d;

  puts("++++----\nTest of netwib_uint64_div");

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64b));
  netwib_er(netwib_uint64_div(ui64a, ui64b, &ui64c, &ui64d));
  netwib_er(netwibtest_uint64_content(ui64c, "0000000000000001"));
  netwib_er(netwibtest_uint64_content(ui64d, "0000000000000000"));

  netwib_er(netwibtest_uint64_init("0000000000000010", &ui64a));
  netwib_er(netwibtest_uint64_init("0000000000000004", &ui64b));
  netwib_er(netwib_uint64_div(ui64a, ui64b, &ui64c, &ui64d));
  netwib_er(netwibtest_uint64_content(ui64c, "0000000000000004"));
  netwib_er(netwibtest_uint64_content(ui64d, "0000000000000000"));

  netwib_er(netwibtest_uint64_init("0000000000384FB9", &ui64a));
  netwib_er(netwibtest_uint64_init("0000000000000A01", &ui64b));
  netwib_er(netwib_uint64_div(ui64a, ui64b, &ui64c, &ui64d));
  netwib_er(netwibtest_uint64_content(ui64c, "00000000000005A1"));
  netwib_er(netwibtest_uint64_content(ui64d, "0000000000000018"));

  netwib_er(netwibtest_uint64_init("0000000000384FB9", &ui64a));
  netwib_er(netwibtest_uint64_init("0000000000000A00", &ui64b));
  netwib_er(netwib_uint64_div(ui64a, ui64b, &ui64c, &ui64d));
  netwib_er(netwibtest_uint64_content(ui64c, "00000000000005A1"));
  netwib_er(netwibtest_uint64_content(ui64d, "00000000000005B9"));

  netwib_er(netwibtest_uint64_init("2468ACF02468ACF0", &ui64a));
  netwib_er(netwibtest_uint64_init("0000000000000002", &ui64b));
  netwib_er(netwib_uint64_div(ui64a, ui64b, &ui64c, &ui64d));
  netwib_er(netwibtest_uint64_content(ui64c, "1234567812345678"));
  netwib_er(netwibtest_uint64_content(ui64d, "0000000000000000"));

  netwib_er(netwibtest_uint64_init("2468ACF02468ACF1", &ui64a));
  netwib_er(netwibtest_uint64_init("0000000000000002", &ui64b));
  netwib_er(netwib_uint64_div(ui64a, ui64b, &ui64c, &ui64d));
  netwib_er(netwibtest_uint64_content(ui64c, "1234567812345678"));
  netwib_er(netwibtest_uint64_content(ui64d, "0000000000000001"));

  netwib_er(netwibtest_uint64_init("EDCBA987EDCBA988", &ui64a));
  netwib_er(netwibtest_uint64_init("FFFFFFFFFFFFFFFF", &ui64b));
  netwib_er(netwib_uint64_div(ui64a, ui64b, &ui64c, &ui64d));
  netwib_er(netwibtest_uint64_content(ui64c, "0000000000000000"));
  netwib_er(netwibtest_uint64_content(ui64d, "EDCBA987EDCBA988"));

  netwib_er(netwibtest_uint64_init("FFFFFFFFFFFFFFFE", &ui64a));
  netwib_er(netwibtest_uint64_init("FFFFFFFFFFFFFFFF", &ui64b));
  netwib_er(netwib_uint64_div(ui64a, ui64b, &ui64c, &ui64d));
  netwib_er(netwibtest_uint64_content(ui64c, "0000000000000000"));
  netwib_er(netwibtest_uint64_content(ui64d, "FFFFFFFFFFFFFFFE"));

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_int64_div_test(void)
{
  netwib_int64 i64a, i64b, i64c, i64d;

  puts("++++----\nTest of netwib_int64_div");

  netwib_er(netwibtest_int64_init("1234567812345678", &i64a));
  netwib_er(netwibtest_int64_init("1234567812345678", &i64b));
  netwib_er(netwib_int64_div(i64a, i64b, &i64c, &i64d));
  netwib_er(netwibtest_int64_content(i64c, "0000000000000001"));
  netwib_er(netwibtest_int64_content(i64d, "0000000000000000"));

  netwib_er(netwibtest_int64_init("2468ACF02468ACF0", &i64a));
  netwib_er(netwibtest_int64_init("0000000000000002", &i64b));
  netwib_er(netwib_int64_div(i64a, i64b, &i64c, &i64d));
  netwib_er(netwibtest_int64_content(i64c, "1234567812345678"));
  netwib_er(netwibtest_int64_content(i64d, "0000000000000000"));

  netwib_er(netwibtest_int64_init("2468ACF02468ACF1", &i64a));
  netwib_er(netwibtest_int64_init("0000000000000002", &i64b));
  netwib_er(netwib_int64_div(i64a, i64b, &i64c, &i64d));
  netwib_er(netwibtest_int64_content(i64c, "1234567812345678"));
  netwib_er(netwibtest_int64_content(i64d, "0000000000000001"));

  netwib_er(netwibtest_int64_init("F468ACF02468ACF1", &i64a));
  netwib_er(netwibtest_int64_init("0000000000000007", &i64b));
  netwib_er(netwib_int64_div(i64a, i64b, &i64c, &i64d));
  netwib_er(netwibtest_int64_content(i64c, "FE5818B4977CAAFE"));
  netwib_er(netwibtest_int64_content(i64d, "FFFFFFFFFFFFFFFF"));

  netwib_er(netwibtest_int64_init("F468ACF02468ACF0", &i64a));
  netwib_er(netwibtest_int64_init("0000000000000007", &i64b));
  netwib_er(netwib_int64_div(i64a, i64b, &i64c, &i64d));
  netwib_er(netwibtest_int64_content(i64c, "FE5818B4977CAAFE"));
  netwib_er(netwibtest_int64_content(i64d, "FFFFFFFFFFFFFFFE"));

  netwib_er(netwibtest_int64_init("EDCBA987EDCBA988", &i64a));
  netwib_er(netwibtest_int64_init("FFFFFFFFFFFFFFFF", &i64b));
  netwib_er(netwib_int64_div(i64a, i64b, &i64c, &i64d));
  netwib_er(netwibtest_int64_content(i64c, "1234567812345678"));
  netwib_er(netwibtest_int64_content(i64d, "0000000000000000"));

  netwib_er(netwibtest_int64_init("EDCBA987EDCBA988", &i64a));
  netwib_er(netwibtest_int64_init("FFFFFFFFFFFFFFF0", &i64b));
  netwib_er(netwib_int64_div(i64a, i64b, &i64c, &i64d));
  netwib_er(netwibtest_int64_content(i64c, "0123456781234567"));
  netwib_er(netwibtest_int64_content(i64d, "FFFFFFFFFFFFFFF8"));

  netwib_er(netwibtest_int64_init("3DCBA987EDCBA988", &i64a));
  netwib_er(netwibtest_int64_init("FFFFFFFFFFFFFFFF", &i64b));
  netwib_er(netwib_int64_div(i64a, i64b, &i64c, &i64d));
  netwib_er(netwibtest_int64_content(i64c, "C234567812345678"));
  netwib_er(netwibtest_int64_content(i64d, "0000000000000000"));

  netwib_er(netwibtest_int64_init("3DCBA987EDCBA988", &i64a));
  netwib_er(netwibtest_int64_init("FFFFFFFFFFFFFFEF", &i64b));
  netwib_er(netwib_int64_div(i64a, i64b, &i64c, &i64d));
  netwib_er(netwibtest_int64_content(i64c, "FC5D6E7F8899AABC"));
  netwib_er(netwibtest_int64_content(i64d, "0000000000000004"));

  netwib_er(netwibtest_int64_init("FFFFFFFFFFFFFFFE", &i64a));
  netwib_er(netwibtest_int64_init("FFFFFFFFFFFFFFFF", &i64b));
  netwib_er(netwib_int64_div(i64a, i64b, &i64c, &i64d));
  netwib_er(netwibtest_int64_content(i64c, "0000000000000002"));
  netwib_er(netwibtest_int64_content(i64d, "0000000000000000"));

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_uint64_inc_test(void)
{
  netwib_uint64 ui64a;

  puts("++++----\nTest of netwib_uint64_inc");

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwib_uint64_inc(&ui64a));
  netwib_er(netwibtest_uint64_content(ui64a, "1234567812345679"));

  netwib_er(netwibtest_uint64_init("12345678FFFFFFFF", &ui64a));
  netwib_er(netwib_uint64_inc(&ui64a));
  netwib_er(netwibtest_uint64_content(ui64a, "1234567900000000"));

  netwib_er(netwibtest_uint64_init("FFFFFFFFFFFFFFFF", &ui64a));
  netwib_er(netwib_uint64_inc(&ui64a));
  netwib_er(netwibtest_uint64_content(ui64a, "0000000000000000"));

  netwib_er(netwibtest_uint64_init("FFFFFFFFFFFFFFFE", &ui64a));
  netwib_er(netwib_uint64_inc(&ui64a));
  netwib_er(netwibtest_uint64_content(ui64a, "FFFFFFFFFFFFFFFF"));

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_uint64_dec_test(void)
{
  netwib_uint64 ui64a;

  puts("++++----\nTest of netwib_uint64_dec");

  netwib_er(netwibtest_uint64_init("1234567812345679", &ui64a));
  netwib_er(netwib_uint64_dec(&ui64a));
  netwib_er(netwibtest_uint64_content(ui64a, "1234567812345678"));

  netwib_er(netwibtest_uint64_init("1234567900000000", &ui64a));
  netwib_er(netwib_uint64_dec(&ui64a));
  netwib_er(netwibtest_uint64_content(ui64a, "12345678FFFFFFFF"));

  netwib_er(netwibtest_uint64_init("0000000000000000", &ui64a));
  netwib_er(netwib_uint64_dec(&ui64a));
  netwib_er(netwibtest_uint64_content(ui64a, "FFFFFFFFFFFFFFFF"));

  netwib_er(netwibtest_uint64_init("FFFFFFFFFFFFFFFF", &ui64a));
  netwib_er(netwib_uint64_dec(&ui64a));
  netwib_er(netwibtest_uint64_content(ui64a, "FFFFFFFFFFFFFFFE"));

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_uint64_cmp_test(void)
{
  netwib_uint64 ui64a, ui64b;
  netwib_cmp cmp;

  puts("++++----\nTest of netwib_uint64_cmp");

  netwib_er(netwibtest_uint64_init("1234567812345679", &ui64a));
  netwib_er(netwibtest_uint64_init("1234567812345679", &ui64b));
  netwib_er(netwib_uint64_cmp(ui64a, ui64b, &cmp));
  if (cmp != NETWIB_CMP_EQ) {
    return(NETWIBTEST_ERR_3);
  }
  netwib_er(netwibtest_uint64_init("123456781234567A", &ui64b));
  netwib_er(netwib_uint64_cmp(ui64a, ui64b, &cmp));
  if (cmp != NETWIB_CMP_LT) {
    return(NETWIBTEST_ERR_3);
  }
  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64b));
  netwib_er(netwib_uint64_cmp(ui64a, ui64b, &cmp));
  if (cmp != NETWIB_CMP_GT) {
    return(NETWIBTEST_ERR_3);
  }

  netwib_er(netwibtest_uint64_init("F234567812345679", &ui64a));
  netwib_er(netwibtest_uint64_init("F234567812345679", &ui64b));
  netwib_er(netwib_uint64_cmp(ui64a, ui64b, &cmp));
  if (cmp != NETWIB_CMP_EQ) {
    return(NETWIBTEST_ERR_3);
  }
  netwib_er(netwibtest_uint64_init("F23456781234567A", &ui64b));
  netwib_er(netwib_uint64_cmp(ui64a, ui64b, &cmp));
  if (cmp != NETWIB_CMP_LT) {
    return(NETWIBTEST_ERR_3);
  }
  netwib_er(netwibtest_uint64_init("F234567812345678", &ui64b));
  netwib_er(netwib_uint64_cmp(ui64a, ui64b, &cmp));
  if (cmp != NETWIB_CMP_GT) {
    return(NETWIBTEST_ERR_3);
  }

  netwib_er(netwibtest_uint64_init("1234567812345679", &ui64a));
  netwib_er(netwibtest_uint64_init("F234567812345679", &ui64b));
  netwib_er(netwib_uint64_cmp(ui64a, ui64b, &cmp));
  if (cmp != NETWIB_CMP_LT) {
    return(NETWIBTEST_ERR_3);
  }

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_int64_cmp_test(void)
{
  netwib_int64 i64a, i64b;
  netwib_cmp cmp;

  puts("++++----\nTest of netwib_int64_cmp");

  netwib_er(netwibtest_int64_init("1234567812345679", &i64a));
  netwib_er(netwibtest_int64_init("1234567812345679", &i64b));
  netwib_er(netwib_int64_cmp(i64a, i64b, &cmp));
  if (cmp != NETWIB_CMP_EQ) {
    return(NETWIBTEST_ERR_3);
  }
  netwib_er(netwibtest_int64_init("123456781234567A", &i64b));
  netwib_er(netwib_int64_cmp(i64a, i64b, &cmp));
  if (cmp != NETWIB_CMP_LT) {
    return(NETWIBTEST_ERR_3);
  }
  netwib_er(netwibtest_int64_init("1234567812345678", &i64b));
  netwib_er(netwib_int64_cmp(i64a, i64b, &cmp));
  if (cmp != NETWIB_CMP_GT) {
    return(NETWIBTEST_ERR_3);
  }

  netwib_er(netwibtest_int64_init("F234567812345679", &i64a));
  netwib_er(netwibtest_int64_init("F234567812345679", &i64b));
  netwib_er(netwib_int64_cmp(i64a, i64b, &cmp));
  if (cmp != NETWIB_CMP_EQ) {
    return(NETWIBTEST_ERR_3);
  }
  netwib_er(netwibtest_int64_init("F23456781234567A", &i64b));
  netwib_er(netwib_int64_cmp(i64a, i64b, &cmp));
  if (cmp != NETWIB_CMP_LT) {
    return(NETWIBTEST_ERR_3);
  }
  netwib_er(netwibtest_int64_init("F234567812345678", &i64b));
  netwib_er(netwib_int64_cmp(i64a, i64b, &cmp));
  if (cmp != NETWIB_CMP_GT) {
    return(NETWIBTEST_ERR_3);
  }

  netwib_er(netwibtest_int64_init("1234567812345679", &i64a));
  netwib_er(netwibtest_int64_init("F234567812345679", &i64b));
  netwib_er(netwib_int64_cmp(i64a, i64b, &cmp));
  if (cmp != NETWIB_CMP_GT) {
    return(NETWIBTEST_ERR_3);
  }

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_int64_neg_test(void)
{
  netwib_int64 i64a, i64b;

  puts("++++----\nTest of netwib_int64_neg");

  netwib_er(netwibtest_int64_init("FFFFFFFFFFFFFFFF", &i64a));
  netwib_er(netwib_int64_neg(i64a, &i64b));
  netwib_er(netwibtest_int64_content(i64b, "0000000000000001"));

  netwib_er(netwibtest_int64_init("FFFFFFFFFFFFFFFE", &i64a));
  netwib_er(netwib_int64_neg(i64a, &i64b));
  netwib_er(netwibtest_int64_content(i64b, "0000000000000002"));

  netwib_er(netwibtest_int64_init("0000000000000002", &i64a));
  netwib_er(netwib_int64_neg(i64a, &i64b));
  netwib_er(netwibtest_int64_content(i64b, "FFFFFFFFFFFFFFFE"));

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_uint64_shl_test(void)
{
  netwib_uint64 ui64a, ui64b;

  puts("++++----\nTest of netwib_uint64_shl");

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwib_uint64_shl(ui64a, 1, &ui64b));
  netwib_er(netwibtest_uint64_content(ui64b, "2468ACF02468ACF0"));

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwib_uint64_shl(ui64a, 2, &ui64b));
  netwib_er(netwibtest_uint64_content(ui64b, "48D159E048D159E0"));

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwib_uint64_shl(ui64a, 31, &ui64b));
  netwib_er(netwibtest_uint64_content(ui64b, "091A2B3C00000000"));

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwib_uint64_shl(ui64a, 32, &ui64b));
  netwib_er(netwibtest_uint64_content(ui64b, "1234567800000000"));

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwib_uint64_shl(ui64a, 33, &ui64b));
  netwib_er(netwibtest_uint64_content(ui64b, "2468ACF000000000"));

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwib_uint64_shl(ui64a, 4, &ui64b));
  netwib_er(netwibtest_uint64_content(ui64b, "2345678123456780"));

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwib_uint64_shl(ui64a, 60, &ui64b));
  netwib_er(netwibtest_uint64_content(ui64b, "8000000000000000"));

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_uint64_shr_test(void)
{
  netwib_uint64 ui64a, ui64b;

  puts("++++----\nTest of netwib_uint64_shr");

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwib_uint64_shr(ui64a, 1, &ui64b));
  netwib_er(netwibtest_uint64_content(ui64b, "091A2B3C091A2B3C"));

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwib_uint64_shr(ui64a, 4, &ui64b));
  netwib_er(netwibtest_uint64_content(ui64b, "0123456781234567"));

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwib_uint64_shr(ui64a, 31, &ui64b));
  netwib_er(netwibtest_uint64_content(ui64b, "000000002468ACF0"));

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwib_uint64_shr(ui64a, 32, &ui64b));
  netwib_er(netwibtest_uint64_content(ui64b, "0000000012345678"));

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwib_uint64_shr(ui64a, 33, &ui64b));
  netwib_er(netwibtest_uint64_content(ui64b, "00000000091A2B3C"));

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwib_uint64_shr(ui64a, 60, &ui64b));
  netwib_er(netwibtest_uint64_content(ui64b, "0000000000000001"));

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_uint64_and_test(void)
{
  netwib_uint64 ui64a, ui64b, ui64c;

  puts("++++----\nTest of netwib_uint64_and");

  netwib_er(netwibtest_uint64_init("1234067812345678", &ui64a));
  netwib_er(netwibtest_uint64_init("120C5678123F5678", &ui64b));
  netwib_er(netwib_uint64_and(ui64a, ui64b, &ui64c));
  netwib_er(netwibtest_uint64_content(ui64c, "1204067812345678"));

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_uint64_or_test(void)
{
  netwib_uint64 ui64a, ui64b, ui64c;

  puts("++++----\nTest of netwib_uint64_or");

  netwib_er(netwibtest_uint64_init("1234067812345678", &ui64a));
  netwib_er(netwibtest_uint64_init("120C5678123F5678", &ui64b));
  netwib_er(netwib_uint64_or(ui64a, ui64b, &ui64c));
  netwib_er(netwibtest_uint64_content(ui64c, "123C5678123F5678"));

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_uint64_xor_test(void)
{
  netwib_uint64 ui64a, ui64b, ui64c;

  puts("++++----\nTest of netwib_uint64_xor");

  netwib_er(netwibtest_uint64_init("1234067812345678", &ui64a));
  netwib_er(netwibtest_uint64_init("120C5678123F5678", &ui64b));
  netwib_er(netwib_uint64_xor(ui64a, ui64b, &ui64c));
  netwib_er(netwibtest_uint64_content(ui64c, "00385000000B0000"));

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_uint64_not_test(void)
{
  netwib_uint64 ui64a, ui64b;

  puts("++++----\nTest of netwib_uint64_not");

  netwib_er(netwibtest_uint64_init("FFFFFFFFFFFFFFFF", &ui64a));
  netwib_er(netwib_uint64_not(ui64a, &ui64b));
  netwib_er(netwibtest_uint64_content(ui64b, "0000000000000000"));

  netwib_er(netwibtest_uint64_init("1234567812345678", &ui64a));
  netwib_er(netwib_uint64_not(ui64a, &ui64b));
  netwib_er(netwibtest_uint64_content(ui64b, "EDCBA987EDCBA987"));

  return(NETWIB_ERR_OK);
}


/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/
netwib_err netwibtest_dat_uint64(void)
{
  puts("###########################################\nTest of dat/uint64");
  netwib_er(netwib_uint64_init_uint32_test());
  netwib_er(netwib_uint64_init_uintmax_test());
  netwib_er(netwib_uint64_init_uintptr_test());
  netwib_er(netwib_int64_init_int32_test());
  netwib_er(netwib_int64_init_intmax_test());
  netwib_er(netwib_int64_init_intptr_test());
  netwib_er(netwib_uint32_init_uint64_test());
  netwib_er(netwib_uintmax_init_uint64_test());
  netwib_er(netwib_uintptr_init_uint64_test());
  netwib_er(netwib_int32_init_int64_test());
  netwib_er(netwib_intmax_init_int64_test());
  netwib_er(netwib_intptr_init_int64_test());

  netwib_er(netwib_uint64_init_int64_test());
  netwib_er(netwib_int64_init_uint64_test());

  netwib_er(netwib_uint64_add_test());
  netwib_er(netwib_uint64_sub_test());
  netwib_er(netwib_uint64_mul_test());
  netwib_er(netwib_uint64_div_test());
  netwib_er(netwib_int64_div_test());

  netwib_er(netwib_uint64_inc_test());
  netwib_er(netwib_uint64_dec_test());

  netwib_er(netwib_uint64_cmp_test());
  netwib_er(netwib_int64_cmp_test());

  netwib_er(netwib_int64_neg_test());

  netwib_er(netwib_uint64_shl_test());
  netwib_er(netwib_uint64_shr_test());
  netwib_er(netwib_uint64_and_test());
  netwib_er(netwib_uint64_or_test());
  netwib_er(netwib_uint64_xor_test());
  netwib_er(netwib_uint64_not_test());

  return(NETWIB_ERR_OK);
}
