/*
                                  NETWIB
                             Network library
                Copyright(c) 1999-2006 Laurent Constantin
                                  -----

  Main server    : http://www.laurentconstantin.com/
  Backup servers : http://go.to/laurentconstantin/
                   http://laurentconstantin.est-la.com/
                   http://laurentconstantin.free.fr/
                   http://membres.lycos.fr/lauconstantin/
  [my current email address is on the web servers]

                                  -----
  This file is part of Netwib.

  Netwib is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  version 2 as published by the Free Software Foundation.

  Netwib is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details (http://www.gnu.org/).

------------------------------------------------------------------------
*/

#include <netwib/inc/maininc.h>

/*-------------------------------------------------------------*/
struct netwib_hash_index {
  netwib_hash *phash;
  netwib_bool lastset;
  netwib_uint32 lastindex;
  netwib_hashitem *plastitem;
  netwib_bool nextset;
  netwib_uint32 nextindex;
  netwib_hashitem *pnextitem;
};

/*-------------------------------------------------------------*/
netwib_err netwib_hash_index_init(netwib_consthash *phash,
                                  netwib_hash_index **pphashindex)
{
  netwib_hash_index *phashindex;

  /* parameter verification */
  if (pphashindex == NULL) {
    return(NETWIB_ERR_PANULLPTR);
  }

  /* allocate needed memory to store phash */
  netwib_er(netwib_ptr_malloc(sizeof(netwib_hash_index),
                              (netwib_ptr*)&phashindex));
  *pphashindex = phashindex;
  phashindex->phash = netwib_priv_castptr(phash);
  phashindex->lastset = NETWIB_FALSE;
  phashindex->nextset = NETWIB_FALSE;

#if NETWIB_DEBUG_LEAK==1
  netwib_er(netwib_debug_leak_valid_hash(phash));
  netwib_er(netwib_debug_leak_add_hashi(phashindex));
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_hash_index_close(netwib_hash_index **pphashindex)
{
  netwib_hash_index *phashindex;

  /* parameter verification */
  if (pphashindex == NULL) {
    return(NETWIB_ERR_PANULLPTR);
  }
  phashindex = *pphashindex;

#if NETWIB_DEBUG_LEAK==1
  netwib_er(netwib_debug_leak_del_hashi(phashindex));
#endif

  /* free memory */
  netwib_er(netwib_ptr_free((netwib_ptr*)&phashindex));

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_hash_index_ctl_set(netwib_hash_index *phashindex,
                                     netwib_hash_index_ctltype type,
                                     netwib_ptr p,
                                     netwib_uint32 ui)
{
  netwib_hash_index *phashindexref;

  /* parameter verification */
  if (phashindex == NULL) {
    return(NETWIB_ERR_PANULLPTR);
  }
#if NETWIB_DEBUG_LEAK==1
  netwib_er(netwib_debug_leak_valid_hashi(phashindex));
#endif

  switch(type) {
    case NETWIB_HASH_INDEX_CTLTYPE_REWIND :
      phashindex->lastset = NETWIB_FALSE;
      phashindex->nextset = NETWIB_FALSE;
      return(NETWIB_ERR_OK);
      break;
    case NETWIB_HASH_INDEX_CTLTYPE_INDEX :
      phashindexref = (netwib_hash_index *)p;
      phashindex->phash = phashindexref->phash;
      phashindex->lastset = phashindexref->lastset;
      if (phashindexref->lastset) {
        phashindex->lastindex = phashindexref->lastindex;
        phashindex->plastitem = phashindexref->plastitem;
      }
      phashindex->nextset = phashindexref->nextset;
      if (phashindexref->nextset) {
        phashindex->nextindex = phashindexref->nextindex;
        phashindex->pnextitem = phashindexref->pnextitem;
      }
      return(NETWIB_ERR_OK);
      break;
    default :
      return(NETWIB_ERR_PAINVALIDTYPE);
      /* perhaps used in those cases */
      p = p;
      ui = ui;
  }

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_hash_index_ctl_get(netwib_hash_index *phashindex,
                                     netwib_hash_index_ctltype type,
                                     netwib_ptr p,
                                     netwib_uint32 *pui)
{
  /* parameter verification */
  if (phashindex == NULL) {
    return(NETWIB_ERR_PANULLPTR);
  }
#if NETWIB_DEBUG_LEAK==1
  netwib_er(netwib_debug_leak_valid_hashi(phashindex));
#endif

  switch(type) {
    case NETWIB_HASH_INDEX_CTLTYPE_REWIND :
    case NETWIB_HASH_INDEX_CTLTYPE_INDEX :
      return(NETWIB_ERR_PAINVALIDTYPE);
      break;
    default :
      return(NETWIB_ERR_PAINVALIDTYPE);
      /* perhaps used in those cases */
      p = p;
      pui = pui;
  }

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_hash_index_next_criteria(netwib_hash_index *phashindex,
                                           netwib_hash_criteria_pf pfunc_search,
                                           netwib_ptr pinfos,
                                           netwib_buf *pkey,
                                           netwib_ptr *ppitem)
{
  netwib_hash *phash;
  netwib_hashitem *workptr;
  netwib_uint32 workindex;
  netwib_bool b;
  netwib_buf key;
  netwib_err ret;

  /* parameter verification */
  if (phashindex == NULL) {
    return(NETWIB_ERR_PANULLPTR);
  }
  phash = phashindex->phash;
#if NETWIB_DEBUG_LEAK==1
  netwib_er(netwib_debug_leak_valid_hashi(phashindex));
  netwib_er(netwib_debug_leak_valid_hash(phash));
#endif

  /* point to first candidate */
  if (phashindex->nextset) {
    workindex = phashindex->nextindex;
    workptr = phashindex->pnextitem;
  } else if (phashindex->lastset) {
    workindex = phashindex->lastindex;
    workptr = phashindex->plastitem->pnext;
  } else {
    workindex = 0;
    workptr = phash->table[workindex];
  }

  /* main loop */
  b = NETWIB_TRUE;
  while (NETWIB_TRUE) {
    /* eventually skip empty */
    while (workptr == NULL) {
      workindex++;
      if (workindex > phash->tablemax) {
        return(NETWIB_ERR_DATAEND);
      }
      workptr = phash->table[workindex];
    }
    /* test it */
    if (pfunc_search != NULL) {
      netwib_er(netwib_buf_init_ext_datafilled(workptr->key, workptr->keysize,
                                             &key));
      b = NETWIB_FALSE;
      ret = (*pfunc_search)(&key, workptr->pitem, pinfos, &b);
      if (ret != NETWIB_ERR_OK) return(ret);
    }
    if (b) {
      netwib_er(netwib_buf_append_data(workptr->key, workptr->keysize, pkey));
      if (ppitem != NULL) *ppitem = workptr->pitem;
      phashindex->lastindex = workindex;
      phashindex->plastitem = workptr;
      phashindex->lastset = NETWIB_TRUE;
      phashindex->nextset = NETWIB_FALSE;
      return(NETWIB_ERR_OK);
    }
    /* jump to next */
    workptr = workptr->pnext;
  }

  return(NETWIB_ERR_LOINTERNALERROR);
}

/*-------------------------------------------------------------*/
netwib_err netwib_hash_index_this_value(netwib_hash_index *phashindex,
                                        netwib_buf *pkey,
                                        netwib_ptr *ppitem)
{
  netwib_hash *phash;

  /* parameter verification */
  if (phashindex == NULL) {
    return(NETWIB_ERR_PANULLPTR);
  }
  phash = phashindex->phash;
#if NETWIB_DEBUG_LEAK==1
  netwib_er(netwib_debug_leak_valid_hashi(phashindex));
  netwib_er(netwib_debug_leak_valid_hash(phash));
#endif

  if (!phashindex->lastset) {
    return(NETWIB_ERR_PAINDEXNODATA);
  }

  netwib_er(netwib_buf_append_data(phashindex->plastitem->key,
                                 phashindex->plastitem->keysize, pkey));
  if (ppitem != NULL) *ppitem = phashindex->plastitem->pitem;

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_hash_index_this_del(netwib_hash_index *phashindex,
                                      netwib_bool eraseitem)
{
  netwib_hash *phash;
  netwib_hashitem *phashitem, **addressprevious;
  netwib_bool isok;

  /* parameter verification */
  if (phashindex == NULL) {
    return(NETWIB_ERR_PANULLPTR);
  }
  phash = phashindex->phash;
#if NETWIB_DEBUG_LEAK==1
  netwib_er(netwib_debug_leak_valid_hashi(phashindex));
  netwib_er(netwib_debug_leak_valid_hash(phash));
#endif

  if (!phashindex->lastset) {
    return(NETWIB_ERR_PAINDEXNODATA);
  }

  /* check if lastindex is still correct */
  if (phashindex->lastindex > phash->tablemax) {
    return(NETWIB_ERR_PAINDEXNODATA);
  }

  /* check if plastitem is still correct */
  phashitem = phash->table[phashindex->lastindex];
  addressprevious = &(phash->table[phashindex->lastindex]);
  isok = NETWIB_FALSE;
  while (phashitem != NULL) {
    if (phashitem == phashindex->plastitem) {
      isok = NETWIB_TRUE;
      break;
    }
    addressprevious = &(phashitem->pnext);
    phashitem = phashitem->pnext;
  }
  if (!isok) {
    return(NETWIB_ERR_PAINDEXNODATA);
  }

  /* now remove the item */
  if (eraseitem && phash->pfunc_erase != NULL) {
    netwib_er((*phash->pfunc_erase)(phashitem->pitem));
  }
  *addressprevious = phashitem->pnext;
  netwib_er(netwib_ptr_free((netwib_ptr*)&phashitem));
  phash->numberofitems--;

  /* last set become unusable : will use nextset */
  phashindex->lastset = NETWIB_FALSE;

  /* ensure next call to netwib_hash_index_next_criteria will return
     the correct value */
  phashindex->nextset = NETWIB_TRUE;
  phashindex->nextindex = phashindex->lastindex;
  phashindex->pnextitem = *addressprevious; /* can be NULL, but dealt in next_criteria */

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_hash_index_this_replace(netwib_hash_index *phashindex,
                                          netwib_constptr pitem,
                                          netwib_bool erasepreviousitem)
{
  netwib_hash *phash;
  netwib_hashitem *phashitem, **addressprevious;
  netwib_bool isok;

  /* parameter verification */
  if (phashindex == NULL) {
    return(NETWIB_ERR_PANULLPTR);
  }
  phash = phashindex->phash;
#if NETWIB_DEBUG_LEAK==1
  netwib_er(netwib_debug_leak_valid_hashi(phashindex));
  netwib_er(netwib_debug_leak_valid_hash(phash));
#endif

  if (!phashindex->lastset) {
    return(NETWIB_ERR_PAINDEXNODATA);
  }

  /* check if lastindex is still correct */
  if (phashindex->lastindex > phash->tablemax) {
    return(NETWIB_ERR_PAINDEXNODATA);
  }

  /* check if plastitem is still correct */
  phashitem = phash->table[phashindex->lastindex];
  addressprevious = &(phash->table[phashindex->lastindex]);
  isok = NETWIB_FALSE;
  while (phashitem != NULL) {
    if (phashitem == phashindex->plastitem) {
      isok = NETWIB_TRUE;
      break;
    }
    addressprevious = &(phashitem->pnext);
    phashitem = phashitem->pnext;
  }
  if (!isok) {
    return(NETWIB_ERR_PAINDEXNODATA);
  }

  /* now replace the item */
  if (erasepreviousitem && phash->pfunc_erase != NULL) {
    netwib_er((*phash->pfunc_erase)(phashitem->pitem));
  }
  phashitem->pitem = netwib_priv_castpptr(pitem);

  return(NETWIB_ERR_OK);
}
