/*  Nomad:  nomad-util.c
 *
 *  Copyright (C) 2002 David A Knight <david@ritter.demon.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <glib/gconvert.h>
#include <glib/gunicode.h>
#include <glib/gstrfuncs.h>
#include <glib/gstring.h>

#ifndef HAVE_GNOME_VFS

#include <glib/gfileutils.h>
#include <glib/gdir.h>

#include <sys/stat.h>
#include <sys/types.h>

#else

#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-directory.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-ops.h>

#endif

#include "nomad-util.h"

/* Converts a figure representing a number of seconds to
 * a string in mm:ss notation */
gchar *seconds_to_mmss(guint seconds)
{ 
	gchar tmp2[4];
	gchar tmp[10];
	
	if (!seconds)
		return g_strdup("0:00");
	sprintf(tmp2, "0%u", seconds%60);
	while (strlen(tmp2)>2) {
		tmp2[0]=tmp2[1];
		tmp2[1]=tmp2[2];
		tmp2[2]='\0';
	}
	sprintf(tmp, "%lu:%s", (unsigned long)seconds/60, tmp2);
	return g_strdup(tmp);
}

guint mmss_to_seconds( const gchar *mmss )
{
	gchar **tmp;
	guint seconds = 0;
	
	if( mmss ) {
		tmp = g_strsplit(mmss, ":", 0);
		if( tmp[ 0 ] && tmp[ 1 ] && ! tmp[ 2 ] ) {
			gchar *dummy;
			
			seconds = 60 * strtoul( tmp[0], &dummy, 10 );
			seconds += strtoul( tmp[1], &dummy, 10 );
		}
		if (tmp != NULL) {
			g_strfreev(tmp);
		}
	}

	return seconds;
}

gpointer nomad_create( const gchar *path )
{
	gpointer handle;
	gchar *dir;
	
	dir = g_path_get_dirname( path );
	handle = NULL;
	if( nomad_mkdir_recursive( dir ) ) {
#ifndef HAVE_GNOME_VFS
		handle = fopen( path, "w+" );
#else
		gnome_vfs_create( (GnomeVFSHandle**)&handle, path, 
				GNOME_VFS_OPEN_READ | 
				GNOME_VFS_OPEN_WRITE |
				GNOME_VFS_OPEN_RANDOM, 
				FALSE, 
				GNOME_VFS_PERM_USER_READ |
				GNOME_VFS_PERM_USER_WRITE );
#endif
	}
	g_free( dir );

	return handle;
}

void nomad_close( gpointer handle )
{
#ifndef HAVE_GNOME_VFS
	fclose( handle );
#else
	gnome_vfs_close( (GnomeVFSHandle*)handle );
#endif
}

gboolean nomad_write( gpointer handle, const char *buf, guint len )
{
#ifndef HAVE_GNOME_VFS
	ssize_t res;
	
	do {
		res = fwrite( buf, 1, len, handle );
		if( res > 0 ) {
			len -= res;
			buf += res;
		}
	} while( res > 0 );

	return ( res == 0 );
#else
	GnomeVFSFileSize written;
	GnomeVFSResult res;

	res = GNOME_VFS_OK;
	while( res == GNOME_VFS_OK && ( len > 0 ) ) {
		res = gnome_vfs_write( (GnomeVFSHandle*)handle, buf, len, &written );
		len -= written;
		buf += written;
	}
	
	return ( res == GNOME_VFS_OK );
#endif
}

gboolean nomad_seek( gpointer handle, gint pos, gint offset )
{
#ifndef HAVE_GNOME_VFS
	return ( fseek( handle, offset, pos ) == 0 );
#else
	GnomeVFSSeekPosition spos;
	if( pos == SEEK_SET ) {
		spos = GNOME_VFS_SEEK_START;
	} else if( pos == SEEK_CUR ) {
		spos = GNOME_VFS_SEEK_CURRENT;
	} else {
		spos = GNOME_VFS_SEEK_END;
	}
	return ( gnome_vfs_seek( handle, spos, 
				offset ) == GNOME_VFS_OK );
#endif
}

gboolean nomad_read( gpointer handle, gchar *buf, guint len )
{
#ifndef HAVE_GNOME_VFS
	return ( fread( buf, 1, len, handle ) == len );
#else
	GnomeVFSFileSize got;
	return ( gnome_vfs_read( handle, buf, len, &got ) == GNOME_VFS_OK );
#endif
}

gint nomad_tell( gpointer handle )
{
#ifndef HAVE_GNOME_VFS
	return ftell( handle );
#else
	GnomeVFSFileSize offset;

	offset = 0;
	gnome_vfs_tell( handle, &offset );

	return offset;
#endif
}

gboolean nomad_mkdir_recursive( const gchar *path )
{
#ifndef HAVE_GNOME_VFS
	gchar *dir;
	gboolean exists;
	gboolean isdir;
	
	dir = g_path_get_dirname( path );
	if( dir && *dir != '\0' && 
		! ( dir[ 0 ] == G_DIR_SEPARATOR && dir[ 1 ] == '\0' ) ) {
		nomad_mkdir_recursive( dir );
	}
	g_free( dir );
	exists = g_file_test( path, G_FILE_TEST_EXISTS );
	isdir = FALSE;
	if( exists ) {
		isdir = g_file_test( path, G_FILE_TEST_IS_DIR );
	}
	if( ! exists ) {
		isdir = ( mkdir( path, 
				S_IRWXU | 
				S_IRWXG | 
				S_IROTH | S_IXOTH ) == 0 );
	}

	return isdir;
#else
	GnomeVFSURI *uri;
	gboolean ret = FALSE;
	
	uri = gnome_vfs_uri_new( path );
	if( uri ) {
		ret = TRUE;

		if( gnome_vfs_uri_has_parent( uri ) ) {
			GnomeVFSURI *parent;
			gchar *parentname;

			parent = gnome_vfs_uri_get_parent( uri );

			parentname = gnome_vfs_uri_to_string( parent,
							      GNOME_VFS_URI_HIDE_NONE );
			gnome_vfs_uri_unref( parent );

			ret = nomad_mkdir_recursive( parentname );
		}
		if( ret ) {
			GnomeVFSResult result;

			result=gnome_vfs_make_directory_for_uri(uri, 
								GNOME_VFS_PERM_USER_ALL|
								GNOME_VFS_PERM_GROUP_ALL|
								GNOME_VFS_PERM_OTHER_READ|
								GNOME_VFS_PERM_OTHER_EXEC);
			if( result == GNOME_VFS_ERROR_FILE_EXISTS ) {
				/* ensure it is a directory */
				GnomeVFSFileInfo *info;
				GnomeVFSFileInfoOptions options;
				
				info = gnome_vfs_file_info_new();
				options = GNOME_VFS_FILE_INFO_DEFAULT;
				gnome_vfs_get_file_info_uri( uri, info,
							     options );
				if(info->type == GNOME_VFS_FILE_TYPE_DIRECTORY)
					result = GNOME_VFS_OK;
				gnome_vfs_file_info_unref( info );
			}
			
			ret = ( result == GNOME_VFS_OK );
		}
		gnome_vfs_uri_unref( uri );
	}

	return ret;
#endif
}

gchar *nomad_filename_escape( const gchar *orig )
{
	GString *string;
	const gchar *temp;
	gchar *ret;

	string = g_string_new( "" );
	for( temp = orig; *temp; ++ temp ) {
		if( *temp == G_DIR_SEPARATOR ||
		    *temp == G_SEARCHPATH_SEPARATOR ) {
			g_string_append_c( string, '-' );
		} else {
			g_string_append_c( string, *temp );
		}
	}
	ret = string->str;

	g_string_free( string, FALSE );

	return ret;
}


