/* Class functions ... really part of heap.c, but split out here to make it
 * more manageable.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#include "ip.h"

/*
#define DEBUG_MEMBER
#define DEBUG_VERBOSE
#define DEBUG
#define DEBUG_BUILD
 */

static gboolean
class_isclass( PElement *instance )
{
	if( !PEISCLASS( instance ) ) {
		BufInfo buf;
		char txt[50];

		buf_init_static( &buf, txt, 50 );
		if( !itext_decompile( &buf, instance ) )
			return( FALSE );
		ierrors( "object: %s\nis not a class", buf_all( &buf ) );

		return( FALSE );
	}

	return( TRUE );
}

/* Look up "super" in a class ... try to do it quickly.
 */
gboolean
class_get_super( PElement *instance, PElement *out )
{
	Compile *compile;

	if( !class_isclass( instance ) )
		return( FALSE );
	compile = PEGETCLASSCOMPILE( instance );
	assert( compile->super );

	return( class_get_symbol( instance, compile->super, out ) );
}

/* Look up a member in a class instance by name. If lookup fails in this 
 * instance, try the superclass. Don't search secrets. Point "out" at the 
 * value.
 */
gboolean
class_get_member( PElement *instance, const char *name, PElement *out )
{
	PElement member;
	PElement super;
	HeapNode *p;

#ifdef DEBUG_MEMBER
	printf( "class_get_member: looking up \"%s\" in class ", name );
	pgraph( instance );
#endif /*DEBUG_MEMBER*/

	if( !class_isclass( instance ) )
		return( FALSE );

	/* Search this instance member list.
	 */
	PEGETCLASSMEMBER( &member, instance );
	if( !PEISELIST( &member ) )
		for( p = PEGETVAL( &member ); p; p = GETRIGHT( p ) ) {
			PElement s;
			HeapNode *hn;
			Symbol *sym;

			/* Get the sym/value pair, get the sym.
			 */
			hn = GETLEFT( p );
			PEPOINTLEFT( hn, &s );

			/* Match? 
			 */
			sym = PEGETSYMREF( &s );
			if( strcmp( MODEL( sym )->name, name ) == 0 ) {
				/* Found!
				 */
				PEPOINTRIGHT( hn, out );

#ifdef DEBUG_MEMBER
				printf( "class_get_member: found: " );
				pgraph( out );
#endif /*DEBUG_MEMBER*/

				return( TRUE );
			}
		}

	/* Nope ... try the superclass.
	 */
	if( !class_get_super( instance, &super ) || !PEISELIST( &super ) ) {
		/*

			FIXME ... gcc 2.95.2 gets this wrong, tries to
				  eliminate the tail recursion with -O2
				  and makes bad code
			      ... guess how long that took to find
			      ... put this back at some point

		return( class_get_member( &super, name, out ) ); 
		 */
		gboolean result = class_get_member( &super, name, out ); 

		return( result );
	}

	ierrors( "member \"%s\" not found in class \"%s\"",
		name, MODEL( PEGETCLASSCOMPILE( instance )->sym )->name );

	return( FALSE );
}

/* Look up a symbol in a class. Write to out, or FALSE for not
 * found. Look up by symbol pointer. Search secrets as
 * well. Try the superclass if lookup fails.
 */
gboolean
class_get_symbol( PElement *instance, Symbol *sym, PElement *out )
{
	HeapNode *p;
	PElement secret;
	PElement super;

#ifdef DEBUG_MEMBER
	printf( "class_get_symbol: looking up " );
	symbol_name_print( sym );
	printf( " in class " );
	pgraph( instance );
#endif /*DEBUG_MEMBER*/

	if( !class_isclass( instance ) )
		return( FALSE );

	PEGETCLASSSECRET( &secret, instance );
	if( PEISNODE( &secret ) )
		for( p = PEGETVAL( &secret ); p; p = GETRIGHT( p ) ) {
			PElement s;
			HeapNode *hn;

			/* Get the sym/value pair, get the sym.
			 */
			hn = GETLEFT( p );
			PEPOINTLEFT( hn, &s );

			/* Match?
			 */
			if( PEGETSYMREF( &s ) == sym ) {
				/* Found!
				 */
				PEPOINTRIGHT( hn, out );

#ifdef DEBUG_MEMBER
				printf( "class_get_symbol: found: " );
				pgraph( out );
#endif /*DEBUG_MEMBER*/

				return( TRUE );
			}
		}

	/* Nope ... try the superclass.
	 */
	if( !class_get_super( instance, &super ) || !PEISELIST( &super ) ) {
		/*

			FIXME ... gcc 2.95.2 gets this wrong, tries to
				  eliminate the tail recursion with -O2
				  and makes bad code
			      ... guess how long that took to find
			      ... put this back at some point

		return( class_get_member( &super, name, out ) ); 
		 */
		gboolean result = class_get_symbol( &super, sym, out ); 

		return( result );
	}

	return( FALSE );
}

/* Search back up the inheritance tree for an exact instance of this
 * class.
 */
gboolean
class_get_exact( PElement *instance, const char *name, PElement *out )
{
	PElement pe;

	pe = *instance;
	while( !reduce_isinstanceof_exact( reduce_context, name, &pe ) ) {
		if( !class_get_super( instance, &pe ) || PEISELIST( &pe ) ) 
			return( FALSE );
	}

	*out = pe;

	return( TRUE );
}

/* Stuff we need for class build.
 */
typedef struct {
	Heap *hi;		/* Heap to build on */
	Symbol *sym;		/* Sym we are local to */
	PElement *arg;		/* Args to constructor */
	PElement *this;		/* Base of instance we are building */
	int i;			/* Index in arg list */
	Compile *compile;	/* Compile for our class */
} ClassBuildInfo;

/* Member sym of class pbi->sym needs secret as an argument ... add it!
 */
static gboolean
class_member_secret( ClassBuildInfo *pbi, 
	Symbol *sym, GSList *secret, PElement *out )
{
	Symbol *ssym;
	Heap *hi = pbi->hi;
	HeapNode *apl;
#ifdef DEBUG_VERBOSE
	PElement p1;
	BufInfo buf;
	char txt[1024];
#endif /*DEBUG_VERBOSE*/

	if( !secret )
		return( TRUE );
	ssym = SYMBOL( secret->data );

	/* Make function application for this member.
	 */
	if( NEWNODE( hi, apl ) )
		return( FALSE );
	apl->type = TAG_APPL;
	PEPUTLEFT( apl, out );

	/* Is the secret "this"? Easy.
	 */
	if( ssym == pbi->sym->expr->compile->this ) {
		PEPUTRIGHT( apl, pbi->this );
	}
	else {
		/* Look up ssym in sym's secrets ... should be there 
		 * somewhere. Use it's index to find the pbi->arg[] we need.
		 */
		int pos = g_slist_index( sym->expr->compile->secret, ssym );

		assert( pos >= 0 && pos < sym->expr->compile->nsecret );

		PEPUTRIGHT( apl, &pbi->arg[pos] );
	}

	PEPUTP( out, ELEMENT_NODE, apl );

#ifdef DEBUG_VERBOSE
	buf_init_static( &buf, txt, 1024 );
	PEPOINTRIGHT( apl, &p1 );
	graph_pelement( hi, &buf, &p1, TRUE );
	printf( "class_member_secret: secret arg \"%s\" "
		"to member \"%s\" = %s\n", 
		MODEL( ssym )->name, MODEL( sym )->name, buf_all( &buf ) );
#endif /*DEBUG_VERBOSE*/

	return( class_member_secret( pbi, sym, secret->next, out ) );
}

/* Add a member to a class.
 */
static void *
add_class_member( Symbol *sym, ClassBuildInfo *pbi, PElement *out )
{
	Heap *hi = pbi->hi;
	HeapNode *base, *sv;
	PElement v;
#ifdef DEBUG_VERBOSE
	BufInfo buf;
	char txt[1024];
#endif /*DEBUG_VERBOSE*/

	/* Is this something that should be part of a class.
	 */
	if( sym->type != SYM_VALUE )
		return( NULL );

	/* Make new class-local-list element for this local.
	 */
	if( NEWNODE( hi, base ) )
		return( sym );
	base->type = TAG_CONS;
	PPUTLEFT( base, ELEMENT_ELIST, NULL );
	PEPUTRIGHT( base, out );
	PEPUTP( out, ELEMENT_NODE, base );

	/* Make sym/value pair for this local.
	 */
	if( NEWNODE( hi, sv ) )
		return( sym );
	sv->type = TAG_CONS;
	PPUT( sv, ELEMENT_SYMREF, sym, ELEMENT_SYMBOL, sym ); 
	PPUTLEFT( base, ELEMENT_NODE, sv );

	/* Build value ... apply args to the symbol.
	 */
	PEPOINTRIGHT( sv, &v );
	if( !class_member_secret( pbi, sym, sym->expr->compile->secret, &v ) )
		return( sym );

#ifdef DEBUG_VERBOSE
	buf_init_static( &buf, txt, 1024 );
	graph_pelement( hi, &buf, &v, TRUE );
	printf( "add_class_member: member \"%s\" of class \"%s\" = %s\n",
		MODEL( sym )->name, MODEL( pbi->sym )->name, buf_all( &buf ) );
#endif /*DEBUG_VERBOSE*/

	return( NULL );
}

/* Add a symbol/value pair to a class.
 */
static gboolean
add_class_svpair( ClassBuildInfo *pbi, 
	Symbol *sym, PElement *val, PElement *out )
{
	Heap *hi = pbi->hi;
	HeapNode *base, *sv;
#ifdef DEBUG_VERBOSE
	BufInfo buf;
	char txt[1024];
#endif /*DEBUG_VERBOSE*/

#ifdef DEBUG_VERBOSE
	buf_init_static( &buf, txt, 1024 );
	graph_pelement( hi, &buf, val, TRUE );
	printf( "add_class_svpair: adding parameter \"%s\" to class "
		"\"%s\" = %s\n", MODEL( sym )->name, MODEL( pbi->sym )->name,
		buf_all( &buf ) );
#endif /*DEBUG_VERBOSE*/

	/* Make new class-local-list element for this parameter.
	 */
	if( NEWNODE( hi, base ) )
		return( FALSE );
	base->type = TAG_CONS;
	PPUTLEFT( base, ELEMENT_ELIST, NULL );
	PEPUTRIGHT( base, out );
	PEPUTP( out, ELEMENT_NODE, base );

	/* Make sym/value pair for this parameter.
	 */
	if( NEWNODE( hi, sv ) )
		return( FALSE );
	sv->type = TAG_CONS;
	PPUTLEFT( sv, ELEMENT_SYMREF, sym )
	PEPUTRIGHT( sv, val );
	PPUTLEFT( base, ELEMENT_NODE, sv );

	return( TRUE );
}

/* Add a parameter (secret or real) to a class.
 */
static void *
add_class_parameter( Symbol *sym, ClassBuildInfo *pbi, PElement *out )
{
	/* Add this symbol/value pair.
	 */
	if( !add_class_svpair( pbi, sym, &pbi->arg[pbi->i], out ) )
		return( sym );

	/* Move arg index on.
	 */
	pbi->i += 1;

	return( NULL );
}

/* Add the name member ... build the name string carefully.
 */
static void *
class_new_single_name( Heap *hi, PElement *pe,
	ClassBuildInfo *pbi, PElement *instance )
{
	Symbol *snm = stable_find( pbi->compile->locals, MEMBER_NAME );
	BufInfo buf;
	char str[256];

	/* Make class name string.
	 */
	buf_init_static( &buf, str, 256 );
	symbol_qualified_name( pbi->sym, &buf );
	PEPUTP( pe, ELEMENT_ELIST, NULL );
	if( !heap_string_new( hi, buf_all( &buf ), pe ) ) 
		return( hi );

	/* Add as a member.
	 */
	if( !add_class_svpair( pbi, snm, pe, instance ) ) 
		return( hi );

	return( NULL );
}

/* Make a single level class instance ... fn below then loops over a class
 * heierarchy with this.
 */
static gboolean
class_new_single( Heap *hi, 
	Compile *compile, PElement *arg, PElement *this, PElement *out )
{
	Symbol *sym = compile->sym;
	Symbol *sths = compile->this;

	HeapNode *base, *sm;
	PElement p1;
	ClassBuildInfo pbi;

	/* Make class base.
	 */
	if( NEWNODE( hi, base ) )
		return( FALSE );
	base->type = TAG_CLASS;
	PPUT( base, ELEMENT_COMPILEREF, compile, ELEMENT_ELIST, NULL ); 
	PEPUTP( out, ELEMENT_NODE, base );

	/* Make node for holding secrets and members.
	 */
	if( NEWNODE( hi, sm ) )
		return( FALSE );
	sm->type = TAG_CONS;
	PPUT( sm, ELEMENT_ELIST, NULL, ELEMENT_ELIST, NULL ); 
	PPUTRIGHT( base, ELEMENT_NODE, sm );

	/* Build list of members.
	 */
	pbi.hi = hi;
	pbi.sym = sym;
	pbi.arg = arg;
	pbi.this = this;
	pbi.compile = compile;
	PEPOINTRIGHT( sm, &p1 );
	if( stable_map_rev( compile->locals, 
		(symbol_map_fn) add_class_member, &pbi, &p1, NULL ) ) 
		return( FALSE );

	/* Add name member.
	 */
	if( heap_safe_pointer( hi,
		(heap_safe_pointer_fn) class_new_single_name, 
		&pbi, &p1, NULL, NULL ) )
		return( FALSE );

	/* Add this member.
	 */
	if( !add_class_svpair( &pbi, sths, this, &p1 ) )
		return( FALSE );

	/* Add class parameters to member list.
	 */
	pbi.i = 0;
	if( slist_map2_rev( compile->param, 
		(SListMap2Fn) add_class_parameter, &pbi, &p1 ) )
		return( FALSE );

	/* Now ... secret list starts off pointing to head of member list.
	 */
	PEPUTLEFT( sm, &p1 );

	/* Add all secret parameters to secret list.
	 */
	PEPOINTLEFT( sm, &p1 );
	if( slist_map2_rev( compile->secret, 
		(SListMap2Fn) add_class_parameter, &pbi, &p1 ) )
		return( FALSE );

#ifdef DEBUG
{
	BufInfo buf;
	char str[256];

	buf_init_static( &buf, str, 256 );
	graph_pelement( hi, &buf, out, TRUE );
	printf( "class_new_single: built instance of \"%s\": %s\n", 
		MODEL( sym )->name, buf_all( &buf ) );
}
#endif /*DEBUG*/

	return( TRUE );
}

static void *
class_new_super_sub( Heap *hi, PElement *p1,
	Compile *compile, PElement *arg, PElement *this, PElement *super )
{
	/* Build the superclass ... we overwrite the super 
	 * list with the constructed class, so make a copy of 
	 * the pointer to stop it being GCed.
	 */
	PEPUTPE( p1, super );

	if( !class_new_single( hi, compile, arg, this, super ) ) 
		return( hi );

	return( NULL );
}

/* Does this class have a "super"? Build it and recurse.
 */
gboolean
class_new_super( Heap *hi, 
	Compile *compile, PElement *this, PElement *instance )
{
	PElement super;

	if( compile->has_super && class_get_super( instance, &super ) ) {
		Compile *super_compile;
		int len;
		PElement arg0;

		/* It must be a list whose first element is the superclass
		 * constructor, or the superclass itself (if it has no args).
		 */
		if( (len = heap_list_length( &super )) < 1 ||
			!heap_list_index( &super, 0, &arg0 ) ||
			!heap_reduce_strict( &arg0 ) )
			return( FALSE );

		if( PEISCONSTRUCTOR( &arg0 ) ) {
			int i;
			PElement arg[MAX_SYSTEM];

			super_compile = PEGETCOMPILE( &arg0 );

			/* It's a constructor ... we need to give it some
			 * args.
			 */
			if( len >= MAX_SYSTEM ) {
				ierrors( "too many args to superclass "
					"constructor %s",
					symbol_name( super_compile->sym ) );
				return( FALSE );
			}

			if( super_compile->nparam != len - 1 ) {
				ierrors( "superclass constructor %s "
					"expects %d arguments, not %d",
					symbol_name( super_compile->sym ),
					super_compile->nparam,
					len - 1 );

				return( FALSE );
			}
			if( super_compile->nsecret != 0 ) {
				ierrors( "superclass constructor %s "
					"should have no secrets",
					symbol_name( super_compile->sym ) );

				return( FALSE );
			}

			/* Build the arg array.
			 */
			for( i = 0; i < len - 1; i++ ) {
				if( !heap_list_index( &super, len - i - 1, 
					&arg[i] ) )
					return( FALSE );
			}

			/* Build the superclass ... we overwrite the super 
			 * list with the constructed class, so make a copy of 
			 * the pointer to stop it being GCed.
			 */
			if( heap_safe_pointer( hi, 
				(heap_safe_pointer_fn) class_new_super_sub,
				super_compile, arg, this, &super ) )
				return( FALSE );
		}
		else if( PEISCLASS( &arg0 ) ) {
			super_compile = PEGETCLASSCOMPILE( &arg0 );

			/* It's ready built ... there should not be any args.
			 */
			if( super_compile->nparam != len - 1 ) {
				ierrors( "superclass constructor %s "
					"expects %d arguments, not %d",
					symbol_name( super_compile->sym ),
					super_compile->nparam,
					len - 1 );
				return( FALSE );
			}

			/* The instance we have does not have the right "this"
			 * wired into it ... we must construct a new one.
			 */
			if( !class_new_single( hi, 
				super_compile, NULL, this, &super ) ) 
				return( FALSE );
		}
		else {
			ierrors( "first element in superclass of %s must be "
				"class or constructor",
				symbol_name( compile->sym ) );
			return( FALSE );
		}

		/* And recursively build any superclasses.
		 */
		if( !class_new_super( hi, super_compile, this, &super ) )
			return( FALSE );
	}

	return( TRUE );
}

/* Make a class instance. 
 */
gboolean
class_new( Heap *hi, Compile *compile, HeapNode **arg, PElement *out )
{
	int i;
	PElement pe_arg[MAX_SYSTEM];

	/* Make a set of arg pointers.
	 */
	if( compile->nparam + compile->nsecret >= MAX_SYSTEM ) {
		ierrors( "too many args to class" );
		return( FALSE );
	}
	for( i = 0; i < compile->nparam + compile->nsecret; i++ ) {
		PEPOINTRIGHT( arg[i], &pe_arg[i] );
	}

	/* Build the base instance.
	 */
	if( !class_new_single( hi, compile, pe_arg, out, out ) )
		return( FALSE );

	/* And recursively build any superclasses.
	 */
	if( !class_new_super( hi, compile, out, out ) )
		return( FALSE );

#ifdef DEBUG_BUILD
{
	BufInfo buf;
	char str[MAX_STRSIZE];

	buf_init_static( &buf, str, MAX_STRSIZE );
	graph_pelement( hi, &buf, out, TRUE );
	printf( "class_new: built instance of \"%s\": %s\n", 
		MODEL( compile->sym )->name, buf_all( &buf ) );
}
#endif /*DEBUG_BUILD*/

	return( TRUE );
}

/* Clone a class instance. Copy pointers to the the args, secrets and super; 
 * regenerate all the members. instance and out can be equal.
 */
gboolean
class_clone_args( Heap *hi, PElement *instance, PElement *out )
{
	HeapNode *arg[MAX_SYSTEM];
	Compile *compile = PEGETCLASSCOMPILE( instance );
	const int nargs = compile->nsecret + compile->nparam;
	PElement secret;
	int i;
#ifdef DEBUG_VERBOSE
	BufInfo buf;
	char str[NAMELEN];
#endif /*DEBUG_VERBOSE*/

	assert( nargs <= MAX_SYSTEM );

#ifdef DEBUG_VERBOSE
	buf_init_static( &buf, str, NAMELEN );
	graph_pelement( hi, &buf, instance, TRUE );
	printf( "class_clone_args: about to clone \"%s\": %s\n", 
		MODEL( compile->sym )->name, buf_all( &buf ) );
#endif /*DEBUG_VERBOSE*/

	/* Pull out values of secrets and class args into RHS of arg[].
	 */
	PEGETCLASSSECRET( &secret, instance );
	for( i = 0; i < nargs; i++ ) {
		HeapNode *hn = PEGETVAL( &secret );
		HeapNode *sv = GETLEFT( hn );
		int index = nargs - i - 1;

		arg[index] = sv;
		PEPOINTRIGHT( hn, &secret );
	}

	/* Build class again.
	 */
	return( class_new( hi, compile, &arg[0], out ) );
}

/* Build a class instance picking parameters from C args ... handy for
 * making a new toggle instance on a click, for example.
 */
gboolean
class_newv( Heap *hi, const char *name, PElement *out, ... )
{
	va_list ap;
	Symbol *sym;
	Compile *compile;
	HeapNode args[ MAX_SYSTEM ];
	HeapNode *pargs[ MAX_SYSTEM ];
	int i;

	if( !(sym = stable_find( symbol_root->expr->compile->locals, name )) ||
		!is_value( sym ) || !is_class( sym->expr->compile ) ) {
		ierrors( "class \"%s\" not found", name );
		return( FALSE );
	}
	compile = sym->expr->compile;
	if( compile->nparam >= MAX_SYSTEM ) {
		ierrors( "too many arguments to class" );
		return( FALSE );
	}

        va_start( ap, out );
	for( i = 0; i < compile->nparam; i++ ) {
		PElement *arg = va_arg( ap, PElement * );
		PElement rhs;

		pargs[i] = &args[i];
		PEPOINTRIGHT( pargs[i], &rhs );
		PEPUTPE( &rhs, arg );
	}
        va_end( ap );

	return( class_new( hi, compile, &pargs[0], out ) );
}

static void
class_typecheck_error( PElement *instance, const char *name, const char *type )
{
	PElement val;
	gboolean res;
	BufInfo buf;
	char txt[1024];

	res = class_get_member( instance, name, &val );
	assert( res );

	buf_init_static( &buf, txt, 1024 );
	buf_appendf( &buf, "member \"%s\" of class \"%s\" should be of type "
		"\"%s\", instead it's:\n   ",
		name, MODEL( PEGETCLASSCOMPILE( instance )->sym )->name, type );
	if( !itext_decompile( &buf, &val ) )
		return;

	ierrors( "%s", buf_all( &buf ) );
}

/* A function that gets a type from a class.
 */
typedef gboolean (*ClassGetFn)( PElement *, void * );

static gboolean
class_get_member_check( PElement *instance, const char *name, const char *type,
	ClassGetFn fn, void *a )
{
	PElement val;

	if( !class_get_member( instance, name, &val ) )
		return( FALSE );

	if( !fn( &val, a ) ) {
		class_typecheck_error( instance, name, type );
		return( FALSE );
	}

	return( TRUE );
}

gboolean
class_get_member_bool( PElement *instance, const char *name, gboolean *out )
{
	return( class_get_member_check( instance, name, "bool",
		(ClassGetFn) heap_get_bool, out ) );
}

gboolean
class_get_member_real( PElement *instance, const char *name, double *out )
{
	return( class_get_member_check( instance, name, "real",
		(ClassGetFn) heap_get_real, out ) );
}

gboolean
class_get_member_int( PElement *instance, const char *name, int *out )
{
	double d;

	if( !class_get_member_check( instance, name, "real",
		(ClassGetFn) heap_get_real, &d ) )
		return( FALSE );
	*out = IM_RINT( d );

	return( TRUE );
}

gboolean
class_get_member_class( PElement *instance, const char *name, 
	const char *type, PElement *out )
{
	gboolean result;

	if( !class_get_member_check( instance, name, type,
		(ClassGetFn) heap_get_class, out ) )
		return( FALSE );

	if( !heap_isinstanceof( type, out, &result ) ) 
		return( FALSE );
	if( !result ) {
		class_typecheck_error( instance, name, type );
		return( FALSE );
	}

	return( TRUE );
}

gboolean
class_get_member_image( PElement *instance, const char *name, Imageinfo **out )
{
	return( class_get_member_check( instance, name, "image",
		(ClassGetFn) heap_get_image, out ) );
}

gboolean
class_get_member_lstring( PElement *instance, const char *name, 
	GSList **labels )
{
	return( class_get_member_check( instance, name, "finite [[char]]",
		(ClassGetFn) heap_get_lstring, labels ) );
}

gboolean
class_get_member_string( PElement *instance, const char *name, 
	char *buf, int sz )
{
	PElement val;

	if( !class_get_member( instance, name, &val ) )
		return( FALSE );

	if( !heap_get_string( &val, buf, sz ) ) {
		class_typecheck_error( instance, name, "finite [char]" );
		return( FALSE );
	}

	return( TRUE );
}

gboolean
class_get_member_instance( PElement *instance, 
	const char *name, const char *klass, PElement *out )
{
	gboolean result;

	return( class_get_member( instance, name, out ) &&
		heap_isinstanceof( klass, out, &result ) &&
		result );
}

gboolean
class_get_member_matrix_size( PElement *instance, const char *name, 
	int *xsize, int *ysize )
{
	PElement val;

	if( !class_get_member( instance, name, &val ) )
		return( FALSE );

	if( !heap_get_matrix_size( &val, xsize, ysize ) ) {
		class_typecheck_error( instance, name, 
			"finite rectangular [[real]]" );
		return( FALSE );
	}

	return( TRUE );
}

gboolean
class_get_member_matrix( PElement *instance, const char *name, 
	double *buf, int n, int *xsize, int *ysize )
{
	PElement val;

	if( !class_get_member( instance, name, &val ) )
		return( FALSE );

	if( !heap_get_matrix( &val, buf, n, xsize, ysize ) ) {
		class_typecheck_error( instance, name, 
			"finite rectangular [[real]]" );
		return( FALSE );
	}

	return( TRUE );
}

gboolean
class_get_member_realvec( PElement *instance, const char *name, 
	double *buf, int n, int *length )
{
	PElement val;
	int l;

	if( !class_get_member( instance, name, &val ) )
		return( FALSE );

	if( (l = heap_get_realvec( &val, buf, n )) < 0 ) {
		class_typecheck_error( instance, name, "finite [real]" );
		return( FALSE );
	}

	*length = l;

	return( TRUE );
}
