/* Manage display conversion parameters.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

#include "ip.h"

#define CONVERSION_SCALE (watch_double_get( "DISPLAY_CONVERSION_SCALE", 1.0 ))
#define CONVERSION_OFFSET (watch_double_get( "DISPLAY_CONVERSION_OFFSET", 1.0 ))
#define DISPLAY_STATUS (watch_bool_get( "DISPLAY_STATUS", FALSE ))
#define DISPLAY_CONVERSION (watch_bool_get( "DISPLAY_CONVERSION", FALSE ))

static ModelClass *parent_class = NULL;

static void
conversion_destroy( GtkObject *object )
{
	Conversion *conv;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_CONVERSION( object ) );

	conv = CONVERSION( object );

#ifdef DEBUG
	printf( "conversion_destroy: 0x%x\n", (unsigned int) conv );
#endif /*DEBUG*/

	FREEF( im_region_free, conv->ireg );
	FREEF( im_region_free, conv->reg );
	FREEF( im_threadgroup_free, conv->tg );
	FREE( conv->errstr );
	FREEF( imageinfo_destroy_nonheap, conv->display_ii );
	FREEF( imageinfo_destroy_nonheap, conv->visual_ii );
	FREESID( conv->changed_sid, conv->ii );
	FREEF( imageinfo_destroy_nonheap, conv->ii );
	FREEF( imageinfo_destroy_nonheap, conv->paintbox_ink );
	FREE( conv->paintbox_text );
	FREEF( imageinfo_destroy_nonheap, conv->paintbox_text_mask );
	FREE( conv->paintbox_font_name );
	FREEF( gdk_font_unref, conv->paintbox_font );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

static void
conversion_class_init( ConversionClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;

	parent_class = gtk_type_class( TYPE_MODEL );

	object_class->destroy = conversion_destroy;

	/* Create signals.
	 */

	/* Init methods.
	 */
}

static void
conversion_init( Conversion *conv )
{
	static const Rect emptyrect = { 0, 0, 0, 0 };

#ifdef DEBUG
	printf( "conversion_init: 0x%x\n", (unsigned int) conv );
#endif /*DEBUG*/

	conv->ii = NULL;
	conv->changed_sid = 0;
	conv->reg = NULL;
	conv->visual_ii = NULL;
	conv->display_ii = NULL;
	conv->ireg = NULL;
	conv->tg = NULL;
	conv->err = FALSE;
	conv->errstr = NULL;

	conv->underlay = emptyrect;
	conv->image = emptyrect;
	conv->canvas = emptyrect;
	conv->visible = emptyrect;
	conv->mag = 1;

	conv->enabled = DISPLAY_CONVERSION;
	conv->scale = CONVERSION_SCALE;
	conv->offset = CONVERSION_OFFSET;
	conv->falsecolour = FALSE;
	conv->type = TRUE;

	conv->status = DISPLAY_STATUS;

	conv->paintbox = FALSE;
	conv->paintbox_tool = PAINTBOX_PEN;
	conv->paintbox_nib = PAINTBOX_1ROUND;
	conv->paintbox_ink = NULL;
	conv->paintbox_font_name = im_strdup( NULL, PAINTBOX_FONT );
	conv->paintbox_font = NULL;
	conv->paintbox_text = NULL;
	conv->paintbox_text_mask = NULL;
}

GtkType
conversion_get_type( void )
{
	static GtkType conversion_type = 0;

	if( !conversion_type ) {
		static const GtkTypeInfo info = {
			"Conversion",
			sizeof( Conversion ),
			sizeof( ConversionClass ),
			(GtkClassInitFunc) conversion_class_init,
			(GtkObjectInitFunc) conversion_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		conversion_type = gtk_type_unique( TYPE_MODEL, &info );
	}

	return( conversion_type );
}

static void
conversion_link( Conversion *conv, Imageinfo *ii )
{
	model_set( MODEL( conv ), "display_conversion", NULL );
	conversion_set_image( conv, ii );
}

Conversion *
conversion_new( Imageinfo *ii )
{
	Conversion *conv = gtk_type_new( TYPE_CONVERSION );

	conversion_link( conv, ii );

	return( conv );
}

void
conversion_error_set( Conversion *conv )
{
	if( !conv->err ) {
		conv->err = TRUE;
		SETSTR( conv->errstr, error_string );

		/* We're now blocked on error ... we can free up a bunch of 
		 * stuff.
		 */
		FREEF( im_region_free, conv->ireg );
		FREEF( im_region_free, conv->reg );
		FREEF( im_threadgroup_free, conv->tg );
		FREEF( imageinfo_destroy_nonheap, conv->display_ii );

		/*

		  This causes chunk_alloc() coredumps, not quite sure why

		  FIXME ... put in a special signal for "error state change"
		  so iimageview can change colour and show the error message

		model_changed( MODEL( conv ) );
		 */
	}
}

static void
conversion_error_clear( Conversion *conv )
{
	if( conv->err ) {
		conv->err = FALSE;
		FREE( conv->errstr );

		model_changed( MODEL( conv ) );
	}
}

/* Make the visualisation image ... eg. we im_histplot histograms, and we
 * rotate foureier images.
 */
static IMAGE *
conversion_visualise( Conversion *conv, IMAGE *in )
{
	IMAGE *out = im_open( "conversion_visualise", "p" );
	int tconv = !(conv && conv->enabled && !conv->type);

        /* Histogram type ... plot the histogram
         */
        if( tconv && in->Type == IM_TYPE_HISTOGRAM && 
		(in->Xsize == 1 || in->Ysize == 1) ) {
                IMAGE *t[3];
  
                if( im_open_local_array( out, t, 3, "conv-1", "p" ) ||
                        im_histnorm( in, t[0] ) ||
                        im_histplot( t[0], t[1] ) ) {
                        im_close( out );
                        return( NULL );
                }
  
                /* Scale to a sensible size ... aim for a height of 256
                 * elements.
                 */
                if( in->Xsize == 1 && t[1]->Xsize > 256 ) {
                        if( im_subsample( t[1], t[2], t[1]->Xsize / 256, 1 ) ) {
                                im_close( out );
                                return( NULL );
                        }
                }
                else if( in->Ysize == 1 && t[1]->Ysize > 256 ) {
                        if( im_subsample( t[1], t[2], 1, t[1]->Ysize / 256 ) ) {
                                im_close( out );
                                return( NULL );
                        }
                }
                else
                        t[2] = t[1];
  
                in = t[2];
        }

	/* IM_TYPE_FOURIER type ... pow/log scale, good for fourier 
	 * transforms.
	 */
	if( tconv && in->Type == IM_TYPE_FOURIER ) {
		IMAGE *t[2];

		if( im_open_local_array( out, t, 2, "conv-1", "p" ) ||
			im_abs( in, t[0] ) || 
			im_scaleps( t[0], t[1] ) ) {
			im_close( out );
			return( NULL );
		}

		in = t[1];
	}

	if( im_copy( in, out ) ) {
		im_close( out );
		return( NULL );
	}

	return( out );
}

/* Make a display conversion pipeline. Turn any IMAGE into a 1/3 band 
 * IM_BANDFMT_UCHAR ready for gdk_rgb_*().
 */
static IMAGE *
conversion_apply( Conversion *conv, IMAGE *in )
{
	IMAGE *out = im_open( "conversion_apply:1", "p" );

	/* FIXME ... should let other displays be used here, see
	 * ../scraps/calibrate.[hc]
	 */
	struct im_col_display *display = im_col_displays( 7 );
	static void *table = NULL;

	/* Do we do colorimetric type conversions? Look for
	 * interpret-type-toggle.
	 */
	int tconv = !(conv && conv->enabled && !conv->type);

	if( !out )
		return( NULL );

	if( conv->mag < 0 ) {
		/* Ordinary image ... use im_subsample().

			FIXME ... look for pyramid TIFFs here

		 */
		IMAGE *t = im_open_local( out, "conv:1", "p" );

		/* Don't shrink by more than the image size (ie. to less than
		 * 1 pixel).
		 */
		int xshrink = IM_MIN( -conv->mag, in->Xsize );
		int yshrink = IM_MIN( -conv->mag, in->Ysize );

		if( !t || im_subsample( in, t, xshrink, yshrink ) ) {
			im_close( out );
			return( NULL );
		}

		in = t;
	}

	/* Special case: if this is a IM_CODING_LABQ, and the display control 
	 * bar is turned off, we can go straight to IM_TYPE_RGB for speed.
	 */
	if( in->Coding == IM_CODING_LABQ && !(conv && conv->enabled) ) {
		IMAGE *t = im_open_local( out, "conv:1", "p" );

		/* Make sure fast LabQ2disp tables are built.
		 */
		if( !table ) 
			table = im_LabQ2disp_build_table( NULL, display );

		if( !t || im_LabQ2disp_table( in, t, table ) ) {
			im_close( out );
			return( NULL );
		}

		in = t;
	}

	/* Get the bands right. If we have >3, drop down to 3. If we have 2,
	 * drop down to 1.
	 */
	if( in->Coding == IM_CODING_NONE ) {
		if( in->Bands == 2 ) {
			IMAGE *t = im_open_local( out, "conv:1", "p" );

			if( !t || im_extract_band( in, t, 0 ) ) {
				im_close( out );
				return( NULL );
			}

			in = t;
		}
		else if( in->Bands > 3 ) {
			IMAGE *t[4];

			if( im_open_local_array( out, t, 4, "conv-4", "p" ) ||
				im_extract_band( in, t[0], 0 ) ||
				im_extract_band( in, t[1], 1 ) ||
				im_extract_band( in, t[2], 2 ) ||
				im_gbandjoin( t, t[3], 3 ) ) {
				im_close( out );
				return( NULL );
			}

			in = t[3];
		}
	}

	/* Interpret the Type field for colorimetric images.
	 */
	if( tconv &&
		in->Bands == 3 && in->BandFmt == IM_BANDFMT_SHORT && 
		in->Type == IM_TYPE_LABS ) {
		IMAGE *t = im_open_local( out, "conv:1", "p" );

		if( !t || im_LabS2LabQ( in, t ) ) {
			im_close( out );
			return( NULL );
		}

		in = t;
	}

	if( in->Coding == IM_CODING_LABQ ) {
		IMAGE *t = im_open_local( out, "conv:1", "p" );

		if( !t || im_LabQ2Lab( in, t ) ) {
			im_close( out );
			return( NULL );
		}

		in = t;
	}

	if( in->Coding != IM_CODING_NONE ) {
		im_close( out );
		return( NULL );
	}

	if( tconv &&
		in->Bands == 3 && in->Type == IM_TYPE_LCH ) {
		IMAGE *t[2];

                if( im_open_local_array( out, t, 2, "conv-1", "p" ) ||
			im_clip2fmt( in, t[0], IM_BANDFMT_FLOAT ) ||
			im_LCh2Lab( t[0], t[1] ) ) {
			im_close( out );
			return( NULL );
		}

		in = t[1];
	}

	if( tconv &&
		in->Bands == 3 && in->Type == IM_TYPE_YXY ) {
		IMAGE *t[2];

                if( im_open_local_array( out, t, 2, "conv-1", "p" ) ||
			im_clip2fmt( in, t[0], IM_BANDFMT_FLOAT ) ||
			im_Yxy2XYZ( t[0], t[1] ) ) {
			im_close( out );
			return( NULL );
		}

		in = t[1];
	}

	if( tconv &&
		in->Bands == 3 && in->Type == IM_TYPE_UCS ) {
		IMAGE *t[2];

                if( im_open_local_array( out, t, 2, "conv-1", "p" ) ||
			im_clip2fmt( in, t[0], IM_BANDFMT_FLOAT ) ||
			im_UCS2XYZ( t[0], t[1] ) ) {
			im_close( out );
			return( NULL );
		}

		in = t[1];
	}

	if( tconv &&
		in->Bands == 3 && in->Type == IM_TYPE_LAB ) {
		IMAGE *t[2];

                if( im_open_local_array( out, t, 2, "conv-1", "p" ) ||
			im_clip2fmt( in, t[0], IM_BANDFMT_FLOAT ) ||
			im_Lab2XYZ( t[0], t[1] ) ) {
			im_close( out );
			return( NULL );
		}

		in = t[1];
	}

	if( tconv &&
		in->Bands == 3 && in->Type == IM_TYPE_XYZ ) {
		IMAGE *t[2];

                if( im_open_local_array( out, t, 2, "conv-1", "p" ) ||
			im_clip2fmt( in, t[0], IM_BANDFMT_FLOAT ) ||
			im_XYZ2disp( t[0], t[1], display ) ) {
			im_close( out );
			return( NULL );
		}

		in = t[1];
	}

	/* Scale and offset?
	 */
	if( conv && conv->enabled ) {
		if( conv->scale != 1.0 || conv->offset != 0.0 ) {
			IMAGE *t = im_open_local( out, "conv:1", "p" );

			if( !t || im_lintra( conv->scale, in, 
				conv->offset, t ) ) {
				im_close( out );
				return( NULL );
			}

			in = t;
		}
	}

	/* Clip to uchar if not there already.
	 */
	if( in->BandFmt != IM_BANDFMT_UCHAR ) {
		IMAGE *t = im_open_local( out, "conv:1", "p" );

		if( !t || im_clip( in, t ) ) {
			im_close( out );
			return( NULL );
		}

		in = t;
	}

	/* Falsecolour. 1-band images only.
	 */
	if( conv && conv->enabled && conv->falsecolour && in->Bands == 1 ) {
		IMAGE *t = im_open_local( out, "conv:1", "p" );

		if( !t || im_falsecolour( in, t ) ) {
			im_close( out );
			return( NULL );
		}

		in = t;
	}

	/* Zoom, if necessary. 
	 */
	if( conv->mag > 1 ) {
		IMAGE *t = im_open_local( out, "conv:1", "p" );

		if( !t || im_zoom( in, t, conv->mag, conv->mag ) ) {
			im_close( out );
			return( NULL );
		}

		in = t;
	}

	/* All done!
	 */
	if( im_copy( in, out ) ) {
		im_close( out );
		return( NULL );
	}

	return( out );
}

/* Something has changed ... remake the display image stuff.
 */
void
conversion_refresh( Conversion *conv )
{
	IMAGE *im = imageinfo_get( FALSE, conv->ii );

	IMAGE *visual_im;
	IMAGE *new_display_im;
	Imageinfo *new_display_ii;
	REGION *new_reg;
	REGION *new_ireg;
	im_threadgroup_t *new_tg;

#ifdef DEBUG
	printf( "conversion_refresh: 0x%x\n", (unsigned int) conv );
#endif /*DEBUG*/

	if( conv->visual_ii )
		visual_im = imageinfo_get( FALSE, conv->visual_ii );
	else
		visual_im = NULL;

	/* Make the new stuff first.
	 */
	if( im ) {
		if( !(new_display_im = conversion_apply( conv, visual_im )) ) 
			return;
		if( !(new_display_ii = 
			imageinfo_new( NULL, new_display_im )) ) {
			im_close( new_display_im );
			return;
		}
		imageinfo_sub_add( new_display_ii, conv->visual_ii );

		if( !(new_reg = im_region_create( im )) ) 
			return;

		if( !(new_ireg = im_region_create( new_display_im )) ) {
			FREEF( im_region_free, new_reg );
			return;
		}

		if( !(new_tg = im_threadgroup_create( new_display_im )) ) {
			FREEF( im_region_free, new_ireg );
			FREEF( im_region_free, new_reg );
			return;
		}
	}

	FREEF( im_region_free, conv->ireg );
	FREEF( im_region_free, conv->reg );
	FREEF( im_threadgroup_free, conv->tg );
	FREEF( imageinfo_destroy_nonheap, conv->display_ii );

	if( visual_im ) {
		conv->display_ii = new_display_ii;
		imageinfo_dup_nonheap( conv->display_ii );

		conv->ireg = new_ireg;
		conv->reg = new_reg;
		conv->tg = new_tg;

		conv->canvas.width = new_display_im->Xsize;
		conv->canvas.height = new_display_im->Ysize;
	}	

	model_changed( MODEL( conv ) );
}

/* Remake the ink image to match ii.
 */
static void
conversion_refresh_ink( Conversion *conv, Imageinfo *ii )
{
	IMAGE *main_im = imageinfo_get( FALSE, ii );

	FREEF( imageinfo_destroy_nonheap, conv->paintbox_ink );

	if( (conv->paintbox_ink = 
		imageinfo_new_temp( reduce_context->hi, "t" )) ) {
		imageinfo_dup_nonheap( conv->paintbox_ink );

		im_initdesc( conv->paintbox_ink->im, 1, 1, main_im->Bands, 
			main_im->Bbits, main_im->BandFmt, 
			main_im->Coding, main_im->Type, 
			1.0, 1.0, 0, 0 );
		if( im_setupout( conv->paintbox_ink->im ) )
			FREEF( imageinfo_destroy_nonheap, conv->paintbox_ink );
	}

	if( conv->paintbox_ink && conv->paintbox_ink->im && 
		conv->paintbox_ink->im->data )
		memset( conv->paintbox_ink->im->data, 0, 
			IM_IMAGE_SIZEOF_LINE( conv->paintbox_ink->im ) );
}

/* Update the paintbox_text_mask.
 */
gboolean
conversion_refresh_text( Conversion *conv )
{
	const char *text = conv->paintbox_text;

	if( !text || strspn( text, " \t\b\n" ) == strlen( text ) ) {
		ierrors( "no text specified" );
		return( FALSE );
	}

	if( !conv->paintbox_font ) {
		if( !(conv->paintbox_font = 
			gdk_font_load( conv->paintbox_font_name )) ) {
			ierrors( "unable to load font \"%s\"", 
				conv->paintbox_font_name );
			return( FALSE );
		}
	}

	FREEF( imageinfo_destroy_nonheap, conv->paintbox_text_mask );

	if( !(conv->paintbox_text_mask =
		imageinfo_new_temp( reduce_context->hi, "t" )) ) 
		return( FALSE );

	imageinfo_dup_nonheap( conv->paintbox_text_mask );

	if( !imageinfo_paint_text( conv->paintbox_text_mask, 
		conv->paintbox_font, conv->paintbox_text, 
		&conv->paintbox_text_area ) )
		return( FALSE );

	return( TRUE );
}

/* Remake the visualisation image.
 */
static void
conversion_refresh_visual( Conversion *conv )
{
	IMAGE *im = imageinfo_get( FALSE, conv->ii );
	IMAGE *new_visual_im;
	Imageinfo *new_visual_ii;

	/* Make new visualization image.
	 */
	if( !(new_visual_im = conversion_visualise( conv, im )) )
		return;
	if( !(new_visual_ii = imageinfo_new( NULL, new_visual_im )) ) {
		im_close( new_visual_im );
		return;
	}
	imageinfo_sub_add( new_visual_ii, conv->ii );

	/* Junk old stuff.
	 */
	FREEF( imageinfo_destroy_nonheap, conv->visual_ii );

	/* Install new stuff.
	 */
	conv->visual_ii = new_visual_ii;
	imageinfo_dup_nonheap( conv->visual_ii );
	conv->image.width = new_visual_im->Xsize;
	conv->image.height = new_visual_im->Ysize;
}

/* Imageinfo has changed signal. The ii is the same, but the image it
 * represents may have changed from "p" to "r" or whatever. Assume size/etc.
 * stay the same.
 */
static void
conversion_ii_changed_cb( Imageinfo *ii, Conversion *conv )
{
	conversion_error_clear( conv );
	conversion_refresh_visual( conv );
	conversion_refresh( conv );
}

/* Install a new image. 
 */
void 
conversion_set_image( Conversion *conv, Imageinfo *ii )
{
	IMAGE *im;

	if( ii == conv->ii )
		return;

	conversion_error_clear( conv );

	if( ii )
		im = imageinfo_get( FALSE, ii );
	else
		im = NULL;

	/* Junk old stuff.
	 */
	FREESID( conv->changed_sid, conv->ii );
	FREEF( imageinfo_destroy_nonheap, conv->ii );
	conv->image.width = -1;
	conv->image.height = -1;

	/* Install new stuff.
	 */
	if( ii ) {
		conv->ii = ii;
		imageinfo_dup_nonheap( conv->ii );
		conv->changed_sid = gtk_signal_connect( GTK_OBJECT( ii ), 
				"changed", 
				GTK_SIGNAL_FUNC( conversion_ii_changed_cb ), 
				conv );
	}
	if( im ) {
		conv->underlay.width = im->Xsize;
		conv->underlay.height = im->Ysize;
	}

	/* Make new visualization image.
	 */
	conversion_refresh_visual( conv );

	/* Build a new ink image, if necessary.
	 */
	if( ii && !conv->paintbox_ink )
		conversion_refresh_ink( conv, ii );
	else if( im && conv->paintbox_ink ) {
		IMAGE *ink_im = imageinfo_get( FALSE, conv->paintbox_ink );

		if( im->BandFmt != ink_im->BandFmt ||
			im->Coding != ink_im->Coding ||
			im->Bands != ink_im->Bands ||
			im->Type != ink_im->Type )
			conversion_refresh_ink( conv, ii );
	}

	conversion_refresh( conv );
}

double
conversion_dmag( int mag )
{
	assert( mag != 0 );

	if( mag > 0 )
		return( mag );
	else
		return( 1.0 / (-mag) );
}

/* Zoom in and zoom out scale factors ... a little tricky with our funny -ve
 * representation for subsample.
 */
int 
conversion_double( int mag )
{
	assert( mag != -1 );

	if( mag == -3 )
		return( -2 );
	else if( mag == -2 )
		return( 1 );
	else if( mag > 0 )
		return( mag * 2 );
	else
		return( mag / 2 );
}

int 
conversion_halve( int mag )
{
	assert( mag != -1 );

	if( mag == 1 )
		return( -2 );
	else if( mag > 1 )
		return( mag / 2 );
	else
		return( mag * 2 );
}

/* Convert display to image coordinates and back.
 */
void 
conversion_disp_to_im( Conversion *conv, int dx, int dy, int *ix, int *iy )
{
	double fmag = conversion_dmag( conv->mag );

	*ix = (int) (dx / fmag);
	*iy = (int) (dy / fmag);
}

void 
conversion_im_to_disp( Conversion *conv, int ix, int iy, int *dx, int *dy )
{
	double fmag = conversion_dmag( conv->mag );

	*dx = (int) (ix * fmag);
	*dy = (int) (iy * fmag);
}

/* Same for rects.
 */
void
conversion_disp_to_im_rect( Conversion *conv, Rect *dr, Rect *ir )
{
	int brx, bry;
	Rect out;

	conversion_disp_to_im( conv, dr->left, dr->top, &out.left, &out.top );
	conversion_disp_to_im( conv, 
		IM_RECT_RIGHT( dr ), IM_RECT_BOTTOM( dr ), &brx, &bry );
	out.width = brx - out.left;
	out.height = bry - out.top;

	*ir = out;
}

void
conversion_im_to_disp_rect( Conversion *conv, Rect *ir, Rect *dr )
{
	int brx, bry;
	Rect out;

	conversion_im_to_disp( conv, ir->left, ir->top, &out.left, &out.top );
	conversion_im_to_disp( conv, 
		IM_RECT_RIGHT( ir ), IM_RECT_BOTTOM( ir ), &brx, &bry );
	out.width = brx - out.left;
	out.height = bry - out.top;

	*dr = out;
}
