/* Expressions!
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#include "ip.h"

/* Trace error_set()/_clear().
#define DEBUG_ERROR
 */

/* Trace expr_clone() 
#define DEBUG_CLONE
 */

/*
#define DEBUG
 */

/* Set of expressions containing errors.
 */
GSList *error_list = NULL;

static Expr *
expr_map_all_sub( Symbol *sym, map_expr_fn fn, void *a )
{
	if( !sym->expr )
		return( NULL );
	else 
		return( expr_map_all( sym->expr, fn, a ) );
}

/* Apply a function to a expr ... and any local exprs.
 */
Expr *
expr_map_all( Expr *expr, map_expr_fn fn, void *a )
{
	Expr *res;

	/* Apply to this expr.
	 */
	if( (res = fn( expr, a, NULL )) )
		return( res );

	/* And over any locals.
	 */
	if( expr->compile &&
		(res = (Expr *) stable_map( expr->compile->locals, 
		(symbol_map_fn) expr_map_all_sub, (void *) fn, a, NULL )) )
		return( res );

	return( NULL );
}

void *
expr_name_print( Expr *expr )
{
	printf( "expr(0x%x) ", (unsigned int) expr );
	symbol_name_print( expr->sym );
	printf( " " );

	if( expr->row ) {
		printf( "(row " );
		row_name_print( expr->row );
		printf( ") " );
	}

	return( NULL );
}

Expr *
expr_get_parent( Expr *expr )
{
	Symbol *sym_parent = symbol_get_parent( expr->sym );

	if( !sym_parent )
		return( NULL );

	return( sym_parent->expr );
}

/* Find the enclosing expr in the dynamic scope hierarchy.
 */
static Expr *
expr_get_parent_dynamic( Expr *expr )
{
	Row *row;

	if( !expr->row )
		return( expr_get_parent( expr ) );
	else if( (row = HEAPMODEL( expr->row )->row) )
		/* Enclosing row expr.
		 */
		return( row->expr );
	else {
		/* Enclosing workspace expr.
		 */
		Workspace *ws = expr->row->top_col->ws;

		return( SYMBOL( ws )->expr );
	}
}

/* Look back up to find the root expr.
 */
Expr *
expr_get_root( Expr *expr )
{
	if( is_top( expr->sym ) )
		return( expr );
	else
		return( expr_get_root( expr_get_parent( expr ) ) );
}

/* Look back up to find the root expr using the dynamic hierarchy (if it's
 * there).
 */
Expr *
expr_get_root_dynamic( Expr *expr )
{
	Expr *parent;

	if( is_top( expr->sym ) )
		return( expr );
	else if( (parent = expr_get_parent_dynamic( expr )) )
		return( expr_get_root_dynamic( parent ) );
	else
		return( NULL ); 
}

/* Clean up an expr, ready to have a new def parsed into it.
 */
void *
expr_strip( Expr *expr )
{
	expr_error_clear( expr );

	/* Break top links we're part of.
	 */
	if( slist_map( expr->static_links, 
		(SListMapFn) link_expr_destroy, NULL ) )
		return( expr );
	if( slist_map( expr->dynamic_links, 
		(SListMapFn) link_expr_destroy, NULL ) )
		return( expr );
	assert( !expr->static_links );
	assert( !expr->dynamic_links );

	/* Junk error stuff. 
	 */
	FREE( expr->errstr );

	/* Unref the compile.
	 */
	if( expr->compile )
		(void) compile_expr_link_break( expr->compile, expr );

	return( NULL );
}

void *
expr_destroy( Expr *expr )
{
	Symbol *sym = expr->sym;

#ifdef DEBUG
	printf( "expr_destroy: " );
	expr_name_print( expr );
	printf( "\n" );
#endif /*DEBUG*/

	expr_strip( expr );

	/* Break the value link.
	 */
	expr_value_destroy( expr );

	/* If it's a secondary expr, need to remove from sym.
	 */
	if( g_slist_find( sym->ext_expr, expr ) )
		 sym->ext_expr = g_slist_remove( sym->ext_expr, expr );

	if( expr->row ) {
		Row *row = expr->row;

		/* If this is the sym for a top row, kill the row too.
		 * Otherwise just break the link and wait for the next row
		 * refresh to do the kkill for us.
		 */
		if( row == row->top_row ) {
			/* Nasty: if row has us as an expr_private, we don't 
			 * want row_destroy() to try to kill us in turn. 
			 * Break the link now.
			 */
			if( row->expr_private == expr ) {
				row->expr_private = NULL;
				expr->row = NULL;
			}

			gtk_object_destroy( GTK_OBJECT( row ) );
		}
		else {
			row->expr = NULL;
			row->expr_private = NULL;
			row->sym = NULL;
			row->sym_private = NULL;

			expr->row = NULL;

			/* Make sure we will re-parse and compile any text
			 * with this sym that might have been modified from
			 * the default.
			 */
			if( row->child_rhs && row->child_rhs->itext ) {
				iText *itext = ITEXT( row->child_rhs->itext );

				if( itext->edited )
					heapmodel_set_modified( 
						HEAPMODEL( itext ), TRUE );
			}
		}
	}

	FREE( expr );

	return( NULL );
}

Expr *
expr_new( Symbol *sym )
{
	Expr *expr;

	if( !(expr = IM_NEW( NULL, Expr )) )
		return( NULL );

	expr->sym = sym;
	expr->row = NULL;
	expr->compile = NULL;

#ifdef DEBUG
	printf( "expr_new: " );
	expr_name_print( expr );
	printf( "\n" );
#endif /*DEBUG*/

	expr->static_links = NULL;
	expr->dynamic_links = NULL;

	PEPOINTE( &expr->root, &sym->base );

	expr->imageinfo = NULL;

	expr->err = FALSE;
	expr->errstr = NULL;

	return( expr );
}

/* Clone an existing expr. 
 */
Expr *
expr_clone( Symbol *sym )
{
	Expr *expr;

	if( sym->expr && sym->expr->compile ) {
		/* Make a new expr, share the compile.
		 */
                expr = expr_new( sym );
		compile_expr_link_make( sym->expr->compile, expr );
	}
	else {
		/* No existing expr to copy, make a bare one for the
		 * row, at the same scope level as sym.
		 */
                expr = expr_new( sym );
		/*
                (void) compile_new_local( expr );
		 */
	}

	return( expr );
}

/* Mark an expression as containing an error, save error_string.
 */
void *
expr_error_set( Expr *expr )
{
#ifdef DEBUG_ERROR
	printf( "expr_error_set: error in " );
	symbol_name_print( expr->sym );
	printf( ": %s\n", error_string );
#endif /*DEBUG_ERROR*/

	/* Always update the error message.
	 */
	SETSTR( expr->errstr, error_string );

	/* Was not in error? Add to error set.
	 */
	if( !expr->err ) {
		error_list = g_slist_prepend( error_list, expr );
		expr->err = TRUE;
		if( expr->row )
			row_error_set( expr->row );

		/* If this is the value of a top-level sym, note state
		 * change on symbol.
		 */
		if( is_top( expr->sym ) && expr->sym->expr == expr )
			symbol_state_change( expr->sym );
	}

	return( NULL );
}

/* Extract the error from an expression.
 */
void
expr_error_get( Expr *expr )
{
	if( !expr->err )
		ierrors( "" );
	else {
		assert( expr->errstr );
		ierrors( expr->errstr );
	}
}

/* Clear error state.
 */
void
expr_error_clear( Expr *expr )
{
	if( expr->err ) {
#ifdef DEBUG_ERROR
		printf( "expr_error_clear: " );
		symbol_name_print( expr->sym );
		printf( "\n"  );
#endif /*DEBUG_ERROR*/

		expr->err = FALSE;
		error_list = g_slist_remove( error_list, expr );
		if( expr->row )
			row_error_clear( expr->row );

		if( is_top( expr->sym ) && expr->sym->expr == expr )
			symbol_state_change( expr->sym );
	}
}

/* Mark an expr dirty.
 *
 * Two cases: if expr->row, this is part of a display. Use the row
 * stuff to mark this expr dirty. Then use symbol_dirty() to mark on from the
 * root of this row.
 *
 * Case two: this must be an expr inside a top-level ... just
 * symbol_dirty() on from that top level.
 *
 * FIXME ... we should be able to scrap this expr_get_root() ... we want the
 * 'parent' field in the Link we are probably being called from.
 */
void *
expr_dirty( Expr *expr, int serial )
{
	if( expr->row ) {
		Symbol *top_sym = expr->row->top_row->sym;

		row_dirty( expr->row );
		symbol_dirty( top_sym, serial );
	}
	else
		symbol_dirty( expr_get_root( expr )->sym, serial );

	return( NULL );
}

void *
expr_dirty_intrans( Expr *expr, int serial )
{
	if( expr->row ) {
		row_dirty_intrans( expr->row );
		symbol_dirty( expr->row->top_row->sym, serial );
	}
	else
		symbol_dirty_intrans( expr->sym, serial );

	return( NULL );
}

void
expr_tip_sub( Expr *expr, BufInfo *buf )
{
	Compile *compile = expr->compile;

	if( is_top( expr->sym ) ) 
		buf_appends( buf, "top level " );

	if( compile && is_class( compile ) ) {
		buf_appends( buf, "class " );
		if( compile->nparam == 0 )
			buf_appends( buf, "instance " );
		else
			buf_appends( buf, "definition " );

		buf_appendf( buf, "\"%s\"", MODEL( expr->sym )->name );
	}
	else if( expr->sym->type == SYM_PARAM )
		buf_appendf( buf, "parameter \"%s\"", 
			MODEL( expr->sym )->name );
	else {
		if( is_member( expr->sym ) ) 
			buf_appends( buf, "member " );

		if( compile->nparam == 0 )
			buf_appends( buf, "value " );
		else
			buf_appends( buf, "function " );

		buf_appendf( buf, "\"%s\"", MODEL( expr->sym )->name );
	}

	if( !is_top( expr->sym ) ) {
		buf_appends( buf, " of " );
		expr_tip_sub( expr_get_parent( expr ), buf );
	}
}

/* Look at an expr, make a tooltip.
 */
void
expr_tip( Expr *expr, BufInfo *buf )
{
	if( expr->row )
		row_qualified_name( expr->row, buf );
	else
		symbol_qualified_name( expr->sym, buf );
	buf_appends( buf, ": " );

	expr_tip_sub( expr, buf );
}

/* An expr has lost a value.
 */
void
expr_value_destroy( Expr *expr )
{
	/* Break ImageInfo link (if any).
	 */
	if( expr->imageinfo )
		imageinfo_expr_remove( expr, expr->imageinfo );
}

/* An expr has a new value.
 */
void
expr_value_new( Expr *expr )
{
	PElement *root = &expr->root;

#ifdef DEBUG
	printf( "expr_value_new: " );
	symbol_name_print( expr->sym );
	printf( ": " );
	pgraph( root );
#endif /*DEBUG*/

	expr_value_destroy( expr );
	if( PEISIMAGE( root ) && PEGETII( root ) ) 
		imageinfo_expr_add( PEGETII( root ), expr );
}

/* Bind unresolved refs in an expr. Bind for every enclosing dynamic scope.
 */
void
expr_resolve( Expr *expr )
{
	Expr *top = symbol_root->expr;
	Expr *i;

#ifdef DEBUG
	printf( "expr_resolve: " );
	expr_name_print( expr );
	printf( "\n" );
#endif /*DEBUG*/

	for( i = expr; i != top; i = expr_get_parent_dynamic( i ) ) 
		/* May try to resolve out through a parameter.
		 */
		if( i->compile )
			stable_resolve_dynamic( expr->compile->locals, 
				i->compile->locals );
}
