/* abstract base class for things which form the filemodel half of a 
 * filemodel/view pair
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

/* Don't compress save files.

 	FIXME ... some prebuilt libxml2s on win32 don't support libz
	compression, so don't turn this off

 */
#define DEBUG_SAVEFILE

#include "ip.h"

static ModelClass *parent_class = NULL;

static GSList *filemodel_registered = NULL;

/* Register a file model. Registered models are part of the "xxx has been
 * modified, save before quit?" check.
 */
void 
filemodel_register( Filemodel *filemodel )
{
	if( !filemodel->registered ) {
		filemodel->registered = TRUE;
		filemodel_registered = g_slist_prepend( filemodel_registered, 
			filemodel );
	}
}

static void 
filemodel_unregister( Filemodel *filemodel )
{
	if( filemodel->registered ) {
		filemodel->registered = FALSE;
		filemodel_registered = g_slist_remove( filemodel_registered, 
			filemodel );
	}
}

/* Trigger the top_load method for a filemodel.
 */
void *
filemodel_top_load( Filemodel *filemodel, 
	ModelLoadState *state, Model *parent, xmlNode *xnode )
{
	FilemodelClass *filemodel_class = 
		FILEMODEL_CLASS( GTK_OBJECT( filemodel )->klass );

	if( filemodel_class->top_load ) {
		if( !filemodel_class->top_load( filemodel, state, 
			parent, xnode ) )
			return( filemodel );
	}
	else 
		ierrors( "no _top_load() method for class \"%s\"", 
			OBJECT_CLASS_NAME( filemodel_class ) );

	return( NULL );
}

static void
filemodel_info( Model *model, BufInfo *buf )
{
	Filemodel *filemodel = FILEMODEL( model );

	MODEL_CLASS( parent_class )->info( model, buf );

	buf_appendf( buf, "filename = \"%s\"\n", NN( filemodel->filename ) );
	buf_appendf( buf, "modified = \"%s\"\n", 
		bool_to_char( filemodel->modified ) );
}

void
filemodel_set_filename( Filemodel *filemodel, const char *filename )
{
	if( filename != filemodel->filename ) {
		SETSTR( filemodel->filename, filename );
		model_changed( MODEL( filemodel ) );
	}
}

void 
filemodel_set_modified( Filemodel *filemodel, gboolean state )
{
	if( filemodel->modified != state ) {
#ifdef DEBUG
		printf( "filemodel_set_modified: %s \"%s\" %s\n", 
			OBJECT_CLASS_NAME( filemodel ), 
			NN( MODEL( filemodel )->name ),
			bool_to_char( state ) );
#endif /*DEBUG*/

		filemodel->modified = state;
		model_changed( MODEL( filemodel ) );
	}
}

static void
filemodel_destroy( GtkObject *object )
{
	Filemodel *filemodel;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_FILEMODEL( object ) );

	filemodel = FILEMODEL( object );

#ifdef DEBUG
	printf( "filemodel_destroy: %s \"%s\"\n", 
		OBJECT_CLASS_NAME( object ), NN( MODEL( filemodel )->name ) );
#endif /*DEBUG*/

	filemodel_unregister( filemodel );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

static void
filemodel_finalize( GtkObject *object )
{
	Filemodel *filemodel;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_FILEMODEL( object ) );

	filemodel = FILEMODEL( object );

#ifdef DEBUG
	printf( "filemodel_finalize: %s \"%s\"\n", 
		OBJECT_CLASS_NAME( object ), NN( MODEL( filemodel )->name ) );
#endif /*DEBUG*/

	FREE( filemodel->filename );

	if( GTK_OBJECT_CLASS( parent_class )->finalize )
		(*GTK_OBJECT_CLASS( parent_class )->finalize)( object );
}

static xmlNode *
filemodel_save( Model *model, xmlNode *xnode )
{
	Filemodel *filemodel = FILEMODEL( model );
	xmlNode *xthis;

	if( !(xthis = MODEL_CLASS( parent_class )->save( model, xnode )) )
		return( NULL );

	if( !set_sprop( xthis, "filename", filemodel->filename ) )
		return( NULL );

	return( xthis );
}

static gboolean 
filemodel_load( Model *model,
	ModelLoadState *state, Model *parent, xmlNode *xnode )
{
	Filemodel *filemodel = FILEMODEL( model );

	char buf[NAMELEN];

	if( get_sprop( xnode, "filename", buf, NAMELEN ) )
		filemodel_set_filename( filemodel, buf );

	if( !MODEL_CLASS( parent_class )->load( model, state, parent, xnode ) )
		return( FALSE );

	return( TRUE );
}

static gboolean 
filemodel_real_top_load( Filemodel *filemodel,
	ModelLoadState *state, Model *parent, xmlNode *xnode )
{
	return( TRUE );
}

static void
filemodel_class_init( FilemodelClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass*) klass;
	ModelClass *model_class = (ModelClass*) klass;

	parent_class = gtk_type_class( TYPE_MODEL );

	object_class->destroy = filemodel_destroy;
	object_class->finalize = filemodel_finalize;

	model_class->info = filemodel_info;
	model_class->save = filemodel_save;
	model_class->load = filemodel_load;
	
	klass->top_load = filemodel_real_top_load;
	klass->filetype = NULL;
	klass->default_filetype = 0;
}

static void
filemodel_init( Filemodel *filemodel )
{
	/* Init our instance fields.
	 */
	filemodel->filename = NULL;
	filemodel->modified = FALSE;
	filemodel->registered = FALSE;
	filemodel->x_off = 0;
	filemodel->y_off = 0;
	filemodel->auto_load = FALSE;
}

GtkType
filemodel_get_type( void )
{
	static GtkType filemodel_type = 0;

	if( !filemodel_type ) {
		static const GtkTypeInfo filemodel_info = {
			"Filemodel",
			sizeof( Filemodel ),
			sizeof( FilemodelClass ),
			(GtkClassInitFunc) filemodel_class_init,
			(GtkObjectInitFunc) filemodel_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		filemodel_type = 
			gtk_type_unique( TYPE_MODEL, &filemodel_info );
	}

	return( filemodel_type );
}

void
filemodel_set_offset( Filemodel *filemodel, int x_off, int y_off )
{
#ifdef DEBUG
	printf( "filemodel_set_offset: %s \"%s\" %d x %d\n", 
		OBJECT_CLASS_NAME( filemodel ), 
		NN( MODEL( filemodel )->name ),
		x_off, y_off );
#endif /*DEBUG*/

	filemodel->x_off = x_off;
	filemodel->y_off = y_off;
}

/* Save to filemodel->filename.
 */
static gboolean
filemodel_save_all_xml( Filemodel *filemodel, const char *filename )
{
	xmlDoc *xdoc;

	if( !(xdoc = xmlNewDoc( "1.0" )) ) {
		ierrors( "model_save_filename: xmlNewDoc() failed" );
		return( FALSE );
	}

#ifdef DEBUG_SAVEFILE
#else
	xmlSetDocCompressMode( xdoc, 1 );
#endif /*DEBUG_SAVEFILE*/

	if( !(xdoc->children = xmlNewDocNode( xdoc, NULL, "root", NULL )) ||
		!set_sprop( xdoc->children, "xmlns", FULL_NAMESPACE ) ) {
		ierrors( "model_save_filename: xmlNewDocNode() failed" );
		xmlFreeDoc( xdoc );
		return( FALSE );
	}

	column_set_offset( filemodel->x_off, filemodel->y_off );
	if( model_save( MODEL( filemodel ), xdoc->children ) ) {
		xmlFreeDoc( xdoc );
		return( FALSE );
	}

	prettify_tree( xdoc );
	if( (int) call_string( (call_string_fn) xmlSaveFile, 
		filename, xdoc, NULL, NULL ) == -1 ) {
		ierrors( "save of %s \"%s\" to file \"%s\" failed\n%s",
			OBJECT_CLASS_NAME( filemodel ), 
			NN( MODEL( filemodel )->name ),
			NN( filename ),
			g_strerror( errno ) );

		xmlFreeDoc( xdoc );
		return( FALSE );
	}

	xmlFreeDoc( xdoc );

	return( TRUE );
}

static gboolean
filemodel_save_all_text( Filemodel *filemodel, const char *filename )
{
	iOpenFile *of;

	if( !(of = file_open_write( filename )) ) 
		return( FALSE );

	column_set_offset( filemodel->x_off, filemodel->y_off );
	if( model_save_text( MODEL( filemodel ), of ) ) {
		file_close( of );
		return( FALSE );
	}
	file_close( of );

	return( TRUE );
}

gboolean
filemodel_save_all( Filemodel *filemodel, const char *filename )
{
	ModelClass *model_class = MODEL_CLASS( GTK_OBJECT( filemodel )->klass );

	if( model_class->save_text )
		return( filemodel_save_all_text( filemodel, filename ) );
	else if( model_class->save )
		return( filemodel_save_all_xml( filemodel, filename ) );
	else {
		ierrors( "filemodel_save_all: no save method" );
		return( FALSE );
	}
}

static gboolean
filemodel_load_all_xml( Filemodel *filemodel, 
	Model *parent, const char *filename )
{
	const char *tname = OBJECT_CLASS_NAME( filemodel );

	ModelLoadState *state;
	xmlNode *xnode;
	xmlNode *xstart;

	int major, minor, micro;

	if( !(state = model_loadstate_new( filename )) )
		return( FALSE );

	/* Check the root element for type/version compatibility.
	 */
	if( !(xnode = xmlDocGetRootElement( state->xdoc )) ||
		!xnode->nsDef ||
		!isprefix( NAMESPACE, xnode->nsDef->href ) ) {
		ierrors( "can't load XML file \"%s\"\n"
			"not a " PACKAGE " save file", filename );
		model_loadstate_destroy( state );
		return( FALSE );
	}
	if( sscanf( xnode->nsDef->href + strlen( NAMESPACE ) + 1, "%d.%d.%d",
		&major, &minor, &micro ) != 3 ) {
		ierrors( "can't load XML file \"%s\"\n"
			"unable extract version information from namespace",
			filename );
		model_loadstate_destroy( state );
		return( FALSE );
	}

	/* Don't do anything with this yet.
	printf( "major = %d, minor = %d, micro = %d\n", major, minor, micro );
	 */

	if( !(xstart = get_node( xnode, tname )) ) {
		ierrors( "can't load XML file \"%s\"\n"
			"file does not contain a %s", 
			filename, tname );
		model_loadstate_destroy( state );
		return( FALSE );
	}

	/* Set the global loadstate so the lexer can see it.
	 */
	if( filemodel_top_load( filemodel, state, parent, xstart ) ) {
		model_loadstate_destroy( state );
		return( FALSE );
	}
	model_loadstate_destroy( state );

	return( TRUE );
}

static gboolean
filemodel_load_all_text( Filemodel *filemodel, 
	Model *parent, const char *filename )
{
	iOpenFile *of;

	if( !(of = file_open_read( filename )) ) 
		return( FALSE );

	if( model_load_text( MODEL( filemodel ), parent, of ) ) {
		file_close( of );
		return( FALSE );
	}
	file_close( of );

	return( TRUE );
}

/* _init() from filename.
 */
gboolean
filemodel_load_all( Filemodel *filemodel, Model *parent, const char *filename )
{
	ModelClass *model_class = MODEL_CLASS( GTK_OBJECT( filemodel )->klass );
	char *tname = gtk_type_name( GTK_OBJECT_CLASS( model_class )->type );

	if( model_class->load_text ) {
		if( !filemodel_load_all_text( filemodel, parent, filename ) ) 
			return( FALSE );
	}
	else if( model_class->load ) {
		if( !filemodel_load_all_xml( filemodel, parent, filename ) )
			return( FALSE );
	}
	else {
		ierrors( "no _load() method for class \"%s\"", tname );
		return( FALSE );
	}

	/* Don't recomp here, we may be loading a bunch of inter-dependant
	 * files.
	 */

	return( TRUE );
}

/* Interactive stuff ... save first.
 */

static void
filemodel_inter_save_cb( iWindow *iwnd, 
	void *client, iWindowNotifyFn nfn, void *sys )
{
	Filesel *filesel = FILESEL( iwnd );
	Filemodel *filemodel = FILEMODEL( client );
	const char *fname = filesel_get_filename( filesel );

	filemodel_set_filename( filemodel, fname );
	if( !filemodel_save_all( filemodel, fname ) ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	filemodel_set_modified( filemodel, FALSE );

	nfn( sys, IWINDOW_TRUE );
}

static void
filemodel_inter_saveas_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	Filemodel *filemodel = FILEMODEL( client );
	FilemodelClass *filemodel_class = FILEMODEL_CLASS( 
		GTK_OBJECT( filemodel )->klass );

	Filesel *filesel = FILESEL( filesel_new() );

	iwindow_set_title( IWINDOW( filesel ), "Save %s %s",
		OBJECT_CLASS_NAME( filemodel ), 
		NN( MODEL( filemodel )->name ) );
	filesel_set_flags( filesel, FALSE, TRUE );
	filesel_set_type( filesel, 
		filemodel_class->filetype, filemodel_class->default_filetype ); 
	idialog_set_parent( IDIALOG( filesel ), GTK_WIDGET( iwnd ) );
	filesel_set_done( filesel, filemodel_inter_save_cb, filemodel );
	idialog_set_notify( IDIALOG( filesel ), nfn, sys );
	iwindow_build( IWINDOW( filesel ) );
	if( filemodel->filename )
		filesel_set_filename( filesel, filemodel->filename );

	gtk_widget_show( GTK_WIDGET( filesel ) );
}

void
filemodel_inter_saveas( iWindow *parent, Filemodel *filemodel )
{
	filemodel_inter_saveas_cb( parent, filemodel, 
		iwindow_notify_null, NULL );
}

void
filemodel_inter_save( iWindow *parent, Filemodel *filemodel )
{
	if( filemodel->filename ) {
		if( !filemodel_save_all( filemodel, filemodel->filename ) ) 
			box_alert( GTK_WIDGET( parent ) );
		else {
			filemodel_set_modified( filemodel, FALSE );
			box_info( GTK_WIDGET( parent ),
				"%s \"%s\" successfully saved to file \"%s\"",
				OBJECT_CLASS_NAME( filemodel ), 
				NN( MODEL( filemodel )->name ),
				filemodel->filename );
		}
	}
	else 
		filemodel_inter_saveas( parent, filemodel );
}

/* Now "empty" ... do an 'are you sure' check if modified has been set.
 */

static void
filemodel_inter_empty_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	Filemodel *filemodel = FILEMODEL( client );

	(void) model_empty( MODEL( filemodel ) );
	filemodel_set_modified( filemodel, FALSE );

	nfn( sys, IWINDOW_TRUE );
}

static void
filemodel_inter_savenempty_ok_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	iWindowSusp *susp = iwindow_susp_new( filemodel_inter_empty_cb, 
		iwnd, client, nfn, sys );

	filemodel_inter_saveas_cb( iwnd, client, iwindow_susp_comp, susp );
}

static void
filemodel_inter_savenempty_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	Filemodel *filemodel = FILEMODEL( client );
	const char *tname = OBJECT_CLASS_NAME( filemodel );

	if( filemodel->modified ) {
		if( filemodel->filename )
			box_savenosave( GTK_WIDGET( iwnd ), 
				filemodel_inter_savenempty_ok_cb, 
				filemodel_inter_empty_cb, filemodel, 
				nfn, sys,
				"%s \"%s\" (from file \"%s\") has been "
				"modified\n"
				"Do you want to save it before you junk it?",
				tname, 
				NN( MODEL( filemodel )->name ),
				NN( filemodel->filename ) );
		else
			box_savenosave( GTK_WIDGET( iwnd ), 
				filemodel_inter_savenempty_ok_cb, 
				filemodel_inter_empty_cb, filemodel, 
				nfn, sys,
				"%s \"%s\" has been modified\n"
				"Do you want to save it before you junk it?",
				tname, 
				NN( MODEL( filemodel )->name ) );
	}
	else
		filemodel_inter_empty_cb( NULL, filemodel, nfn, sys );
}

void
filemodel_inter_savenempty( iWindow *parent, Filemodel *filemodel )
{
	filemodel_inter_savenempty_cb( parent, filemodel, 
		iwindow_notify_null, NULL );
}

/* Now "close" ... easy: just savenempty, then destroy.
 */

static void
filemodel_inter_close_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	Filemodel *filemodel = FILEMODEL( client );

	FREEO( filemodel );

	nfn( sys, IWINDOW_TRUE );
}

static void
filemodel_inter_savenclose_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	iWindowSusp *susp = iwindow_susp_new( filemodel_inter_close_cb, 
		iwnd, client, nfn, sys );

	filemodel_inter_savenempty_cb( iwnd, client, iwindow_susp_comp, susp );
}

/* If savenclose is OK, make sure we have a valid workspace (and column).
 */
static void
filemodel_inter_savenclose_tidy( void *client, iWindowResult result ) 
{
	if( result == IWINDOW_TRUE ) 
		workspacegroup_pick( main_workspacegroup );
}

void
filemodel_inter_savenclose( iWindow *parent, Filemodel *filemodel )
{
	filemodel_inter_savenclose_cb( parent, filemodel, 
		filemodel_inter_savenclose_tidy, NULL );
}

/* Now "load" ... add stuff to a model from a file.
 */

static void
filemodel_inter_load_cb( iWindow *iwnd, 
	void *client, iWindowNotifyFn nfn, void *sys )
{
	Filesel *filesel = FILESEL( iwnd );
	Filemodel *filemodel = FILEMODEL( client );
	const char *filename = filesel_get_filename( filesel );
	Model *parent = MODEL( filemodel )->parent;

	filemodel_set_filename( filemodel, filename );
	if( !filemodel_load_all( filemodel, parent, filename ) ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	nfn( sys, IWINDOW_TRUE );
}

static void
filemodel_inter_loadas_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	Filemodel *filemodel = FILEMODEL( client );
	FilemodelClass *filemodel_class = FILEMODEL_CLASS( 
		GTK_OBJECT( filemodel )->klass );

	Filesel *filesel = FILESEL( filesel_new() );

	iwindow_set_title( IWINDOW( filesel ), "Load %s",
		OBJECT_CLASS_NAME( filemodel ) );
	filesel_set_flags( filesel, FALSE, TRUE );
	filesel_set_type( filesel, 
		filemodel_class->filetype, filemodel_class->default_filetype ); 
	idialog_set_parent( IDIALOG( filesel ), GTK_WIDGET( iwnd ) );
	filesel_set_done( filesel, filemodel_inter_load_cb, filemodel );
	idialog_set_notify( IDIALOG( filesel ), nfn, sys );
	iwindow_build( IWINDOW( filesel ) );
	if( filemodel->filename )
		filesel_set_filename( filesel, filemodel->filename );

	gtk_widget_show( GTK_WIDGET( filesel ) );
}

void
filemodel_inter_loadas( iWindow *parent, Filemodel *filemodel )
{
	filemodel_inter_loadas_cb( parent, filemodel, 
		iwindow_notify_null, NULL );
}

/* Finally "replace" ... empty, then load.
 */

static void
filemodel_inter_replace_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	iWindowSusp *susp = iwindow_susp_new( filemodel_inter_loadas_cb, 
		iwnd, client, nfn, sys );

	filemodel_inter_savenempty_cb( iwnd, client, iwindow_susp_comp, susp );
}

void
filemodel_inter_replace( iWindow *parent, Filemodel *filemodel )
{
	filemodel_inter_replace_cb( parent, filemodel, 
		iwindow_notify_null, NULL );
}

/* Close all registered filemodels.
 */

void
filemodel_inter_close_registered_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	if( filemodel_registered ) {
		Filemodel *filemodel = FILEMODEL( filemodel_registered->data );
		iWindowSusp *susp = iwindow_susp_new( 
			filemodel_inter_close_registered_cb, 
			iwnd, client, nfn, sys );

		filemodel_inter_savenclose_cb( iwnd, filemodel, 
			iwindow_susp_comp, susp );
	}
	else
		nfn( sys, IWINDOW_TRUE );
}

/* Mark something as having been loaded (or made) during startup. If we loaded
 * from one of the system areas, zap the filename so that we will save to the
 * user's area on changes.
 */
void
filemodel_set_auto_load( Filemodel *filemodel ) 
{
	filemodel->auto_load = TRUE;

	/* 

		FIXME ... not very futureproof

	 */
	if( filemodel->filename && 
		strstr( filemodel->filename, 
			"share" IM_DIR_SEP_STR PACKAGE ) ) {
		char *p = strrchr( filemodel->filename, IM_DIR_SEP );
		char buf[PATHLEN];

		assert( p );

		im_snprintf( buf, PATHLEN, "$HOME" IM_DIR_SEP_STR
			".$PACKAGE-$VERSION" IM_DIR_SEP_STR 
			"start" IM_DIR_SEP_STR "%s", p + 1 );
		filemodel_set_filename( filemodel, buf );
	}
}
