/* Heap management.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

/* GC on every alloc too! Extraordinarily slow. Turn on DEBUG_HEAP in ip.h
 * first. Good for spotting heap pointer errors.
#define DEBUG_HEAP_GC
 */

#include "ip.h"

static GSList *heap_all = NULL;

/* Call a function, passing in a "safe" PElement ... ie. the PElement points
 * at a fresh element which will be safe from the GC.
 */
void *
heap_safe_pointer( Heap *hi, heap_safe_pointer_fn fn, 
	void *a, void *b, void *c, void *d )
{
	Element e;
	PElement pe;
	void *result;

	e.type = ELEMENT_NOVAL;
	e.ele = NULL;
	PEPOINTE( &pe, &e );
	heap_register_element( hi, &e );

	result = fn( hi, &pe, a, b, c, d );

	heap_unregister_element( hi, &e );

	return( result );
}

/* Map a function over a piece of graph.
 */
void *
heap_map( HeapNode *hn, heap_map_fn fn, void *a, void *b )
{
	void *c;

	if( !hn )
		return( NULL );

	switch( hn->type ) {
	case TAG_APPL:
	case TAG_CONS:
		if( (c = fn( hn, a, b )) )
			return( c );

		if( GETLT( hn ) == ELEMENT_NODE && 
			(c = heap_map( GETLEFT( hn ), fn, a, b )) )
			return( c );
		if( GETRT( hn ) == ELEMENT_NODE && 
			(c = heap_map( GETRIGHT( hn ), fn, a, b )) )
			return( c );

		return( NULL );

	case TAG_REFERENCE:
	case TAG_COMPLEX:
	case TAG_GEN:
	case TAG_CLASS:
	case TAG_DOUBLE:
	case TAG_DOT:
		return( fn( hn, a, b ) );

	case TAG_SHARED:
		if( (c = fn( hn, a, b )) )
			return( c );

		return( heap_map( GETLEFT( hn ), fn, a, b ) );

	case TAG_FREE:
	default:
		assert( FALSE );

		/* Keep gcc happy.
		 */
		return( NULL );
	}
}

/* Debugging ... check that all nodes on the free list are TAG_FREE, and that
 * all other nodes are not TAG_FREE.
 */
static void
heap_check_free( Heap *hi )
{
	HeapNode *hn;
	HeapBlock *hb;
	GSList *p;

	/* Clear all the DEBUG flags.
	 */
	for( hb = hi->hb; hb; hb = hb->next ) {
		int i;

		for( i = 0; i < hb->sz; i++ ) {
			HeapNode *hn = &hb->node[i];

			hn->flgs &= FLAG_DEBUG ^ FLAG_ALL;
		}
	}

	/* Check free list.
	 */
	for( hn = hi->free; hn; hn = GETLEFT( hn ) ) {
		assert( hn->type == TAG_FREE );

		hn->flgs |= FLAG_DEBUG;
	}

	/* Check for all non-free.
	 */
	for( hb = hi->hb; hb; hb = hb->next ) {
		int i;

		for( i = 0; i < hb->sz; i++ ) {
			HeapNode *hn = &hb->node[i];

			assert( hn->type != TAG_FREE || 
				(hn->flgs & FLAG_DEBUG) );
		}
	}

	/* Test itable list.
	 */
	for( p = hi->itable; p; p = p->next ) {
		Imageinfo *ii = (Imageinfo *) p->data;

		assert( ii );
	}
}

/* Test for sanity. 
 */
int
heap_sanity( Heap *hi )
{
	heap_check_free( hi );

#ifdef DEBUG_HEAP_GC
	heap_gc( hi );
	heap_check_free( hi );
#endif /*DEBUG_HEAP_GC*/

	return( 0 );
}

/* Debugging ... check that all heaps have been closed, dump any which
 * haven't.
 */
void
heap_check_all_destroyed( void )
{
	slist_map( heap_all, (SListMapFn) dump_heapinfo, NULL );
}

/* Free a HeapBlock.
 */
static void
heapblock_free( HeapBlock *hb )
{
#ifdef DEBUG
	printf( "heapblock_free\n" );
#endif /*DEBUG*/

	if( hb->next )
		heapblock_free( hb->next );
	if( hb->node )
		FREE( hb->node );
	FREE( hb );
}

/* Free a Heap.
 */
void
heap_destroy( Heap *hi )
{
	/* Force a garbage-collect to free any image temps.
	 */
	(void) heap_gc( hi );

	/* Check all images are dead.
	 */
	slist_map( hi->itable, (SListMapFn) imageinfo_dump, NULL );

	if( hi->hb )
		heapblock_free( hi->hb );
	FREEF( g_hash_table_destroy, hi->emark );
	FREEF( g_slist_free, hi->rmark );

	heap_all = g_slist_remove( heap_all, hi );

	FREE( hi->name );

	FREEFI( gtk_timeout_remove, hi->gc_tid );

	FREE( hi );
}

/* Empty a heap block.
 */
static void
heapblock_empty( HeapBlock *hb )
{
	int i;

	/* Set as empty free-list.
	 */
	for( i = 0; i < hb->sz; i++ ) {
		HeapNode *hn = &hb->node[i];

		hn->type = TAG_FREE;
		hn->flgs = 0;
		PPUTLEFT( hn, ELEMENT_NODE, hn + 1 );
	}
	PPUTLEFT( &hb->node[hb->sz - 1], ELEMENT_NODE, NULL );
}

/* Add another HeapBlock.
 */
static gboolean
heapblock_create( Heap *hi, int sz )
{
	HeapBlock *hb;

	if( hi->nb > hi->mxb ) {
		hi->mxb = 1 + (hi->max_fn( hi ) / hi->rsz);
		if( hi->nb > hi->mxb ) 
			/* Hit limit ... caller detects full by ->free becoming
			 * NULL.
			 */
			return( TRUE );
	}

#ifdef DEBUG
	printf( "heapblock_create: new block, size %d\n", sz );
#endif /*DEBUG*/

	if( !(hb = IM_NEW( NULL, HeapBlock )) ) {
		ierrors( "out of memory" );
		return( FALSE );
	}
	hb->hi = hi;
	hb->next = NULL;
	hb->node = NULL;
	hb->sz = sz;

	if( !(hb->node = IM_ARRAY( NULL, sz, HeapNode )) ) {
		heapblock_free( hb );
		ierrors( "out of memory" );
		return( FALSE );
	}
	heapblock_empty( hb );

	/* Link to existing blocks.
	 */
	hb->next = hi->hb;
	hi->hb = hb;
	PPUTLEFT( &hb->node[hb->sz - 1], ELEMENT_NODE, hi->free );
	hi->free = &hb->node[0];
	hi->nb++;

	return( TRUE );
}

/* Create an empty heap. mxsz is maximum size of heap in units of nodes, 
 * stsz is start size, rsz is heap growth unit.
 */
Heap *
heap_new( Compile *compile, heap_max_fn max_fn, int stsz, int rsz )
{
	Heap *hi = IM_NEW( NULL, Heap );

	if( !hi )
		return( NULL );
	hi->compile = compile;
	hi->max_fn = max_fn;
	hi->mxb = -1;
	hi->rsz = rsz;
	hi->nb = 0;
	hi->hb = NULL;
	hi->free = NULL;
	hi->emark = g_hash_table_new( NULL, g_direct_equal );
	hi->rmark = NULL;
	hi->nfree = 0;
	hi->ncells = 0;
	hi->itable = NULL;
	hi->serial = 0;
	hi->filled = FALSE;
	hi->name = NULL;
	hi->gc_tid = 0;

	if( !heapblock_create( hi, stsz ) ) {
		heap_destroy( hi );
		return( NULL );
	}

	if( compile )
		SETSTR( hi->name, MODEL( compile->sym )->name );

	heap_all = g_slist_prepend( heap_all, hi );

	/* Can now set max blocks.
	 */
	hi->mxb = 1 + (hi->max_fn( hi ) / rsz);

	return( hi );
}

/* Set flags on a heap.
 */
void
heap_set( Heap *hi, NodeFlags setmask )
{
	HeapBlock *hb;
	int i;

	for( hb = hi->hb; hb; hb = hb->next ) 
		for( i = 0; i < hb->sz; i++ )
			hb->node[i].flgs |= setmask;
}

/* Clear flags on a heap.
 */
void
heap_clear( Heap *hi, NodeFlags clearmask )
{
	HeapBlock *hb;
	int i;
	int cmask = clearmask ^ FLAG_ALL;

	for( hb = hi->hb; hb; hb = hb->next ) 
		for( i = 0; i < hb->sz; i++ )
			hb->node[i].flgs &= cmask;
}

/* Allocate a new serial number for a heap. On return, we guarantee that 
 * hi->serial is a value not used by any nodes in the heap.
 */
int
heap_serial_new( Heap *hi )
{
	hi->serial += 1;
	if( hi->serial > FLAG_SERIAL ) {
		hi->serial = 1;

		/* Expensive! But does not happen too often.
		 */
		heap_clear( hi, FLAG_SERIAL );
	}

	return( hi->serial );
}

static void *
image_clear( Imageinfo *ii )
{
	ii->marked = FALSE;

	return( NULL );
}

static void
heap_mark_image( Imageinfo *ii, Heap *hi )
{
	/* If it's a zombie, should have no heap pointers to it.
	 */
	assert( !ii->zombie );

	/* Make sure it's on this heap's image table.
	 */
	assert( ii->hi == hi );

	imageinfo_mark( ii );
}

/* Mark a tree. Avoid recursion because of the danger of C stack overflow on 
 * large heaps.
 */
static void 
heap_mark_tree( Heap *hi, HeapNode *hn )
{
	GSList *pending = NULL;

	pending = g_slist_prepend( pending, hn );

	while( pending ) {
		hn = (HeapNode *) pending->data;
		pending = g_slist_remove( pending, hn );

		/* Chase down the LHS of the nodes, add the RHS nodes we pass 
		 * to the pending list.
		 */
		for(;;) {
			if( hn->flgs & FLAG_MARK ) 
				break;

			hn->flgs |= FLAG_MARK;

			/* Don't modify hn for the do-nothing case: we'll
			 * break on the next loop.
			 */
			switch( hn->type ) {
			case TAG_GEN:
			case TAG_COMPLEX:
			case TAG_CLASS:
			case TAG_APPL:
			case TAG_CONS:
				if( hn->rtype == ELEMENT_IMAGE ) {
					Imageinfo *ii;

					if( (ii = (Imageinfo *) 
						hn->body.ptrs.right) )
						heap_mark_image( ii, hi );
				}
				if( hn->ltype == ELEMENT_IMAGE ) {
					Imageinfo *ii;

					if( (ii = (Imageinfo *) 
						hn->body.ptrs.left) )
						heap_mark_image( ii, hi );
				}

				if( hn->rtype == ELEMENT_NODE ) {
					if( hn->ltype == ELEMENT_NODE ) {
						pending = g_slist_prepend( 
							pending, 
							GETRIGHT( hn ) );
						hn = GETLEFT( hn );
					}
					else
						hn = GETRIGHT( hn );
				}
				else if( hn->ltype == ELEMENT_NODE ) 
					hn = GETLEFT( hn );

				break;

			case TAG_DOUBLE:
			case TAG_DOT:
				break;

			case TAG_SHARED:
			case TAG_REFERENCE:
				if( hn->ltype == ELEMENT_NODE ) 
					hn = GETLEFT( hn );
				break;

			case TAG_FREE:
			default:
				assert( FALSE );
			}
		}
	}
}

/* Mark an element.
 */
static void *
heap_mark_pelement( PElement *base, Heap *hi )
{
	if( PEISIMAGE( base ) && PEGETII( base ) ) 
		heap_mark_image( PEGETII( base ), hi );
	else if( PEISNODE( base ) ) 
		heap_mark_tree( hi, PEGETVAL( base ) );

	return( NULL );
}

/* Mark an element.
 */
static void 
heap_mark_element( void *key, void *value, Heap *hi )
{
	Element *root = (Element *) value;
	PElement base;

	PEPOINTE( &base, root );
	(void) heap_mark_pelement( &base, hi ); 
}

/* Mark a reduce context ... the heapnodes on the spine stack etc.
 */
static void *
heap_mark_reduce( Reduce *rc, Heap *hi )
{
	int i;

#ifdef DEBUG
	printf( "heap_mark_reduce: marking %d stack elements\n", rc->sp );
#endif /*DEBUG*/

	for( i = 0; i < rc->sp; i++ )
		heap_mark_tree( hi, rc->nstack[i] );

	return( NULL );
}

/* Look over the image table, freeing unreferenced images.
 */
static void *
heap_free_images( Imageinfo *ii )
{
	/* Look for unmarked images which aren't zombied.
	 */
	if( !ii->marked && !ii->zombie ) {
#ifdef DEBUG
		printf( "heap_gc: closing unreferenced image \"%s\"\n",
			ii->im->filename );
#endif /*DEBUG*/

		imageinfo_destroy_heap( ii );

		/* Indicate that we closed to cause a remap. 
		 */
		return( ii );
	}

	return( NULL );
}

/* Do a garbage collect.
 */
gboolean
heap_gc( Heap *hi )
{
	HeapBlock *hb;
	int nfree;
	int ncells;
	int nblocks;

#ifdef DEBUG
	static GTimer *GC_timer = NULL;

	if( !GC_timer )
		GC_timer = g_timer_new();

	g_timer_reset( GC_timer );

	printf( "heap_gc: starting GC for heap %s\n", hi->name );
#endif /*DEBUG*/

	/* Clear marks on images. Nodes should all be clear already.
	 */
	if( slist_map( hi->itable, (SListMapFn) image_clear, NULL ) )
		return( FALSE );

	/* All flgs should be clear, so just mark.
	 */
	g_hash_table_foreach( hi->emark, (GHFunc) heap_mark_element, hi );
	slist_map( hi->rmark, (SListMapFn) heap_mark_reduce, hi );

	/* And sweep up unmarked into new free list.
	 */
	hi->free = NULL;
	ncells = nfree = nblocks = 0;
	for( hb = hi->hb; hb; hb = hb->next ) {
		int i;

		for( i = 0; i < hb->sz; i++ ) {
			HeapNode *hn = &hb->node[i];

			if( !(hn->flgs & FLAG_MARK) ) {
				hn->type = TAG_FREE;
				hn->flgs = 0;
				PPUTLEFT( hn, ELEMENT_NODE, hi->free );
#ifdef DEBUG_HEAP_GC
				/* Not necessary, but may be helpful to zap
				 * any pointer in there.
				 */
				PPUTRIGHT( hn, ELEMENT_NODE, NULL );
#endif /*DEBUG_HEAP_GC*/
				hi->free = hn;
				nfree++;
			}

			hn->flgs &= FLAG_MARK ^ FLAG_ALL;
		}

		ncells += hb->sz;
		nblocks += 1;
	}
	hi->ncells = ncells;
	hi->nfree = nfree;

	/* Repeatedly close unused images. We can't map, since each close can
	 * trigger other closes.
	 */
	while( slist_map( hi->itable, (SListMapFn) heap_free_images, hi ) )
		;

#ifdef DEBUG
	printf( "heap_gc: %d cells in %d blocks, %d in use\n",
		ncells, nblocks, ncells - nfree );
	printf( "(GC took %gs)\n",  g_timer_elapsed( GC_timer, NULL ) );
#endif 

	return( TRUE );
}

static gint
heap_gc_request_cb( Heap *hi )
{
	hi->gc_tid = 0;

	if( !heap_gc( hi ) )
		printf( "help! delayed GC failed!\n" );

	/* Update space free, since tooltip shows heap stats.
	 */
	mainw_free_update();

	return( FALSE );
}

/* Request a delayed garbage collect.
 */
void
heap_gc_request( Heap *hi )
{
	FREEFI( gtk_timeout_remove, hi->gc_tid );
	hi->gc_tid = gtk_timeout_add( 1000, 
		(GtkFunction) heap_gc_request_cb, hi );
}

/* Register a pointer into a heap.
 */
void
heap_register_element( Heap *hi, Element *root )
{
	/* Make sure we're on the GC list.
	 */
	if( !g_hash_table_lookup( hi->emark, root ) ) {
		g_hash_table_insert( hi->emark, root, root );
#ifdef DEBUG
		printf( "heap_register_element: %d pointers\n",
			g_hash_table_size( hi->emark ) );
#endif 
	}
}

/* Unregister a pointer into a heap.
 */
void
heap_unregister_element( Heap *hi, Element *root )
{
	if( g_hash_table_lookup( hi->emark, root ) ) {
		g_hash_table_remove( hi->emark, root );
#ifdef DEBUG
		printf( "heap_unregister_element: %d pointers\n",
			g_hash_table_size( hi->emark ) );
#endif 
	}
}

/* Register a Reduce working on this heap.
 */
void
heap_register_reduce( Heap *hi, Reduce *rc )
{
	/* Make sure we're on the GC list.
	 */
	if( !g_slist_find( hi->rmark, rc ) ) {
		hi->rmark = g_slist_prepend( hi->rmark, rc );
#ifdef DEBUG
		printf( "heap_register_reduce: %d pointers\n",
			g_slist_length( hi->rmark ) );
#endif 
	}
}

/* Unregister a reduce context.
 */
void
heap_unregister_reduce( Heap *hi, Reduce *rc )
{
	if( g_slist_find( hi->rmark, rc ) ) {
		hi->rmark = g_slist_remove( hi->rmark, rc );
#ifdef DEBUG
		printf( "heap_unregister_reduce: %d pointers\n",
			g_slist_length( hi->rmark ) );
#endif 
	}
}

/* Allocate a new HeapNode.
 */
HeapNode *
heap_getmem( Heap *hi ) 
{
	HeapNode *hn;
	int pcused;

	/* Easy case ... this should be handled by the NEWNODE macro, but do
	 * it here as well just in case.
	 */
	if( hi->free ) {
		EXTRACTNODE( hi, hn );
		return( hn );
	}

#ifdef DEBUG
	printf( "heap_getmem: GC on full heap for heap %s\n", hi->name );
#endif /*DEBUG*/

	/* Try a GC.
	 */
	if( !heap_gc( hi ) )
		return( NULL );

	/* Is heap over 80% full? Add another heap block.
	 */
	pcused = 100 * (hi->ncells - hi->nfree) / hi->ncells;
	if( pcused > 80 ) {
		if( !heapblock_create( hi, hi->rsz ) )
			return( NULL );
	}
	if( !hi->free ) {
		ierrors( "heap size limit reached --- runaway recursion?" );
		hi->filled = TRUE;
		return( NULL );
	}

	EXTRACTNODE( hi, hn );
	return( hn );
}

gboolean
heap_bool_new( Heap *hi, gboolean val, PElement *out )
{
	PEPUTP( out, ELEMENT_BOOL, val );

	return( TRUE );
}

/* Write a real to an element.
 */
gboolean
heap_real_new( Heap *hi, double in, PElement *out )
{
	HeapNode *hn;

	if( NEWNODE( hi, hn ) )
		return( FALSE );
	hn->type = TAG_DOUBLE;
	hn->body.num = in;

	PEPUTP( out, ELEMENT_NODE, hn );

	return( TRUE );
}

/* Write an image to an element.
 */
gboolean
heap_image_new( Heap *hi, Imageinfo *ii, PElement *out )
{
	PEPUTP( out, ELEMENT_IMAGE, ii );

	return( TRUE );
}

/* Write an element to an element.
 */
gboolean
heap_element_new( Heap *hi, Element *e, PElement *out )
{
	PEPUTE( out, e );

	return( TRUE );
}

/* Make a complex node from two elements. 
 */
gboolean
heap_complex_element_new( Heap *hi, 
	PElement *rp, PElement *ip, PElement *out )
{
	HeapNode *hn;

	if( NEWNODE( hi, hn ) )
		return( FALSE );
	hn->type = TAG_COMPLEX;
	PPUT( hn, PEGETTYPE( rp ), PEGETVAL( rp ), 
		PEGETTYPE( ip ), PEGETVAL( ip ) ); 

	PEPUTP( out, ELEMENT_NODE, hn );

	return( TRUE );
}

/* Make a complex node.
 */
gboolean
heap_complex_new( Heap *hi, double rp, double ip, PElement *out )
{
	Element dummy;
	PElement t;

	/* Form complex node.
	 */
	dummy.type = ELEMENT_NOVAL;
	dummy.ele = NULL;
	PEPOINTE( &t, &dummy );
	if( !heap_complex_element_new( hi, &t, &t, out ) )
		return( FALSE );

	/* Install real and imag parts.
	 */
	PEPOINTLEFT( PEGETVAL( out ), &t );
	if( !heap_real_new( hi, rp, &t ) )
		return( FALSE );
	PEPOINTRIGHT( PEGETVAL( out ), &t );
	if( !heap_real_new( hi, ip, &t ) )
		return( FALSE );

	return( TRUE );
}

gboolean
heap_dot_tag_new( Heap *hi, const char *tag, Compile *compile, PElement *out )
{
	HeapNode *hn1;

	if( NEWNODE( hi, hn1 ) )
		return( FALSE );
	hn1->type = TAG_DOT;
	PPUT( hn1, ELEMENT_TAG, tag, ELEMENT_COMPILEREF, compile ); 
	PEPUTP( out, ELEMENT_NODE, hn1 );

	return( TRUE );
}

gboolean
heap_dot_sym_new( Heap *hi, Symbol *sym, Compile *compile, PElement *out )
{
	HeapNode *hn1;

	if( NEWNODE( hi, hn1 ) )
		return( FALSE );
	hn1->type = TAG_DOT;
	PPUT( hn1, ELEMENT_SYMREF, sym, ELEMENT_COMPILEREF, compile ); 
	PEPUTP( out, ELEMENT_NODE, hn1 );

	return( TRUE );
}

/* Set list to [].
 */
void
heap_list_init( PElement *list )
{
	PEPUTP( list, ELEMENT_ELIST, NULL );
}

/* Add new node to list, point data at new CONS LHS.
 */
gboolean
heap_list_add( Heap *hi, PElement *list, PElement *data )
{
	HeapNode *hn;

	/* Build CONS node.
	 */
	if( NEWNODE( hi, hn ) )
		return( FALSE );
	hn->type = TAG_CONS;
	PPUTLEFT( hn, ELEMENT_NOVAL, NULL );
	PEPUTRIGHT( hn, list );
	PEPUTP( list, ELEMENT_NODE, hn );

	/* Point data to new LHS.
	 */
	PEPOINTLEFT( hn, data );

	return( TRUE );
}

/* Move list on to the next RHS. list points at [], or pointer to next node.
 */
gboolean
heap_list_next( PElement *list )
{
	HeapNode *hn = PEGETVAL( list );

	if( hn ) {
		PEPOINTRIGHT( hn, list );
		return( TRUE );
	}
	else
		return( FALSE );
}

gboolean
heap_list_cat( Reduce *rc, PElement *a, PElement *b, PElement *out )
{
	PElement list = *out;

	REDUCE_CATCH_START( FALSE );
	reduce_clone_list( rc, a, &list );
	PEPUTPE( &list, b );
	REDUCE_CATCH_STOP;

	return( TRUE );
}

/* Start off a function application.
 */
void
heap_appl_init( PElement *base, PElement *func )
{
	PEPUTPE( base, func );
}

/* Add a new parameter to a function application. base points at the
 * function built so far ... update base to point to new node (old base
 * becomes LHS), return parm pointing to new RHS
 */
gboolean
heap_appl_add( Heap *hi, PElement *base, PElement *parm )
{
	HeapNode *hn;

	/* Build appl node.
	 */
	if( NEWNODE( hi, hn ) )
		return( FALSE );
	hn->type = TAG_APPL;
	PEPUTLEFT( hn, base );
	PPUTRIGHT( hn, ELEMENT_ELIST, NULL );
	PEPUTP( base, ELEMENT_NODE, hn );

	/* Point parm to new RHS.
	 */
	PEPOINTRIGHT( hn, parm );

	return( TRUE );
}

/* Make a string.
 */
gboolean
heap_string_new( Heap *hi, const char *str, PElement *out )
{
	PElement list = *out;
	const int n = strlen( str );
	int i;

	/* Make first RHS ... the end of the list. 
	 */
	heap_list_init( &list ); 

	/* Build a CONS node for each element. 
	 */
	for( i = 0; i < n; i++ ) {
		PElement t;

		if( !heap_list_add( hi, &list, &t ) )
			return( FALSE );
		PEPUTP( &t, ELEMENT_CHAR, (int) str[i] );
		(void) heap_list_next( &list );
	}

	return( TRUE );
}

/* Make a [[char]].
 */
gboolean
heap_lstring_new( Heap *hi, GSList *labels, PElement *out )
{
	PElement list = *out;
	const int n = g_slist_length( labels );
	int i;

	/* Make first RHS ... the end of the list. 
	 */
	heap_list_init( &list ); 

	/* Build a CONS node for each element. 
	 */
	for( i = 0; i < n; i++ ) {
		PElement t;

		if( !heap_list_add( hi, &list, &t ) ||
			!heap_string_new( hi, 
				g_slist_nth_data( labels, i ), &t ) )
			return( FALSE );
		(void) heap_list_next( &list );
	}

	return( TRUE );

}

/* Make a realvec.
 */
gboolean
heap_realvec_new( Heap *hi, int n, double *vec, PElement *out )
{
	PElement list = *out;
	int i;

	/* Make first RHS ... the end of the list. 
	 */
	heap_list_init( &list );

	/* Build a CONS node for each element. 
	 */
	for( i = 0; i < n; i++ ) {
		PElement t;

		if( !heap_list_add( hi, &list, &t ) )
			return( FALSE );
		if( !heap_real_new( hi, vec[i], &t ) )
			return( FALSE );
		(void) heap_list_next( &list );
	}

	return( TRUE );
}

/* Make a matrix.
 */
gboolean
heap_matrix_new( Heap *hi, 
	int xsize, int ysize, double *vec, PElement *out )
{
	PElement list = *out;
	int y, i;

	/* Make first RHS ... the end of the list. 
	 */
	heap_list_init( &list );

	/* Build a CONS node for each element. 
	 */
	for( i = 0, y = 0; y < ysize; y++ ) {
		PElement t;

		if( !heap_list_add( hi, &list, &t ) )
			return( FALSE );
		if( !heap_realvec_new( hi, xsize, vec + i, &t ) )
			return( FALSE );
		i += xsize;
		(void) heap_list_next( &list );
	}

	return( TRUE );
}

/* Map over a heap list. Reduce the list spine as we go, don't reduce the
 * heads. Return base on error, or whatever the user function returns.
 */
void *
heap_map_list( PElement *base, heap_map_list_fn fn, void *a, void *b )
{
	Reduce *rc = reduce_context;
	PElement e = *base;

	if( !reduce_pelement( rc, reduce_spine, &e ) )
		return( base );

	if( !PEISLIST( &e ) ) {
		ierrors( "not []" );
		return( base );
	}

	while( PEISFLIST( &e ) ) {
		PElement head;
		void *res;

		/* Apply user function to the head.
		 */
		PEGETHD( &head, &e );
		if(( res = fn( &head, a, b )) )
			return( res );

		/* Reduce the tail.
		 */
		PEGETTL( &e, &e );
		if( !reduce_pelement( rc, reduce_spine, &e ) )
			return( base );
	}

	if( !PEISELIST( &e ) ) {
		ierrors( "no [] at end of list" );
		return( base );
	}

	return( NULL );
}

/* Evaluate a PElement into a string buffer. 
 */
gboolean
heap_get_string( PElement *base, char *buf, int n )
{
	Reduce *rc = reduce_context;

	REDUCE_CATCH_START( FALSE );
	(void) reduce_get_string( reduce_context, base, buf, n );
	REDUCE_CATCH_STOP; 

	return( TRUE );
}

/* Evaluate a PElement to a [[char]]. 
 */
gboolean
heap_get_lstring( PElement *base, GSList **labels )
{
	Reduce *rc = reduce_context;

	REDUCE_CATCH_START( FALSE );
	(void) reduce_get_lstring( reduce_context, base, labels );
	REDUCE_CATCH_STOP; 

	return( TRUE );
}

/* Get an element as a bool. 
 */
gboolean
heap_get_bool( PElement *base, gboolean *out )
{
	Reduce *rc = reduce_context;

	REDUCE_CATCH_START( FALSE );
	*out = reduce_get_bool( reduce_context, base );
	REDUCE_CATCH_STOP; 

	return( TRUE );
}

/* Get an element as a real. 
 */
gboolean
heap_get_real( PElement *base, double *out )
{
	Reduce *rc = reduce_context;

	REDUCE_CATCH_START( FALSE );
	*out = reduce_get_real( reduce_context, base );
	REDUCE_CATCH_STOP; 

	return( TRUE );
}

/* Get an element as a class ... just reduce and typecheck.
 */
gboolean
heap_get_class( PElement *base, PElement *out )
{
	Reduce *rc = reduce_context;

	REDUCE_CATCH_START( FALSE );
	reduce_get_class( reduce_context, base );
	REDUCE_CATCH_STOP; 

	/* Point out at base ... for consistency with other getters.
	 */
	*out = *base;

	return( TRUE );
}

/* Get an element as an image.
 */
gboolean
heap_get_image( PElement *base, Imageinfo **out )
{
	Reduce *rc = reduce_context;

	REDUCE_CATCH_START( FALSE );
	*out = reduce_get_image( reduce_context, base );
	REDUCE_CATCH_STOP; 

	return( TRUE );
}

/* Get an element as a realvec. Return -1 on error, or length of vector.
 */
int
heap_get_realvec( PElement *base, double *buf, int n )
{
	Reduce *rc = reduce_context;
	int l;

	REDUCE_CATCH_START( -1 );
	l = reduce_get_realvec( reduce_context, base, buf, n );
	REDUCE_CATCH_STOP; 

	return( l );
}

/* Get an element as a matrix. Return -1 on error, or length of buffer used. 
 * Write xsize/ysize to args.
 */
gboolean
heap_get_matrix_size( PElement *base, int *xsize, int *ysize )
{
	Reduce *rc = reduce_context;

	REDUCE_CATCH_START( FALSE );
	(void) reduce_get_matrix_size( reduce_context, base, xsize, ysize );
	REDUCE_CATCH_STOP; 

	return( TRUE );
}

/* Get an element as a matrix. Return -1 on error, or length of buffer used. 
 * Write xsize/ysize to args.
 */
gboolean
heap_get_matrix( PElement *base, double *buf, int n, int *xsize, int *ysize )
{
	Reduce *rc = reduce_context;

	REDUCE_CATCH_START( FALSE );
	(void) reduce_get_matrix( reduce_context, base, buf, n, xsize, ysize );
	REDUCE_CATCH_STOP; 

	return( TRUE );
}

gboolean 
heap_iselist( PElement *base, gboolean *out ) 
{
	Reduce *rc = reduce_context;

	REDUCE_CATCH_START( FALSE );
	*out = reduce_iselist( rc, base );
	REDUCE_CATCH_STOP; 

	return( TRUE );
}

/* Do a get, check it's OK. We don't get very much, in case it's a long
 * string and will take a while to eval.
 */
gboolean 
heap_isstring( PElement *base, gboolean *out ) 
{
	Reduce *rc = reduce_context;

	REDUCE_CATCH_START( FALSE );
	*out = reduce_isstring( rc, base );
	REDUCE_CATCH_STOP; 

	return( TRUE );
}

gboolean 
heap_isrealvec( PElement *base, gboolean *out ) 
{
	Reduce *rc = reduce_context;

	REDUCE_CATCH_START( FALSE );
	*out = reduce_isrealvec( rc, base );
	REDUCE_CATCH_STOP; 

	return( TRUE );
}

gboolean 
heap_ismatrix( PElement *base, gboolean *out ) 
{
	Reduce *rc = reduce_context;

	REDUCE_CATCH_START( FALSE );
	*out = reduce_ismatrix( rc, base );
	REDUCE_CATCH_STOP; 

	return( TRUE );
}

gboolean
heap_isclass( PElement *base, gboolean *out )
{
	Reduce *rc = reduce_context;

	REDUCE_CATCH_START( FALSE );
	*out = reduce_isclass( rc, base );
	REDUCE_CATCH_STOP; 

	return( TRUE );
}

gboolean
heap_isinstanceof_exact( const char *name, PElement *klass, gboolean *out )
{
	Reduce *rc = reduce_context;

	REDUCE_CATCH_START( FALSE );
	*out = reduce_isinstanceof_exact( rc, name, klass );
	REDUCE_CATCH_STOP; 

	return( TRUE );
}

gboolean
heap_isinstanceof( const char *name, PElement *klass, gboolean *out )
{
	Reduce *rc = reduce_context;

	REDUCE_CATCH_START( FALSE );
	*out = reduce_isinstanceof( rc, name, klass );
	REDUCE_CATCH_STOP; 

	return( TRUE );
}

int
heap_list_length( PElement *base )
{
	Reduce *rc = reduce_context;
	int result;

	REDUCE_CATCH_START( -1 );
	result = reduce_list_length( rc, base );
	REDUCE_CATCH_STOP; 

	return( result );
}

gboolean
heap_list_index( PElement *base, int n, PElement *out )
{
	Reduce *rc = reduce_context;

	REDUCE_CATCH_START( FALSE );
	reduce_list_index( rc, base, n, out );
	REDUCE_CATCH_STOP; 

	return( TRUE );
}

gboolean
heap_reduce_strict( PElement *base )
{
	Reduce *rc = reduce_context;

	trace_reset();
	REDUCE_CATCH_START( FALSE );
	reduce_spine_strict( rc, base );
	REDUCE_CATCH_STOP; 
	trace_check();

	return( TRUE );
}

/* hn is a node in a compiled function, out is part of a node in reduce
 * space to which it should be copied. 
 *
 * Have to be careful to copy sym pointers in nodes from compile heap.
 */
static gboolean
heap_copy_node( Heap *hi, HeapNode *ri[], HeapNode *hn, PElement *out )
{
	HeapNode *hn1;
	PElement pleft, pright;
	int i;

	/* Look for relocation nodes.
	 */
	if( hn->type == TAG_SHARED ) {
		/* RHS of SHARE is the index of this share node.
		 */
		i = (int) GETRIGHT( hn );

		/* Skip to shared section.
		 */
		hn = GETLEFT( hn );

		/* Copy and link on this node.
		 */
		if( NEWNODE( hi, hn1 ) )
			return( FALSE );
		*hn1 = *hn;
		PEPUTP( out, ELEMENT_NODE, hn1 );

		/* Note pointer in relocation table.
		 */
		ri[i] = hn1;
	}
	else if( hn->type == TAG_REFERENCE ) {
		/* Must have already copied this SHARE, just link back.
		 */
		hn1 = GETLEFT( hn );
		i = (int) GETRIGHT( hn1 );
		PEPUTP( out, ELEMENT_NODE, ri[i] );

		/* Done!
		 */
		return( TRUE );
	}
	else {
		/* Copy and link on this node.
		 */
		if( NEWNODE( hi, hn1 ) )
			return( FALSE );
		*hn1 = *hn;
		PEPUTP( out, ELEMENT_NODE, hn1 );
	}

	/* If it's a DOUBLE, no more to do.
	 */
	if( hn->type == TAG_DOUBLE )
		return( TRUE );

	if( hn->ltype != ELEMENT_NODE && hn->rtype == ELEMENT_NODE ) {
		/* Right pointer only. Zap pointer so we can GC
		 * safely.
		 */
		hn1->rtype = ELEMENT_CHAR;

		/* Recurse for RHS of node.
		 */
		PEPOINTRIGHT( hn1, &pright );
		if( !heap_copy_node( hi, ri, GETRIGHT( hn ), &pright ) )
			return( FALSE );
	}
	else if( hn->ltype == ELEMENT_NODE && hn->rtype != ELEMENT_NODE ) {
		/* Left pointer only. Zap pointer so we can GC
		 * safely.
		 */
		hn1->ltype = ELEMENT_CHAR;

		/* Recurse for LHS of node.
		 */
		PEPOINTLEFT( hn1, &pleft );
		if( !heap_copy_node( hi, ri, GETLEFT( hn ), &pleft ) )
			return( FALSE );
	}
	else if( hn->ltype == ELEMENT_NODE && hn->rtype == ELEMENT_NODE ) {
		/* Both pointers. Zap pointers so we can GC safely.
		 */
		hn1->ltype = ELEMENT_CHAR;
		hn1->rtype = ELEMENT_CHAR;

		/* Recurse for boths sides of node.
		 */
		PEPOINTLEFT( hn1, &pleft );
		PEPOINTRIGHT( hn1, &pright );
		if( !heap_copy_node( hi, ri, GETLEFT( hn ), &pleft ) ||
			!heap_copy_node( hi, ri, GETRIGHT( hn ), &pright ) )
			return( FALSE );
	}

	return( TRUE );
}

/* Copy a compiled graph into the main reduce space. Overwrite the node at
 * out.
 */
gboolean
heap_copy( Heap *hi, Element *root, PElement *out )
{
	HeapNode *ri[ MAX_RELOC ];

	switch( root->type ) {
	case ELEMENT_NODE:
		/* Need a tree copy.
		 */
		if( !heap_copy_node( hi, &ri[0], (HeapNode *) root->ele, out ) )
			return( FALSE );
		break;

	case ELEMENT_SYMBOL:
	case ELEMENT_CHAR:
	case ELEMENT_BOOL:
	case ELEMENT_BINOP:
	case ELEMENT_SYMREF:
	case ELEMENT_COMPILEREF:
	case ELEMENT_CONSTRUCTOR:
	case ELEMENT_UNOP:
	case ELEMENT_COMB:
	case ELEMENT_TAG:
	case ELEMENT_ELIST:
	case ELEMENT_IMAGE:
		/* Copy value.
		 */
		PEPUTP( out, root->type, root->ele );
		break;

	case ELEMENT_NOVAL:
		/* Nothing to copy. 
		 */
		break;

	default:
		assert( FALSE );
	}

	return( TRUE );
}
