/* Imagedisplay widget code.
 *
 * We do all scrolling/clipping ourselves ... can't use scrolledwindow with a
 * viewport, as we want images larger than 32k pixels.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

/* Just trace geometry changes.
#define DEBUG_GEO
 */

/* Define this to do real pointer grabs for btn1 events. Can make debugging 
 * tricky.
#define GRAB
 */

/* Define to trace button press events.
#define EVENT
 */

#include "ip.h"

/* Our signals.
 */
enum {
	EXPOSE_START,	/* Expose comes in */
	REPAINT,	/* Repaint action happens */
	LAST_SIGNAL
};
static guint imagedisplay_signals[LAST_SIGNAL] = { 0 };

static GtkWidgetClass *parent_class = NULL;

/* We leave a focus border around the edge of the image ... this width.
 */
const int imagedisplay_focus_border = 2;

#ifdef DEBUG_GEO
static void
imagedisplay_rect_print( Rect *rect )
{
	printf( "left=%d, top = %d, width = %d, height=%d",
		rect->left, rect->top, rect->width, rect->height );
}

static void
imagedisplay_geo_print( Imagedisplay *id )
{
	printf( "   screen: " );
	imagedisplay_rect_print( &id->screen );
	printf( "\n" );

	printf( "   visible: " );
	imagedisplay_rect_print( &id->visible );
	printf( "\n" );

	printf( "   image: " );
	imagedisplay_rect_print( &id->conv->image );
	printf( "\n" );

	printf( "   canvas: " );
	imagedisplay_rect_print( &id->conv->canvas );
	printf( "\n" );

}
#endif /*DEBUG_GEO*/

/* Trigger virtual methods.
 */

void
imagedisplay_set_conversion( Imagedisplay *id, Conversion *conv )
{
	ImagedisplayClass *klass = IMAGEDISPLAY_CLASS( 
		GTK_OBJECT( id )->klass );

	if( klass->set_conversion )
		klass->set_conversion( id, conv );
}

static void
imagedisplay_conversion_changed( Imagedisplay *id )
{
	ImagedisplayClass *klass = IMAGEDISPLAY_CLASS( 
		GTK_OBJECT( id )->klass );

	if( klass->conversion_changed )
		klass->conversion_changed( id );
}

static void
imagedisplay_area_changed( Imagedisplay *id, Rect *dirty )
{
	ImagedisplayClass *klass = IMAGEDISPLAY_CLASS( 
		GTK_OBJECT( id )->klass );

	if( klass->area_changed )
		klass->area_changed( id, dirty );
}

/* Imageinfo area has changed signal.
 */
static void
imagedisplay_ii_area_changed_cb( Imageinfo *imageinfo, 
	Rect *dirty, Imagedisplay *id )
{
	Rect display;

	assert( IS_IMAGEINFO( imageinfo ) );
	assert( IS_IMAGEDISPLAY( id ) );

	if( id->conv->ii == imageinfo ) {
		conversion_im_to_disp_rect( id->conv, dirty, &display );
		imagedisplay_area_changed( id, &display );
	}
}

/* Imageinfo has been destroyed signal.
 */
static void
imagedisplay_ii_destroy_cb( Imageinfo *imageinfo, Imagedisplay *id )
{
	assert( IS_IMAGEINFO( imageinfo ) );
	assert( IS_IMAGEDISPLAY( id ) );
	assert( id->ii_area_changed_sid );

	id->ii_area_changed_sid = 0;
	id->ii_destroy_sid = 0;
	id->ii = NULL;
}

/* We've changed imagedisplay geo completely ... remake the adjustments and
 * signal.
 */
static void
imagedisplay_adj_changed( Imagedisplay *id )
{
	Conversion *conv = id->conv;

	if( id->hadj ) {
		id->hadj->lower = 0.0;
		id->hadj->upper = (gfloat) conv->canvas.width;
		id->hadj->value = (gfloat) id->visible.left;
		id->hadj->step_increment = 10.0;
		id->hadj->page_increment = (gfloat) id->screen.width;
		id->hadj->page_size = (gfloat) id->screen.width;
		gtk_signal_emit_by_name( GTK_OBJECT( id->hadj ), "changed" );
	}

	if( id->vadj ) {
		id->vadj->lower = 0.0;
		id->vadj->upper = (gfloat) conv->canvas.height;
		id->vadj->value = (gfloat) id->visible.top;
		id->vadj->step_increment = 10.0;
		id->vadj->page_increment = (gfloat) id->screen.height;
		id->vadj->page_size = (gfloat) id->screen.height;
		gtk_signal_emit_by_name( GTK_OBJECT( id->vadj ), "changed" );
	}
}

/* The conversion has changed ... remake our stuff.
 */
static void 
imagedisplay_rethink_display( Imagedisplay *id )
{
        id->visible.width = IM_MIN( id->screen.width, 
		id->conv->canvas.width - id->visible.left );
        id->visible.height = IM_MIN( id->screen.height, 
		id->conv->canvas.height - id->visible.top );
	id->conv->visible = id->visible;

	imagedisplay_adj_changed( id );
	imagedisplay_repaint_all( id );
}

static void
imagedisplay_real_conversion_changed( Imagedisplay *id )
{
	assert( IS_IMAGEDISPLAY( id ) );

	if( id->conv->ii != id->ii ) {
		FREESID( id->ii_area_changed_sid, id->ii );
		FREESID( id->ii_destroy_sid, id->ii );

		id->ii = id->conv->ii;

		if( id->conv->ii ) {
			id->ii_area_changed_sid = 
				gtk_signal_connect( GTK_OBJECT( id->conv->ii ), 
				"area_changed", 
				GTK_SIGNAL_FUNC( imagedisplay_ii_area_changed_cb ), 
				id );
			id->ii_destroy_sid = 
				gtk_signal_connect( GTK_OBJECT( id->conv->ii ), 
				"destroy", 
				GTK_SIGNAL_FUNC( imagedisplay_ii_destroy_cb ), 
				id );
		}
	}

	imagedisplay_rethink_display( id );

#ifdef DEBUG_GEO
	printf( "imagedisplay_conv_changed: 0x%x geo is now:\n",
		(unsigned int) id );
	imagedisplay_geo_print( id );
#endif /*DEBUG_GEO*/
}

static void
imagedisplay_conversion_changed_cb( Model *model, Imagedisplay *id )
{
	imagedisplay_conversion_changed( id );
}

/* Install a new conversion.
 */
static void
imagedisplay_real_set_conversion( Imagedisplay *id, Conversion *conv )
{
	/* Unlink old conversion.
	 */
	if( id->conv ) {
		FREESID( id->changed_sid, id->conv );
		FREEFO( gtk_object_unref, id->conv );
	}

	/* Set new conversion.
	 */
	if( conv ) {
		id->conv = conv;
		id->changed_sid = gtk_signal_connect( GTK_OBJECT( id->conv ), 
			"changed", 
			GTK_SIGNAL_FUNC( imagedisplay_conversion_changed_cb ), 
			id );
		gtk_object_ref( GTK_OBJECT( conv ) );
		gtk_object_sink( GTK_OBJECT( conv ) );

		/* Trigger a change event on us so we update.
		 */
		imagedisplay_conversion_changed( id );
	}
}

/* Part of the imageinfo we display has changed.
 */
static void
imagedisplay_real_area_changed( Imagedisplay *id, Rect *dirty )
{
	imagedisplay_repaint( id, dirty );
}

static void
imagedisplay_destroy( GtkObject *object )
{
	Imagedisplay *id;

#ifdef DEBUG
	printf( "imagedisplay_destroy\n" );
#endif /*DEBUG*/

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_IMAGEDISPLAY( object ) );

	id = IMAGEDISPLAY( object );

	/* Remove from repaint list.
	 */
	imagedisplay_repaint_remove( id );

	FREEFO( gtk_object_unref, id->hadj );
	FREEFO( gtk_object_unref, id->vadj );
	FREEFO( gtk_object_unref, id->conv );
	FREESID( id->ii_area_changed_sid, id->ii );
	FREESID( id->ii_destroy_sid, id->ii );
	id->ii = NULL;

	FREEF( gdk_gc_unref, id->back_gc );
	FREEF( gdk_gc_unref, id->err_gc );
	FREEF( gdk_gc_unref, id->xor_gc );
	FREEF( gdk_gc_unref, id->top_gc );
	FREEF( gdk_gc_unref, id->bottom_gc );
	FREEF( gdk_gc_unref, id->scroll_gc );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

/* Move to a new position.
 */
static void
imagedisplay_scrollto( Imagedisplay *id, int x, int y )
{
	GtkWidget *widget = GTK_WIDGET( id );
	Conversion *conv = id->conv;

	Rect old, new, overlap, dirty;

	if( !GTK_WIDGET_REALIZED( id ) || !GTK_WIDGET_DRAWABLE( id ) ) {
		/* No painting possible? Just update state. Maybe should clip
		 * position against size?
		 */
		id->visible.left = x;
		id->visible.top = y;
		return;
	}

	/* Clip scroll to limits.
	 */ 
        x = IM_CLIP( 0, x, conv->canvas.width - id->screen.width );
        y = IM_CLIP( 0, y, conv->canvas.height - id->screen.height );

	/* Already there? 
	 */
	if( id->visible.left == x && id->visible.top == y )
		return;

#ifdef DEBUG
	printf( "imagedisplay_scrollto: x=%d, y=%d\n", x, y );
#endif /*DEBUG*/

	/* Old screen area ... id->visible is clipped against the canvas 
	 * size, we want to include any background that's visible too.
	 */
	old = id->visible;
	old.width = id->screen.width;
	old.height = id->screen.height;

	/* New screen area.
 	 */
	new = old;
        new.left = x;
        new.top = y;

        /* How much of the new position is already on the screen? Intersect
         * those two. If there's any, rop it to the new place.
         */
        im_rect_intersectrect( &old, &new, &overlap );
        if( !im_rect_isempty( &overlap ) ) {
                /* We can use some of the stuff already there. Scroll our
                 * drawable. 
                 */
		gdk_window_copy_area( widget->window, id->scroll_gc,
			overlap.left - new.left + id->screen.left, 
			overlap.top - new.top + id->screen.top,
			widget->window,
			overlap.left - old.left + id->screen.left,
			overlap.top - old.top + id->screen.top,
			overlap.width, overlap.height );

                /* Set new posn.
                 */
		im_rect_intersectrect( &new, &conv->canvas, &id->visible );
		conv->visible = id->visible;

		/* Wait for all graphics_expose events to come in before we
		 * allow repaints to start. This won't do any (much) painting:
		 * it just adds the exposes to the queue.
		 */
		process_events_graphics_expose( widget );

		/* We need a rect complement here .. which needs something like
                 * the rectlist stuff in SunView. Can't be bothered to write
                 * all that -- so we special case this. We know that the Rect
                 * we have just ropped into place is sitting on a corner of
                 * the window, so the complement of overlap intersected with
                 * vis can be formed from just two rects. We have to find
                 * them! One will be above or below overlap, one will be left
                 * or right of overlap.
                 */
                dirty.width = new.width - overlap.width;
                dirty.height = overlap.height;
                dirty.top = overlap.top;
                if( overlap.left == new.left )
                        /* Rect to right of overlap needs painting.
                         */
                        dirty.left = overlap.left + overlap.width;
                else
                        /* Rect to left of overlap needs painting.
                         */
                        dirty.left = new.left;
		imagedisplay_repaint( id, &dirty );

                /* Do stripe above or below overlap.
                 */
                dirty.width = new.width;
                dirty.height = new.height - overlap.height;
                dirty.left = new.left;
                if( overlap.top == new.top )
                        /* Rect to below overlap needs painting.
                         */
                        dirty.top = overlap.top + overlap.height;
                else
                        /* Rect to above overlap needs painting.
                         */
                        dirty.top = new.top;
		imagedisplay_repaint( id, &dirty );
	}
	else {
		/* Nothing .. do a complete repaint.
                 */
		im_rect_intersectrect( &new, &conv->canvas, &id->visible );
		conv->visible = id->visible;
                imagedisplay_repaint_all( id );
        }
}

/* We've changed position ... signal to our adjustments.
 */
static void
imagedisplay_adj_value_changed( Imagedisplay *id )
{
	id->hadj->value = (gfloat) id->visible.left;
	id->vadj->value = (gfloat) id->visible.top;

	gtk_signal_emit_by_name( GTK_OBJECT( id->hadj ), "value_changed" );
	gtk_signal_emit_by_name( GTK_OBJECT( id->vadj ), "value_changed" );
}

/* Scroll to a new position, and update the adjustments. This is the function
 * we export for mouse-drag etc.
 */
void
imagedisplay_set_position( Imagedisplay *id, int x, int y )
{
	imagedisplay_scrollto( id, x, y );
	imagedisplay_adj_value_changed( id );
}

void
imagedisplay_set_size( Imagedisplay *id, int w, int h )
{
	if( GTK_WIDGET_CAN_FOCUS( GTK_WIDGET( id ) ) ) 
		/* If we can focus, add some more space for the focus
		 * indicator.
		 */
		gtk_widget_set_usize( GTK_WIDGET( id ), 
			w + id->border * 2, h + id->border * 2 );
	else
		gtk_widget_set_usize( GTK_WIDGET( id ), w, h );

	/* Screen size will be updated when configure is triggered ... which
	 * isn't yet. Hack in ourselves.
	 */
	id->screen.width = w;
	id->screen.height = h;
}

/* Change the mag and position. Arrange for the passed position (in IMAGE 
 * coordinates) to be in the centre of the screen.
 */
void
imagedisplay_set_mag_position( Imagedisplay *id, int mag, int ix, int iy )
{
	Conversion *conv = id->conv;

	int x, y;
	int cw, ch;

	if( !conv->display_ii )
		return;

	/* Check for this-mag-will-cause-integer-overflow. Should flag an 
	 * error, but we just bail out instead.
	 */
	if( mag > 0 &&
		((double) conv->image.width * mag > (double) INT_MAX / 2 ||
		(double) conv->image.height * mag > (double) INT_MAX / 2) ) 
		return;

	/* Will this mag result in width/height of <1? If it will, pick a
	 * mag that most nearly gives us width/height 1.
	 */
	conv->mag = mag;
	conversion_im_to_disp( conv, 
		conv->image.width, conv->image.height, &x, &y );
	if( x <= 0  || y <= 0 ) {
		conv->mag = IM_MAX( -conv->image.width, -conv->image.height );
		if( conv->mag == -1 )
			conv->mag = 1;
	}

	/* What's the new canvas size going to be?
	 */
	cw = conv->image.width * conversion_dmag( conv->mag );
	ch = conv->image.height * conversion_dmag( conv->mag );

	/* Find the new window centre .. then offset and move.
	 */
	conversion_im_to_disp( conv, ix, iy, &x, &y );
	id->visible.left = x - id->visible.width / 2;
	id->visible.top = y - id->visible.height / 2;

	/* Position should be in range.
	 */
	id->visible.left = IM_CLIP( 0, id->visible.left, 
		IM_MAX( 0, cw - id->screen.width ) );
	id->visible.top = IM_CLIP( 0, id->visible.top, 
		IM_MAX( 0, ch - id->screen.height ) );

	/* We may have changed the visible area ... if visible is smaller than
	 * screen, and mag goes up, vis goes up. Rethink.
	 */
        id->visible.width = IM_MIN( id->screen.width, cw - id->visible.left );
        id->visible.height = IM_MIN( id->screen.height, ch - id->visible.top );

	/* Rebuild conversion.
	 */
	conv->visible = id->visible;
	conversion_refresh( conv );

#ifdef DEBUG_GEO
	printf( "imagedisplay_set_mag_position: 0x%x\n", (unsigned int) id );
	imagedisplay_geo_print( id );
#endif /*DEBUG_GEO*/
}

/* Change the mag factor. 
 */
void
imagedisplay_set_mag( Imagedisplay *id, int mag )
{
	Conversion *conv = id->conv;

	Rect *r;
	int x, y;

	if( !conv->display_ii )
		return;

	/* Mag 0 means scale image to fit window.
	 */
	if( mag == 0 ) {
		float xfac = (float) id->screen.width / conv->image.width;
		float yfac = (float) id->screen.height / conv->image.height;
		float fac = IM_MIN( xfac, yfac );

		if( fac >= 1 )
			mag = (int) fac;
		else
			/* 0.999 means we don't round up on an exact fit.

			 	FIXME ... yuk

			 */
			mag = -((int) (0.99999999 + 1.0/fac));

#ifdef DEBUG
		printf( "imagedisplay_set_mag: shrink to fit:\n" );
		printf( " screen %dx%d, image %dx%d\n",
			id->screen.width, id->screen.height,
			conv->image.width, conv->image.height );
		printf( " picked mag of %d\n", mag );
#endif /*DEBUG*/
	}

	/* No change? Do nothing.
	 */
	if( mag == conv->mag )
		return;

	/* New position: find the centre of the current window in IMAGE
	 * coordinates.
	 */
	r = &id->visible;
	conversion_disp_to_im( conv, 
		r->left + r->width / 2, r->top + r->height / 2, &x, &y );

	imagedisplay_set_mag_position( id, mag, x, y );
}

/* Callback ... someone else has modified one of our adjustments. Update our
 * scroll position.
 */
static void
imagedisplay_adj_value_changed_cb( GtkAdjustment *adj, gpointer data )
{
	Imagedisplay *id;

        g_return_if_fail( adj != NULL );
        g_return_if_fail( data != NULL );

        id = IMAGEDISPLAY( data );

	if( adj == id->hadj ) 
		imagedisplay_scrollto( id, (int)adj->value, id->visible.top );
	else
		imagedisplay_scrollto( id, id->visible.left, (int)adj->value );
}

/* Junk old adj, attach new adj.
 */
static void
imagedisplay_swap_adjs( Imagedisplay *id, 
	GtkAdjustment *oadj, GtkAdjustment *nadj )
{
	/* Junk old adjustment, if any.
	 */
        if( oadj ) {
		gtk_signal_disconnect_by_data( 
			GTK_OBJECT( oadj ), (gpointer) id );
		gtk_object_unref( GTK_OBJECT( oadj ) );
	}

	/* Attach new adj.
	 */
        gtk_object_ref( GTK_OBJECT( nadj ) );
        gtk_object_sink( GTK_OBJECT( nadj ) );

        gtk_signal_connect( GTK_OBJECT( nadj ), "value_changed",
		GTK_SIGNAL_FUNC( imagedisplay_adj_value_changed_cb ),
		(gpointer) id );
}

/* Install a new h adjustment.
 */
void
imagedisplay_set_hadj( Imagedisplay *id, GtkAdjustment *hadj )
{
	g_return_if_fail( id != NULL );
	g_return_if_fail( IS_IMAGEDISPLAY( id ) );

	/* Swap hadj.
	 */
	imagedisplay_swap_adjs( id, id->hadj, hadj );
        id->hadj = hadj;

#ifdef DEBUG
	printf( "set_hadj: lower=%g, upper=%g, val=%g\n",
		hadj->lower, hadj->upper, hadj->value ); 
#endif /*DEBUG*/
}

/* Install a new v adjustment.
 */
void
imagedisplay_set_vadj( Imagedisplay *id, GtkAdjustment *vadj )
{
	g_return_if_fail( id != NULL );
	g_return_if_fail( IS_IMAGEDISPLAY( id ) );

	/* Swap vadj.
	 */
	imagedisplay_swap_adjs( id, id->vadj, vadj );
        id->vadj = vadj;

#ifdef DEBUG
	printf( "set_vadj: lower=%g, upper=%g, val=%g\n",
		vadj->lower, vadj->upper, vadj->value ); 
#endif /*DEBUG*/
}

/* Expose signal handler.
 */
static gint
imagedisplay_expose( GtkWidget *widget, GdkEventExpose *event )
{
	Imagedisplay *id = IMAGEDISPLAY( widget );
	Rect expose, clipped;

	if( !GTK_WIDGET_REALIZED( id ) )
		return( FALSE );

#ifdef DEBUG
	printf( "imagedisplay_expose: at %dx%d size %dx%d\n",
		event->area.x, event->area.y,
		event->area.width, event->area.height );
#endif /*DEBUG*/

	gtk_widget_draw_focus( widget );

	/* Translate expose area to image coordinates, clip against section of
	 * DrawingArea we use for display.
	 */
	expose.left = event->area.x;
	expose.top = event->area.y;
	expose.width = event->area.width;
	expose.height = event->area.height;
	im_rect_intersectrect( &expose, &id->screen, &clipped );
	clipped.left += id->visible.left - id->screen.left;
	clipped.top += id->visible.top - id->screen.top;

	imagedisplay_repaint( id, &clipped );

        return( FALSE );
}

/* Resize signal.
 */
static gint
imagedisplay_configure( GtkWidget *widget, GdkEventConfigure *event )
{
	Imagedisplay *id = IMAGEDISPLAY( widget );
	Conversion *conv = id->conv;
	int w, h; 

	/* Note new size.
 	 */
	if( GTK_WIDGET_CAN_FOCUS( widget ) ) {
		id->screen.left = id->border;
		id->screen.top = id->border;
		id->screen.width = IM_MAX( 0, 
			widget->allocation.width - id->border * 2 );
		id->screen.height = IM_MAX( 0, 
			widget->allocation.height - id->border * 2 );
	}
	else {
		id->screen.left = 0;
		id->screen.top = 0;
		id->screen.width = widget->allocation.width;
		id->screen.height = widget->allocation.height;
	}
	w = id->screen.width;
	h = id->screen.height;

#ifdef DEBUG_GEO
	printf( "imagedisplay_configure: 0x%x new size is %dx%d\n", 
		(unsigned int) id, w, h );
#endif /*DEBUG_GEO*/

	/* Has the right edge of the window moved beyond the right 
	 * edge of the image? If yes, try to scroll the image down 
	 * & right to fill it up.
	 */
	imagedisplay_scrollto( id,
		IM_MIN( conv->canvas.width - w, id->visible.left ),
		IM_MIN( conv->canvas.height - h, id->visible.top ) );

	/* Rethink visible part.
	 */
	id->visible.width = IM_MIN( w, 
		conv->canvas.width - id->visible.left );
	id->visible.height = IM_MIN( h, 
		conv->canvas.height - id->visible.top );
	conv->visible = id->visible;

	/* Rethink the adjustments and send an update signal.
	 */
	imagedisplay_adj_changed( id ); 

#ifdef DEBUG_GEO
	printf( "imagedisplay_configure: new geo:\n" );
	imagedisplay_geo_print( id );
#endif /*DEBUG_GEO*/

        return( TRUE );
}

static void
imagedisplay_realize( GtkWidget *widget )
{
	if( GTK_WIDGET_CLASS( parent_class )->realize )
		(*GTK_WIDGET_CLASS( parent_class )->realize)( widget );

	/* Stops auto background clear during configure.
	 */
	gdk_window_set_back_pixmap( widget->window, NULL, FALSE );
}

static gint
imagedisplay_focus_in_event( GtkWidget *widget, GdkEventFocus *event )
{
	GTK_WIDGET_SET_FLAGS( widget, GTK_HAS_FOCUS );
	gtk_widget_draw_focus( widget );

	return( FALSE );
}

static gint
imagedisplay_focus_out_event( GtkWidget *widget, GdkEventFocus *event )
{
	GTK_WIDGET_UNSET_FLAGS( widget, GTK_HAS_FOCUS );
	gtk_widget_draw_focus( widget );

	return( FALSE );
}

static void
imagedisplay_draw_focus( GtkWidget *widget )
{
	if( GTK_WIDGET_CAN_FOCUS( widget ) && GTK_WIDGET_REALIZED( widget ) ) {
		Imagedisplay *id = IMAGEDISPLAY( widget );

		/* Clear focus indicator background.
		 */
		gdk_draw_rectangle( widget->window, 
			widget->style->bg_gc[GTK_STATE_NORMAL], TRUE,
			0, 
			0, 
			id->screen.width + id->border * 2,
			id->screen.top );
		gdk_draw_rectangle( widget->window, 
			widget->style->bg_gc[GTK_STATE_NORMAL], TRUE,
			0, 
			id->screen.height + id->border, 
			id->screen.width + id->border * 2,
			id->screen.top );
		gdk_draw_rectangle( widget->window, 
			widget->style->bg_gc[GTK_STATE_NORMAL], TRUE,
			0, 
			id->border, 
			id->border,
			id->screen.height );
		gdk_draw_rectangle( widget->window, 
			widget->style->bg_gc[GTK_STATE_NORMAL], TRUE,
			id->screen.width + id->border, 
			id->border, 
			id->border,
			id->screen.height );

		if( GTK_WIDGET_HAS_FOCUS( widget ) )
			gtk_paint_focus( widget->style, widget->window,
				NULL, widget, "button",
				0, 0, 
				id->screen.width + id->border * 2 - 1, 
				id->screen.height + id->border * 2 - 1 );
	}
}

static gint
imagedisplay_button_press_event( GtkWidget *widget, GdkEventButton *event )
{
#ifdef EVENT
	printf( "imagedisplay_button_press_event\n" );
#endif /*EVENT*/

	if( event->type == GDK_BUTTON_PRESS ) {
		if( !GTK_WIDGET_HAS_FOCUS( widget ) )
			gtk_widget_grab_focus( widget );

#ifdef GRAB
		if( event->button == 1 )
			gtk_grab_add( widget );
#endif /*GRAB*/
	}

	return( TRUE );
}

static gint
imagedisplay_button_release_event( GtkWidget *widget, GdkEventButton *event )
{
#ifdef GRAB
	if( event->button == 1 )
		gtk_grab_remove( widget );
#endif /*GRAB*/

	return( TRUE );
}

/* Init Imagedisplay class.
 */
static void
imagedisplay_class_init( ImagedisplayClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass*) klass;
        GtkWidgetClass *widget_class = (GtkWidgetClass*) klass;

	/* Init parent class.
	 */
        parent_class = gtk_type_class( GTK_TYPE_DRAWING_AREA );

	/* Make our signals.
	 */
        imagedisplay_signals[EXPOSE_START] = 
		gtk_signal_new( "expose_start",
			GTK_RUN_FIRST,
			object_class->type,
			GTK_SIGNAL_OFFSET( ImagedisplayClass, expose_start ),
			gtk_marshal_NONE__POINTER, 
			GTK_TYPE_NONE, 
			1,
			GTK_TYPE_POINTER );
        imagedisplay_signals[REPAINT] = 
		gtk_signal_new( "repaint",
			GTK_RUN_FIRST,
			object_class->type,
			GTK_SIGNAL_OFFSET( ImagedisplayClass, repaint ),
			gtk_marshal_NONE__POINTER, 
			GTK_TYPE_NONE, 
			1,
			GTK_TYPE_POINTER );

        gtk_object_class_add_signals( object_class, 
		imagedisplay_signals, LAST_SIGNAL );

	/* No default signal action.
	 */
        klass->expose_start = NULL;
        klass->repaint = NULL;

	/* Virtual methods.
	 */
	klass->conversion_changed = imagedisplay_real_conversion_changed;
	klass->set_conversion = imagedisplay_real_set_conversion;
	klass->area_changed = imagedisplay_real_area_changed;

	/* Need special destroy stuff.
	 */
        object_class->destroy = imagedisplay_destroy;

	/* Catch expose stuff.
	 */
	widget_class->expose_event = imagedisplay_expose;
	widget_class->configure_event = imagedisplay_configure;
	widget_class->realize = imagedisplay_realize;

	/* Focus stuff.
	 */
	widget_class->focus_in_event = imagedisplay_focus_in_event;
	widget_class->focus_out_event = imagedisplay_focus_out_event;
	widget_class->draw_focus = imagedisplay_draw_focus;
	widget_class->button_press_event = imagedisplay_button_press_event;
	widget_class->button_release_event = imagedisplay_button_release_event;
}

static void
imagedisplay_make_gcs( Imagedisplay *id )
{
	/* Background stipple.
	 */
	static guchar stipple[8] = {
		0xF0,    /*  ####----  */
		0xE1,    /*  ###----#  */
		0xC3,    /*  ##----##  */
		0x87,    /*  #----###  */
		0x0F,    /*  ----####  */
		0x1E,    /*  ---####-  */
		0x3C,    /*  --####--  */
		0x78,    /*  -####---  */
	};

	GdkGCValues values;
	GdkColor fg, bg;
	GdkColor err_fg, err_bg;

	if( id->scroll_gc )
		return;

	id->scroll_gc = gdk_gc_new( main_window_gdk );
	gdk_gc_set_exposures( id->scroll_gc, TRUE );

	/* Ask for 2 greys to stipple background with.
	 */
	fg.red = fg.green = fg.blue = 0x90 << 8;
	bg.red = bg.green = bg.blue = 0xA0 << 8;
	(void) gdk_colormap_alloc_color( 
		gdk_window_get_colormap( main_window_gdk ), &fg, FALSE, FALSE );
	(void) gdk_colormap_alloc_color( 
		gdk_window_get_colormap( main_window_gdk ), &bg, FALSE, FALSE );

	/* And two reds for an error stipple.
	 */
	err_fg.red = 0xD0 << 8;
	err_fg.green = err_fg.blue = 0x90 << 8;
	err_bg.red = 0xE0 << 8;
	err_bg.green = err_bg.blue = 0xA0 << 8;
	(void) gdk_colormap_alloc_color( 
		gdk_window_get_colormap( main_window_gdk ), 
		&err_fg, FALSE, FALSE );
	(void) gdk_colormap_alloc_color( 
		gdk_window_get_colormap( main_window_gdk ), 
		&err_bg, FALSE, FALSE );

#ifdef HAVE_WINDOWS_H
	/* gtkwin has problems stippling. Just ask for a plain background.
	 */
	values.foreground.pixel = fg.pixel;
	values.background.pixel = bg.pixel;
	values.fill = GDK_SOLID;
	id->back_gc = gdk_gc_new_with_values( main_window_gdk, &values,
		GDK_GC_FOREGROUND | GDK_GC_BACKGROUND | GDK_GC_FILL );
	values.foreground.pixel = err_fg.pixel;
	values.background.pixel = err_bg.pixel;
	values.fill = GDK_SOLID;
	id->err_gc = gdk_gc_new_with_values( main_window_gdk, &values,
		GDK_GC_FOREGROUND | GDK_GC_BACKGROUND | GDK_GC_FILL );
#else /*HAVE_WINDOWS_H*/
	values.foreground.pixel = fg.pixel;
	values.background.pixel = bg.pixel;
	values.fill = GDK_OPAQUE_STIPPLED;
	values.stipple = gdk_bitmap_create_from_data( main_window_gdk,
		(char *) stipple, 8, 8 );
	id->back_gc = gdk_gc_new_with_values( main_window_gdk, &values,
		GDK_GC_FOREGROUND | GDK_GC_BACKGROUND | GDK_GC_FILL |
		GDK_GC_STIPPLE );
	values.foreground.pixel = err_fg.pixel;
	values.background.pixel = err_bg.pixel;
	values.fill = GDK_OPAQUE_STIPPLED;
	id->err_gc = gdk_gc_new_with_values( main_window_gdk, &values,
		GDK_GC_FOREGROUND | GDK_GC_BACKGROUND | GDK_GC_FILL |
		GDK_GC_STIPPLE );
#endif /*HAVE_WINDOWS_H*/

	values.foreground = main_window_top->style->white;
	values.function = GDK_XOR;
	id->xor_gc = gdk_gc_new_with_values( main_window_gdk, &values,
		GDK_GC_FOREGROUND | GDK_GC_FUNCTION );
	gdk_gc_set_line_attributes( id->xor_gc, 
		0, GDK_LINE_ON_OFF_DASH, GDK_CAP_BUTT, GDK_JOIN_MITER );

	id->top_gc = gdk_gc_new( main_window_gdk );
	id->bottom_gc = gdk_gc_new( main_window_gdk );
}

static void
imagedisplay_init( Imagedisplay *id )
{
	static const Rect emptyrect = { 0, 0, 0, 0 };

	id->screen = emptyrect;
	id->visible = emptyrect;
	id->border = imagedisplay_focus_border;

	id->hadj = NULL;
	id->vadj = NULL;

	id->conv = NULL;
	id->changed_sid = 0;
	id->ii_area_changed_sid = 0;
	id->ii_destroy_sid = 0;
	id->ii = NULL;

        id->sheight = 10;
        id->pending = NULL;
        id->update = FALSE;

	id->back_gc = NULL;
	id->err_gc = NULL;
	id->xor_gc = NULL;
	id->top_gc = NULL;
	id->bottom_gc = NULL;
	id->scroll_gc = NULL;

	imagedisplay_make_gcs( id );

	GTK_WIDGET_SET_FLAGS( GTK_WIDGET( id ), GTK_CAN_FOCUS );
}

/* Make class id.
 */
guint
imagedisplay_get_type( void )
{
	static guint id_type = 0;

	if( !id_type ) {
		GtkTypeInfo info = {
			"Imagedisplay",
			sizeof( Imagedisplay ),
			sizeof( ImagedisplayClass ),
			(GtkClassInitFunc) imagedisplay_class_init,
			(GtkObjectInitFunc) imagedisplay_init,
			/* reserved1 */ NULL,
			/* reserved2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		/* Create our type.
		 */
		id_type = gtk_type_unique( GTK_TYPE_DRAWING_AREA, &info );
	}

	return( id_type );
}

static void
imagedisplay_link( Imagedisplay *id, 
	Conversion *conv, GtkAdjustment *hadj, GtkAdjustment *vadj )
{
	gtk_widget_set_events( GTK_WIDGET( id ), GDK_EXPOSURE_MASK );

	imagedisplay_set_conversion( id, conv );

	if( !hadj )
		hadj = GTK_ADJUSTMENT( gtk_adjustment_new( 
			0.0, 0.0, 0.0, 0.0, 0.0, 0.0 ) );
	imagedisplay_set_hadj( id, hadj );

	if( !vadj )
		vadj = GTK_ADJUSTMENT( gtk_adjustment_new( 
			0.0, 0.0, 0.0, 0.0, 0.0, 0.0 ) );
	imagedisplay_set_vadj( id, vadj );
}

/* Make a new Imagedisplay. Pass in the conversion we should show, and the h 
 * and v adjustments for scrolling. If adjs are NULL, make our own. conv can
 * be NULL ... wait for one to be installed.
 */
Imagedisplay *
imagedisplay_new( Conversion *conv, GtkAdjustment *hadj, GtkAdjustment *vadj )
{
	Imagedisplay *id = gtk_type_new( TYPE_IMAGEDISPLAY );

	imagedisplay_link( id, conv, hadj, vadj );

	return( id );
}

GtkAdjustment *
imagedisplay_get_hadj( Imagedisplay *id )
{
	g_return_val_if_fail( id != NULL, NULL );
	g_return_val_if_fail( IS_IMAGEDISPLAY( id ), NULL );

	return( id->hadj );
}

GtkAdjustment *
imagedisplay_get_vadj( Imagedisplay *id )
{
	g_return_val_if_fail( id != NULL, NULL );
	g_return_val_if_fail( IS_IMAGEDISPLAY( id ), NULL );

	return( id->vadj );
}

/* Emit our signals ... called from asynch.c, private really.
 */
void
imagedisplay_emit_expose_start( Imagedisplay *id, Rect *expose )
{
#ifdef DEBUG
	printf( "imagedisplay_emit_expose_start: " );
	imagedisplay_rect_print( expose );
	printf( "\n" );
#endif /*DEBUG*/

	gtk_signal_emit( GTK_OBJECT( id ), 
		imagedisplay_signals[EXPOSE_START], (gpointer) expose );
}

void 
imagedisplay_emit_repaint( Imagedisplay *id, Rect *expose )
{
#ifdef DEBUG
	printf( "imagedisplay_emit_repaint: " );
	imagedisplay_rect_print( expose );
	printf( "\n" );
#endif /*DEBUG*/

	gtk_signal_emit( GTK_OBJECT( id ), 
		imagedisplay_signals[REPAINT], (gpointer) expose );
}

/* Map X event positions to virtual canvas cods and back.
 */
void 
imagedisplay_xev_to_disp( Imagedisplay *id, int ex, int ey, int *dx, int *dy )
{
	*dx = ex + id->visible.left - id->screen.left;
	*dy = ey + id->visible.top - id->screen.top;
}

void 
imagedisplay_disp_to_xev( Imagedisplay *id, int dx, int dy, int *ex, int *ey )
{
	*ex = dx - id->visible.left + id->screen.left;
	*ey = dy - id->visible.top + id->screen.top;
}

/* Same for rects.
 */
void
imagedisplay_xev_to_disp_rect( Imagedisplay *id, Rect *er, Rect *dr )
{
	int brx, bry;

	imagedisplay_xev_to_disp( id, er->left, er->top, &dr->left, &dr->top );
	imagedisplay_xev_to_disp( id, 
		IM_RECT_RIGHT( er ), IM_RECT_BOTTOM( er ), &brx, &bry );
	dr->width = brx - dr->left;
	dr->height = bry - dr->top;
}

void
imagedisplay_disp_to_xev_rect( Imagedisplay *id, Rect *dr, Rect *er )
{
	int brx, bry;

	imagedisplay_disp_to_xev( id, dr->left, dr->top, &er->left, &er->top );
	imagedisplay_disp_to_xev( id, 
		IM_RECT_RIGHT( dr ), IM_RECT_BOTTOM( dr ), &brx, &bry );
	er->width = brx - er->left;
	er->height = bry - er->top;
}
