/* main() ... start everything up. See mainw.c for main window stuff.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#include "ip.h"

/* 
#define DEBUG
 */

/* stop on any gtk error/warning/whatever
#define DEBUG_FATAL
 */

/* Time startup.
#define DEBUG_TIME
 */

/* General stuff. 
 */
GdkWindow *main_window_gdk;	/* Top GdkWindow */
GtkWidget *main_window_top = NULL;/* Secret top window */

Workspacegroup *main_workspacegroup = NULL;	/* All the workspaces */
Toolkitgroup *main_toolkitgroup = NULL;		/* All the toolkits */

void *main_c_stack_base = NULL;	/* Base of C stack */

/* Accumulate startup errors here.
 */
char main_start_error_txt[MAX_STRSIZE];
BufInfo main_start_error;

/* Action on main_quit()
 */
typedef enum {
	EXIT_ACTION_NONE,	/* Just quit */
	EXIT_ACTION_PRINT_LAST,	/* Print last thing we loaded in workspace */
	EXIT_ACTION_PRINT_MAIN	/* Print symbol "main" */
} ExitAction;

static ExitAction action_on_exit = EXIT_ACTION_NONE;

static const char *main_argv0;	/* Name invoked as */
static iOpenFile *main_stdin;	/* stdin as an iOpenFile */

/* Sub-fn of below.
 */
static void *
main_error_print( Expr *expr )
{
	assert( expr->err );

	printf( "error in \"%s\": %s\n", 
		MODEL( expr->sym )->name, expr->errstr );

	return( NULL );
}

/* Clean up our application and quit.
 */
void
main_quit( void )
{
	Symbol *sym;
	Workspace *ws;
	iOpenFile *of;

#ifdef DEBUG
	printf( "main_quit: cleaning up ...\n" );
#endif/*DEBUG*/

	/* Any exit actions?
	 */
	switch( action_on_exit ) {
	case EXIT_ACTION_NONE:	
		break;

	case EXIT_ACTION_PRINT_MAIN:
		sym = stable_find( symbol_root->expr->compile->locals, "main" );

		if( !sym )
			error( "symbol \"main\" not found" );
		if( symbol_recalculate( sym ) ) {
			slist_map( error_list, 
				(SListMapFn) main_error_print, NULL );
			error( "symbol \"main\" has no value" );
		}

		(void) set_output( &sym->expr->root );
		(void) graph_value( &sym->expr->root );
		break;

	case EXIT_ACTION_PRINT_LAST:
		ws = main_workspacegroup->current;

		if( !ws || !ws->last_symbol ||
			ws->last_symbol->type != SYM_VALUE ) 
			break;

		(void) set_output( &SYMBOL( ws->last_symbol )->expr->root );
		(void) graph_value( &SYMBOL( ws->last_symbol )->expr->root );

		break;

	default:
		assert( FALSE );
	}

	/* Force all windows down. 
	
		FIXME ... this can cause a recursive call to us from
		mainw_popdown_cb() ... hmmm

	 */
	iwindow_map_all( (iWindowMapFn) iwindow_kill, NULL );

	/* Junk all symbols.
	 */
	gtk_object_destroy( GTK_OBJECT( symbol_root_root ) );
	main_workspacegroup = NULL;
	main_toolkitgroup = NULL;

	/* Remove any ws retain files.
	 */
	workspace_retain_clean();

	/* Junk reduction machine.
	 */
	reduce_destroy( reduce_context );

	/* Should have freed everything now.
	 */

	/* Make sure!

		FIXME ... #ifdef this lot out at some point

	 */
	imageinfo_check_all_destroyed();
	heap_check_all_destroyed();
	im__print_all();
	util_check_all_destroyed();

	/* Dump wisdom back again.
	 */
#ifdef HAVE_FFTW
	if( (of = file_open_write( IP_SAVEDIR IM_DIR_SEP_STR "wisdom" )) ) {
		fftw_export_wisdom_to_file( of->fp );
		file_close( of );
	}
#endif /*HAVE_FFTW*/

#ifdef DEBUG
	printf( "main_quit: exit( 0 )\n" );
#endif/*DEBUG*/

	/* And exit.
	 */
	exit( 0 );
}

/* Try to load a thing, anything at all.
 */
gboolean
main_load( const char *filename )
{
	Workspace *ws;
	BufInfo buf;
	char txt[NAMELEN];

	if( call_string( (call_string_fn) im_load_plugin,
		filename, NULL, NULL, NULL ) ) 
		return( TRUE );
	if( toolkit_new_from_file( main_toolkitgroup, filename ) )
		return( TRUE );
	if( (ws = workspace_new_from_file( main_workspacegroup, filename )) ) {
		workspacegroup_set_current( main_workspacegroup, ws );
		return( TRUE );
	}

	/* Try as matrix or image. Have to do these via definitions.
	 */
	buf_init_static( &buf, txt, NAMELEN );
	buf_appends( &buf, "Matrix_file \"" );
	buf_appendsc( &buf, filename );
	buf_appends( &buf, "\"" );
	if( mainw_def_new( buf_all( &buf ) ) ) 
		return( TRUE );

	buf_init_static( &buf, txt, NAMELEN );
	buf_appends( &buf, "Image_file \"" );
	buf_appendsc( &buf, filename );
	buf_appends( &buf, "\"" );
	if( mainw_def_new( buf_all( &buf ) ) ) 
		return( TRUE );

	im_clear_error_string();
	ierrors( "unable to load \"%s\"", filename );

	return( FALSE );
}

static void *
main_load_plug( char *name )
{
	if( !call_string( (call_string_fn) im_load_plugin,
		name, NULL, NULL, NULL ) ) {
		verrors( "error loading plug-in \"%s\"", name );
		box_alert( NULL );
	}

	return( NULL );
}

static void *
main_load_def( const char *filename )
{	
	Toolkit *kit;

	if( !(kit = toolkit_new_from_file( main_toolkitgroup, filename )) )
		box_alert( NULL );
	else 
		filemodel_set_auto_load( FILEMODEL( kit ) );

	return( NULL );
}

static void *
main_load_ws( const char *name )
{
	Workspace *ws;

#ifdef DEBUG
	printf( "main_load_ws: %s\n", name );
#endif/*DEBUG*/

	if( !(ws = workspace_new_from_file( main_workspacegroup, name )) ) 
		box_alert( NULL );
	else {
		filemodel_set_auto_load( FILEMODEL( ws ) );
		MODEL( ws )->pos = -1;
		model_pos_sort( MODEL( main_workspacegroup ) );
		model_pos_renumber( MODEL( main_workspacegroup ) );
	}

	return( NULL );
}

/* Link all the packages in a function.
 */
static void *
main_link_package( im_package *pack)
{
	char name[NAMELEN];
	Toolkit *kit;
        int i;

	im_snprintf( name, NAMELEN, "_%s", pack->name );
	kit = toolkit_new( main_toolkitgroup, name );

        for( i = 0; i < pack->nfuncs; i++ ) 
		if( vips_is_callable( pack->table[i] ) ) {
			Symbol *sym;

			sym = symbol_new( symbol_root->expr->compile->locals,
				pack->table[i]->name );
			assert( sym->type == SYM_ZOMBIE );
			sym->type = SYM_EXTERNAL;
			sym->function = pack->table[i];
			sym->fn_nargs = vips_n_args( pack->table[i] );
			(void) tool_new_sym( kit, -1, sym );
			symbol_made( sym );
		}

	filemodel_set_auto_load( FILEMODEL( kit ) );
	filemodel_set_modified( FILEMODEL( kit ), FALSE );
	kit->pseudo = TRUE;

        return( NULL );
}

/* Load all plugins and defs.
 */
static void
main_load_all( void )
{
	const char *prefix;

	/* Add builtin toolkit.
	 */
	builtin_init();

	/* Load up any standard plugs.
	 */
	if( (prefix = im_guess_prefix( main_argv0, "VIPSHOME" )) &&
		im_load_plugins( "%s/lib", prefix ) ) {
		verrors( "Error loading standard plug-ins" );
		box_alert( NULL );
	}

	/* Load any plug-ins on PATH_START. 
	 */
#ifdef DEBUG
	printf( "plug-ins init\n" );
#endif/*DEBUG*/
	(void) path_map_exact( PATH_START, "*.plg", 
		(path_map_fn) main_load_plug, NULL );

	/* Link all VIPS functions as SYM_EXTERNAL.
	 */
        (void) im_map_packages( (im_list_map_fn) main_link_package, NULL );

	/* Load up all defs and wses.
	 */
#ifdef DEBUG
	printf( "definitions init\n" );
#endif/*DEBUG*/
	(void) path_map_exact( PATH_START, "*.def", 
		(path_map_fn) main_load_def, NULL );

#ifdef DEBUG
	printf( "ws init\n" );
#endif/*DEBUG*/
	(void) path_map_exact( PATH_START, "*.ws", 
		(path_map_fn) main_load_ws, NULL );

	symbol_recalculate_all();
}

static void *
main_junk_auto_load( Filemodel *filemodel )
{
	assert( IS_FILEMODEL( filemodel ) );

	if( filemodel->auto_load )
		FREEO( filemodel );

	return( NULL );
}

/* Remove and reload all menus/plugins/workspaces.
 */
void
main_reload( void )
{
	set_hourglass();

	/* Remove.
	 */
	toolkit_map( (toolkit_map_fn) main_junk_auto_load, NULL, NULL );
	workspace_map( (workspace_map_fn) main_junk_auto_load, NULL, NULL );
	im_close_plugins();

	/* Reload.
	 */
	main_load_all();

	set_pointer();
}


/* Init the display connection stuff.
 */
static void
main_x_init( int *argc, char ***argv )
{
	if( main_window_gdk )
		return;

#ifdef DEBUG
	printf( "X11 init\n" );
#endif/*DEBUG*/

	/* Start up gdk and gdkrgb. 
	 */
	gtk_set_locale();
	setlocale( LC_NUMERIC, "C" );
	(void) call_string( (call_string_fn) gtk_rc_add_default_file, 
		"$VIPSHOME" IM_DIR_SEP_STR "share" IM_DIR_SEP_STR 
		PACKAGE IM_DIR_SEP_STR "rc" IM_DIR_SEP_STR 
		"ipgtkrc", NULL, NULL, NULL );
	gtk_init( argc, argv );
#ifdef DEBUG
	gdk_rgb_set_verbose( TRUE );
#endif /*DEBUG*/
	gdk_rgb_set_min_colors( 64 );
	gdk_rgb_init();

        /* Make invisible top level window ... used to get stuff for
         * build.
         */
        main_window_top = gtk_window_new( GTK_WINDOW_TOPLEVEL );
        gtk_widget_realize( main_window_top );
	main_window_gdk = main_window_top->window;
}

/* Start here!
 */
int
main( int argc, char *argv[] )
{
	Workspace *ws;
	int i;
	gboolean interactive = TRUE;
#ifdef HAVE_GETRLIMIT
	struct rlimit rlp;
#endif /*HAVE_GETRLIMIT*/
	const char *prefix;
	char name[NAMELEN];
	iOpenFile *of;

#ifdef DEBUG_TIME
	static GTimer *startup_timer = NULL;

	if( !startup_timer )
		startup_timer = g_timer_new();

	g_timer_reset( startup_timer );

	printf( "%s: startup timer zeroed ...\n", argv[0] );
#endif /*DEBUG_TIME*/

	/* Want numeric locale to be "C", so we have C rules for doing 
	 * double <-> string (ie. no "," for decimal point).
	 */
	setlocale( LC_NUMERIC, "C" );

#ifdef DEBUG
	printf( "main: sizeof( struct heap_node ) == %d\n",
		sizeof( struct heap_node ) );

	/* Should be 3 pointers, hopefully.
	 */
	if( sizeof( struct heap_node ) != 3 * sizeof( void * ) )
		printf( "*** struct packing problem!\n" );
#endif/*DEBUG*/

        buf_init_static( &main_start_error, main_start_error_txt, MAX_STRSIZE );

	main_argv0 = argv[0];
	main_c_stack_base = &argc;

	/* Pass config.h stuff down to .ws files.
	 */
	putenvf( "PACKAGE=%s", PACKAGE );
	putenvf( "VERSION=%s", VERSION );

#ifdef HAVE_WINDOWS_H
        /* No HOME on windows ... make one from HOMEDRIVE and HOMEDIR (via
         * glib).
         */
        if( !getenv( "HOME" ) )
                putenvf( "HOME=%s", g_get_home_dir() );
#endif /*HAVE_WINDOWS_H*/

	/* Try to discover our data files.
	 */
	if( !(prefix = im_guess_prefix( main_argv0, "VIPSHOME" )) ) 
		error_exit( "unable to guess VIPSHOME" );

#ifdef DEBUG_FATAL
	/* Set masks for debugging ... stop on any problem. 
	 */
	g_log_set_always_fatal( 
		G_LOG_FLAG_RECURSION |
		G_LOG_FLAG_FATAL |
		G_LOG_LEVEL_ERROR |
		G_LOG_LEVEL_CRITICAL |
		G_LOG_LEVEL_WARNING );

	printf( "*** DEBUG_FATAL is on ... will abort() on first warning\n" );
#endif /*DEBUG_FATAL*/

	main_stdin = file_open_read_stdin();

#ifdef HAVE_GETRLIMIT
	/* Make sure we have lots of file descriptors.
	 */
	if( getrlimit( RLIMIT_NOFILE, &rlp ) == 0 ) {
		rlp.rlim_cur = rlp.rlim_max;
		if( setrlimit( RLIMIT_NOFILE, &rlp ) == 0 ) {
#ifdef DEBUG
			printf( "set max file descriptors to %d\n", 
				(int) rlp.rlim_max );
#endif /*DEBUG*/
		}
		else {
			g_warning( "unable to change max file descriptors\n"
				"max file descriptors still set to %d",
				(int) rlp.rlim_cur );
		}
	}
	else {
		g_warning( "unable to read max file descriptors" );
	}
#endif /*HAVE_GETRLIMIT*/

	/* Set default values for paths.
	 */
	path_init();

	/* Do we have a save dir? make one if not.
	 */
	if( !existsf( "%s", IP_SAVEDIR ) ) {
		char save_dir[FILENAME_MAX];

		if( !mkdirf( IP_SAVEDIR ) || 
			!mkdirf( IP_SAVEDIR IM_DIR_SEP_STR "tmp" ) || 
			!mkdirf( IP_SAVEDIR IM_DIR_SEP_STR "start" ) || 
			!mkdirf( IP_SAVEDIR IM_DIR_SEP_STR "data" ) ) 
			error_exit( "unable to make " IP_SAVEDIR " area: %s",
				g_strerror( errno ) );

		expand_variables( IP_SAVEDIR, save_dir );
		buf_appendf( &main_start_error, 
"Welcome to " PACKAGE "-" VERSION "!\n"
"\n"
"An area has been created in your home directory to hold startup,\n"
"data and temporary files:\n"
"\n"
"     %s\n"
"\n"
"If you've used previous versions of " PACKAGE ", you will probably want\n"
"to move any files over from your old work area and remove any old temps.",
			save_dir );
	}

	/* Are we going to be running interactively? Start up X if we are.

	 	FIXME ... want to start glib/gtk but not open an X connection
		:-( have to wait for gtk2

	 */

	/*
	if( strcmp( argv[argc-1], "-niscript" ) != 0 &&
		strcmp( argv[argc-1], "-benchmark" ) != 0 &&
		strcmp( argv[argc-1], "-main" ) != 0 )
	 */
		main_x_init( &argc, &argv );

	/* Init other stuff.
	 */
#ifdef HAVE_FFTW
	if( (of = file_open_read( IP_SAVEDIR IM_DIR_SEP_STR "wisdom" )) ) {
		fftw_import_wisdom_from_file( of->fp );
		file_close( of );
	}
#endif /*HAVE_FFTW*/
	reduce_context = reduce_new();
	symbol_root_init();
	model_base_init();
	view_base_init();
	main_workspacegroup = workspacegroup_new( "Workspaces" );
	main_toolkitgroup = toolkitgroup_new();
	main_load_all();

#ifdef DEBUG
	printf( "arg processing\n" );
#endif/*DEBUG*/

	/* Make start workspace.
	 */
	strcpy( name, "untitled" );
	while( stable_find( 
		SYMBOL( main_workspacegroup )->expr->compile->locals, name ) )
		increment_name( name );
	ws = workspace_new( main_workspacegroup, name );
	model_set( MODEL( ws ), NULL, "default workspace" );
	workspacegroup_set_current( main_workspacegroup, ws );

	/* We have to go right-to-left to avoid any X args that may be there.
	 */
	for( i = argc - 1; i > 0; i-- )
		/* Interactive script? 
		 */
		if( *argv[i] == '-' && 
			strcmp( argv[i] + 1, "script" ) == 0 ) {

			/* Behaviour.
			 */
			action_on_exit = EXIT_ACTION_PRINT_LAST;
			interactive = TRUE;

			/* Load stdin as if it were a workspace script.
			 */
			main_load_all();

			/*

				FIXME .. not done yet

			if( !workspace_load_file( main_workspacegroup->current,
				main_stdin ) )
				error( error_string );
			 */
			
			/* No more arg processing.
			 */
			break;
		}
		/* Non-interactive script? 
		 */
		else if( *argv[i] == '-' && 
			strcmp( argv[i] + 1, "niscript" ) == 0 ) {

			/* Behaviour.
			 */
			action_on_exit = EXIT_ACTION_PRINT_LAST;
			interactive = FALSE;

			/* Load stdin as if it were a workspace script.
			 */
			main_load_all();

			/*

				FIXME .. not done yet

			if( !workspace_load_file( main_workspacegroup->current,
				main_stdin ) )
				error( error_string );
			 */

			/* No more arg processing.
			 */
			break;
		}
		/* Defs file from stdin?
		 */
		else if( *argv[i] == '-' && 
			strcmp( argv[i] + 1, "main" ) == 0 ) {
			/* Note we are running as a non-interactive script.
			 */
			action_on_exit = EXIT_ACTION_PRINT_MAIN;
			interactive = FALSE;

			/* Load stdin as a load of defs.
			 */
			main_load_all();
			/*
			if( !toolkit_load_file( main_stdin ) )
				error( error_string );
			 */

			/* No more arg processing.
			 */
			break;
		}
		/* Benchmarking compiler?
		 */
		else if( *argv[i] == '-' && 
			strcmp( argv[i] + 1, "benchmark" ) == 0 ) {
			interactive = FALSE;
		}
		else if( *argv[i] == '-' ) {
			error( "unknown flag \"%s\"", argv[i] );
		}
		/* If no '-', load as file.
		 */
		else {
			if( !main_load( argv[i] ) ) 
				buf_appendf( &main_start_error, "%s\n",
					error_string );
		}

	/* Try a calculation. This should pop up all our display windows and
	 * run any args or scripts.
	 */
#ifdef DEBUG
	printf( "first recalc\n" );
#endif/*DEBUG*/
	symbol_recalculate_all();

#ifdef DEBUG_TIME
	printf( "%s: main init in %gs\n",  
		main_argv0,
		g_timer_elapsed( startup_timer, NULL ) );
#endif /*DEBUG_TIME*/

	/* Are we running interactively? Start the main window and loop.
	 */
	if( interactive ) {
		mainw_new( IP_NAME );
		gtk_widget_show( mainw_window );

		if( !buf_isempty( &main_start_error ) ) {
			ierrors( buf_all( &main_start_error ) );
			box_alert( NULL );
		}

#ifdef DEBUG
		printf( "starting event dispatch loop\n" );
#endif/*DEBUG*/

		gtk_main();
	}

	return( 0 );
}

#ifdef HAVE_WINDOWS_H 
/* Get non-cmd line args on win32.
 */
static int 
breakargs( char *program, char *line, char **argv ) 
{ 
	int argc = 1; 

	argv[0] = program; 

	while( *line && argc < MAX_SYSTEM - 1 ) { 
		while( *line && isspace( *line ) ) 
			line++; 

		if( *line == '"' ) {
			/* Windows-95 quoted arguments 
			 */ 
			char *start = line + 1; 
			char *end = start; 

			while( *end && *end != '"' ) 
				end++; 

			if( *end == '"' ) { 
				*end = '\0'; 
				argv[argc++] = start; 
				line = end + 1; 
				continue; 
			} 
		} 

		if( *line ) { 
			argv[argc++] = line; 
			while( *line && !isspace( *line ) ) 
				line++; 

			if( *line ) 
				*line++ = '\0'; 
		} 
	} 

	/* add trailing NULL pointer to argv 
	 */ 
	argv[argc] = NULL; 

	return( argc ); 
} 

int WINAPI 
WinMain( HINSTANCE hInstance, HINSTANCE hPrevInstance, 
	LPSTR lpszCmdLine, int nShowCmd ) 
{ 
	char *argv[MAX_SYSTEM];
	int argc;
	TCHAR program[MAXPATHLEN];

	GetModuleFileName( hInstance, program, sizeof(program) );
	argc = breakargs( (char *) program, lpszCmdLine, argv );

	return( main( argc, argv ) );
} 
#endif /*HAVE_WINDOWS_H*/ 
