/* an input option ... put/get methods
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#include "ip.h"

static ClassmodelClass *parent_class = NULL;

static void
option_destroy( GtkObject *object )
{
	Option *option;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_OPTION( object ) );

	option = OPTION( object );

	/* My instance destroy stuff.
	 */
	FREEF( slist_free_all, option->labels );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

/* Widgets for option edit.
 */
typedef struct _OptionEdit {
	iDialog *idlg;

	Option *option;

	GtkWidget *caption;
	GtkWidget *labels;		/* List for current options */
	GtkWidget *value;		/* Currently selected */

	GtkWidget *swin;		/* ScrolledWindow for list */
} OptionEdit;

/* Sub fn of below.
 */
static void
option_add_item( GtkWidget *item, GSList **labels )
{
	if( GTK_IS_LABEL( GTK_BIN( item )->child ) )
		*labels = g_slist_append( *labels, 
			g_strdup( orderitem_get_label( item ) ) );
}

/* Done button hit.
 */
/*ARGSUSED*/
static void
option_done_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	OptionEdit *eds = (OptionEdit *) client;
	Classmodel *classmodel = CLASSMODEL( eds->option );

	char *caption;
	GSList *labels;
	int value;

	caption = gtk_editable_get_chars( 
		GTK_EDITABLE( eds->caption ), 0, -1 );

	/* Extract the list of items.
	 */
	orderlist_scan( ORDERLIST( eds->labels ) );
	labels = NULL;
	gtk_container_foreach( GTK_CONTAINER( eds->labels ), 
		(GtkCallback) option_add_item, &labels );
	if( g_slist_length( labels ) == 0 ) {
		ierrors( "You need at least one option in your option list" );
		FREEF( slist_free_all, labels );
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	/* Current value.
	 */
	if( !get_geditable_int( eds->value, &value ) ) {
		FREEF( slist_free_all, labels );
		FREE( caption );
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	/* Update model.
	 */
	FREEF( slist_free_all, eds->option->labels );
	eds->option->labels = labels;
	eds->option->value = value;
	model_set( MODEL( eds->option ), NULL, caption );
	FREE( caption );

	/* Rebuild object.
	 */
	classmodel_update( classmodel );
	symbol_recalculate_all();

	nfn( sys, IWINDOW_TRUE );
}

/* Build the insides of option edit.
 */
static void
option_buildedit( iDialog *idlg, GtkWidget *work, OptionEdit *eds )
{
	Option *option = eds->option;
	GtkWidget *vb;
	GSList *p;
	int width, height;
	int n;

	eds->caption = build_glabeltext3( work, "Caption" );
	idialog_init_entry( idlg, eds->caption,
		"Set option caption here", "%s", MODEL( option )->caption );

	eds->swin = gtk_scrolled_window_new( NULL, NULL );
	gtk_container_set_border_width( GTK_CONTAINER( eds->swin ), 2 );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( eds->swin ),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	eds->labels = orderlist_new();
	gtk_list_set_selection_mode( GTK_LIST( eds->labels ),
		GTK_SELECTION_EXTENDED );
	gtk_scrolled_window_add_with_viewport( 
		GTK_SCROLLED_WINDOW( eds->swin ), eds->labels );
	gtk_container_set_focus_vadjustment( GTK_CONTAINER( eds->labels ),
		gtk_scrolled_window_get_vadjustment( 
			GTK_SCROLLED_WINDOW( eds->swin ) ) );
	gtk_container_set_focus_hadjustment( GTK_CONTAINER( eds->labels ),
		gtk_scrolled_window_get_hadjustment(
			GTK_SCROLLED_WINDOW( eds->swin ) ) );

	vb = build_glabelframe2( eds->swin, "Options" );
	gtk_box_pack_start( GTK_BOX( work ), vb, TRUE, TRUE, 0 );

	width = 100;
	height = 10;
	for( n = 0, p = option->labels; p; p = p->next, n++ ) {
		const char *text = (const char *) p->data;
		Rect geo;

		orderlist_add_label( ORDERLIST( eds->labels ), text, -1 );
		get_geo( eds->labels, text, &geo );

		if( geo.width > width ) 
			width = geo.width;
		if( geo.height > height ) 
			height = geo.height;

	}

	/* +40 to allow for width of vertical scrollbar. + 4 to allow for
	 * inter-line space. + 32 to allow for edit box and frame
	 * the edit box and the text frame on the edit box

		FIXME ... gtk2 does this much better 

	 */
	gtk_widget_set_usize( eds->swin, 
		IM_CLIP( 250, width + 40, 400 ), 
		IM_MIN( 200, (height + 4) * n + 32 ) );

        eds->value = build_glabeltext3( work, "Value" );
	idialog_init_entry( idlg, eds->value,
		"Set option default value here", "%d", option->value );

        gtk_widget_show_all( work );
}

/* Pop up a option edit box.
 */
static void 
option_edit( GtkWidget *parent, Model *model )
{
	Option *option = OPTION( model );
	OptionEdit *eds = IM_NEW( NULL, OptionEdit );
	GtkWidget *idlg;

	eds->option = option;

	idlg = idialog_new();
	iwindow_set_title( IWINDOW( idlg ), "Edit option" );
	idialog_set_build( IDIALOG( idlg ), 
		(iWindowBuildFn) option_buildedit, eds, NULL, NULL );
	idialog_set_callbacks( IDIALOG( idlg ), 
		iwindow_true_cb, NULL, NULL, idialog_free_client, eds );
	idialog_add_ok( IDIALOG( idlg ), option_done_cb, "Set option" );
	idialog_set_parent( IDIALOG( idlg ), parent );
	iwindow_build( IWINDOW( idlg ) );

	gtk_widget_show( GTK_WIDGET( idlg ) );
}

static xmlNode *
option_save( Model *model, xmlNode *xnode )
{
	Option *option = OPTION( model );

	xmlNode *xthis;

	if( !(xthis = MODEL_CLASS( parent_class )->save( model, xnode )) )
		return( NULL );

	if( CLASSMODEL( model )->edited ) {
		if( !set_slprop( xthis, "labels", option->labels ) ||
			!set_prop( xthis, "value", "%d", option->value ) )
			return( NULL );
	}

	return( xthis );
}

static gboolean
option_load( Model *model, 
	ModelLoadState *state, Model *parent, xmlNode *xnode )
{
	Option *option = OPTION( model );

	if( !IS_RHS( parent ) ) {
		ierrors( "option_load: can only add a option to a Rhs" );
		return( FALSE );
	}

	FREEF( slist_free_all, option->labels );

	if( get_slprop( xnode, "labels", &option->labels ) &&
		get_iprop( xnode, "value", &option->value ) )
		classmodel_set_edited( CLASSMODEL( model ), TRUE );

	return( MODEL_CLASS( parent_class )->load( model, 
		state, parent, xnode ) );
}

/* Update Option from heap.
 */
static gboolean
option_class_get( Classmodel *classmodel, PElement *root )
{
	Option *option = OPTION( classmodel );
	char buf[MAX_STRSIZE];
	GSList *labels;
	int value;

	/* Reread from heap.
	 */
	labels = NULL;
	if( !class_get_member_string( root, MEMBER_CAPTION, 
		buf, MAX_STRSIZE ) ||
		!class_get_member_int( root, MEMBER_VALUE, &value ) ||
		!class_get_member_lstring( root, MEMBER_LABELS, &labels ) )
		return( FALSE );

	/* Install.
	 */
	FREEF( slist_free_all, option->labels );
	SETSTR( MODEL( option )->caption, buf );
	option->labels = labels;
	option->value = value;

	return( TRUE );
}

/* Make a new "fn caption labels value" application.
 */
static gboolean
option_class_new( Classmodel *classmodel, PElement *fn, PElement *out )
{
	Heap *hi = reduce_context->hi;
	Option *option = OPTION( classmodel );

	PElement rhs;

	/* Make application nodes.
	 */
	heap_appl_init( out, fn );
	if( !heap_appl_add( hi, out, &rhs ) ||
		!heap_string_new( hi, MODEL( classmodel )->caption, &rhs ) ||
		!heap_appl_add( hi, out, &rhs ) ||
		!heap_lstring_new( hi, option->labels, &rhs ) ||
		!heap_appl_add( hi, out, &rhs ) ||
		!heap_real_new( hi, (double) option->value, &rhs ) )
		return( FALSE );

	return( TRUE );
}

static void
option_class_init( OptionClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	ModelClass *model_class = (ModelClass *) klass;
	ClassmodelClass *classmodel_class = (ClassmodelClass *) klass;

	parent_class = gtk_type_class( TYPE_CLASSMODEL );

	object_class->destroy = option_destroy;

	/* Create signals.
	 */

	/* Init methods.
	 */
	model_class->view_new = optionview_new;
	model_class->edit = option_edit;
	model_class->save = option_save;
	model_class->load = option_load;

	classmodel_class->class_get = option_class_get;
	classmodel_class->class_new = option_class_new;

	/* Static init.
	 */
	model_register_loadable( MODEL_CLASS( klass ) );
}

static void
option_init( Option *option )
{
        option->labels = NULL;
	option->value = 1;

	model_set( MODEL( option ), CLASS_OPTION, NULL );
}

GtkType
option_get_type( void )
{
	static GtkType option_type = 0;

	if( !option_type ) {
		static const GtkTypeInfo info = {
			"Option",
			sizeof( Option ),
			sizeof( OptionClass ),
			(GtkClassInitFunc) option_class_init,
			(GtkObjectInitFunc) option_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		option_type = gtk_type_unique( TYPE_CLASSMODEL, &info );
	}

	return( option_type );
}

Classmodel *
option_new( Rhs *rhs )
{
	Option *option = gtk_type_new( TYPE_OPTION );

	model_child_add( MODEL( rhs ), MODEL( option ), -1 );

	return( CLASSMODEL( option ) );
}
