/* Basic ops on symbols.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#include "ip.h"

/* All debug
#define DEBUG
 */

/* Just trace create/destroy.
#define DEBUG_MAKE
 */

/* Time recomputes.
#define DEBUG_TIME
 */

/* If DEBUG is on, make sure other debugs are on too.
 */
#ifdef DEBUG
#ifndef DEBUG_MAKE
#define DEBUG_MAKE
#endif
#ifndef DEBUG_TIME
#define DEBUG_TIME
#endif
#endif

#define CALC_RECOMP (watch_bool_get( "CALC_RECOMP", TRUE ))

/* Global symbol - all top-level definitions are locals to this root symbol.
 */
Symbol *symbol_root = NULL;

/* As we allow references to "root", root has itself to be on a table ...
 * this symbol, $$ROOT, holds that.
 */
Symbol *symbol_root_root = NULL;

/* Set of dirty top-level symbols with no dirty children which do not contain
 * errors. Used to generate next-to-recalc.
 */
static GSList *symbol_leaf_set = NULL;

/* Set if we are currently recalculating.
 */
gboolean symbol_busy = FALSE;

static FilemodelClass *parent_class = NULL;

/* Apply a function to a symbol ... and any locals.
 */
Symbol *
symbol_map_all( Symbol *sym, symbol_map_fn fn, void *a, void *b )
{
	Symbol *res;

	/* Apply to this sym.
	 */
	if( (res = fn( sym, a, b, NULL )) )
		return( res );

	/* And over any locals of those locals.
	 */
	if( sym->expr && sym->expr->compile && 
		(res = stable_map( sym->expr->compile->locals, 
			(symbol_map_fn) symbol_map_all, 
			(void *) fn, a, b )) )
		return( res );

	return( NULL );
}

/* Find a symbol's enclosing sym.
 */
Symbol *
symbol_get_parent( Symbol *sym )
{
	if( !sym->tab )
		return( NULL );

	return( sym->tab->compile->sym );
}

/* Find the enclosing workspace, if any.
 */
Workspace *
symbol_get_workspace( Symbol *sym )
{
	Symbol *i;

	for( i = sym; i && i->type != SYM_WORKSPACE; 
		i = symbol_get_parent( i ) )
		;
	if( i )
		return( WORKSPACE( i ) );

	return( NULL );
}

/* Find the enclosing tool, if any.
 */
Tool *
symbol_get_tool( Symbol *sym )
{
	Symbol *i;

	for( i = sym; i && !i->tool; i = symbol_get_parent( i ) )
		;
	if( i )
		return( i->tool );

	return( NULL );
}

/* Make a fully-qualified symbol name .. eg fred.jim, given jim.
 */
void
symbol_qualified_name( Symbol *sym, BufInfo *buf )
{
	/* Is this the root symbol? If it is, no qualification needed.
	 */
	if( sym == symbol_root )
		buf_appends( buf, MODEL( sym )->name );
	else {
		/* Qualify our parents, then do us. Special case: if our
		 * parent is root, don't bother printing it.
		 */
		if( sym->tab && symbol_get_parent( sym ) != symbol_root ) {
			symbol_qualified_name( symbol_get_parent( sym ), buf );
			buf_appends( buf, "." );
		}
		buf_appends( buf, NN( MODEL( sym )->name ) );
	}
}

/* Make a symbol name relative to a scope context ... ie. from the point of
 * view of a local of context, what name will find sym.
 */
void
symbol_qualified_name_relative( Symbol *context, Symbol *sym, BufInfo *buf )
{
	Symbol *parent = symbol_get_parent( sym );

	if( parent && !is_ancestor( context, parent ) ) {
		symbol_qualified_name_relative( context, parent, buf );
		buf_appends( buf, "." );
	}

	buf_appends( buf, NN( MODEL( sym )->name ) );
}

/* Handy for error messages ... but nowt else.
 */
const char *
symbol_name( Symbol *sym )
{
	static BufInfo buf;
	static char txt[200];

	buf_init_static( &buf, txt, 200 );
	buf_appends( &buf, "\"" );
	symbol_qualified_name( sym, &buf );
	buf_appends( &buf, "\"" );

	return( buf_all( &buf ) );
}

/* Convenience ... print a qual name to stdout.
 */
void
symbol_name_print( Symbol *sym )
{
	printf( "%s", symbol_name( sym ) );
}

/* Patch a pointer on a patch list. 
 */
static void *
symbol_patch_pointers_sub( void **pnt, void *nsym, void *osym )
{	
	assert( *pnt == osym );

	*pnt = nsym;

	return( NULL );
}

/* Patch pointers to old to point to new instead.
 */
void
symbol_patch_pointers( Symbol *nsym, Symbol *osym )
{
	(void) slist_map2( osym->patch, 
		(SListMap2Fn) symbol_patch_pointers_sub, nsym, osym );
}

/* Add a pointer to a patch list.
 */
void *
symbol_patch_add( void **pnt, Symbol *sym )
{
	assert( sym->type == SYM_ZOMBIE );

	sym->patch = g_slist_prepend( sym->patch, pnt );

	return( NULL );
}

static void
symbol_clear( Symbol *sym )
{
	sym->type = SYM_ZOMBIE;

	sym->dest = 0;

	sym->patch = NULL;

	sym->expr = NULL;
	sym->ext_expr = NULL;

	sym->base.type = ELEMENT_NOVAL;

	sym->dirty = FALSE;
	sym->parents = NULL;

	sym->topchildren = NULL;
	sym->topparents = NULL;
	sym->ndirtychildren = 0;
	sym->leaf = FALSE;

	sym->tool = NULL;
}

/* Initialise root symbol.
 */
void
symbol_root_init()
{
	symbol_root_root = gtk_type_new( TYPE_SYMBOL );

	symbol_clear( symbol_root_root );
	model_set( MODEL( symbol_root_root ), "$$ROOT", NULL );
	symbol_root_root->tab = NULL;
	symbol_root_root->type = SYM_ROOT;
	symbol_root_root->expr = expr_new( symbol_root_root );
	(void) compile_new_local( symbol_root_root->expr );

	symbol_root = 
		symbol_new( symbol_root_root->expr->compile->locals, "root" );
	symbol_root->type = SYM_ROOT;
	symbol_root->expr = expr_new( symbol_root );
	(void) compile_new_root( symbol_root->expr );
}

/* Should a symbol be in the leaf set?
 */
static gboolean
symbol_isleafable( Symbol *sym )
{
	if( is_top( sym ) && sym->dirty && sym->expr && !sym->expr->err && 
		!sym->dest && sym->ndirtychildren == 0 )
		return( TRUE );

	return( FALSE );
}

#ifdef DEBUG
/* Do a sanity check on a symbol.
 */
void *
symbol_sanity( Symbol *sym ) 
{
	if( is_top( sym ) ) {
		if( symbol_ndirty( sym ) != sym->ndirtychildren )
			error( "sanity failure #1 for sym \"%s\"", 
				symbol_name( sym ) );
		if( sym->ext_expr )
			error( "sanity failure #4 for sym \"%s\"", 
				symbol_name( sym ) );
	}

	if( symbol_isleafable( sym ) && !sym->leaf )
		error( "sanity failure #2 for sym \"%s\"", symbol_name( sym ) );
	if( !symbol_isleafable( sym ) && sym->leaf )
		error( "sanity failure #3 for sym \"%s\"", symbol_name( sym ) );
	if( sym->leaf && !g_slist_find( symbol_leaf_set, sym ) )
		error( "sanity failure #6 for sym \"%s\"", symbol_name( sym ) );
	if( !sym->leaf && g_slist_find( symbol_leaf_set, sym ) )
		error( "sanity failure #7 for sym \"%s\"", symbol_name( sym ) );

	return( NULL );
}
#endif/*DEBUG*/

#ifdef DEBUG
/* Test the leaf set for sanity.
 */
void
symbol_leaf_set_sanity( void )
{
	slist_map( symbol_leaf_set, (SListMapFn) symbol_sanity, NULL );
	if( symbol_root->expr->compile->locals )
		stable_map( symbol_root->expr->compile->locals,
			(symbol_map_fn) symbol_sanity, NULL, NULL, NULL );

	/* Commented out to reduce spam
	 *
	printf( "Leaf set: " );
	slist_map( symbol_leaf_set, (SListMapFn) dump_tiny, NULL );
	printf( "\n" );
	 *
	 */
}
#endif /*DEBUG*/

/* Strip a symbol down, ready for redefinition. 
 */
void *
symbol_strip( Symbol *sym )
{
#ifdef DEBUG_MAKE
	printf( "symbol_strip: " );
	symbol_name_print( sym );
	printf( "\n" );
#endif /*DEBUG_MAKE*/

	/* Clean out old exprinfo.
	 */
	if( sym->expr )
		expr_strip( sym->expr );
	(void) slist_map( sym->ext_expr, (SListMapFn) expr_strip, NULL );

	/* Set inital state again. 
	 */
	if( is_top( sym ) ) 
		symbol_dirty_clear( sym );

	/* Note the impending destruction of this symbol.
	 */
	sym->dest++;

	/* Free any top-links we made.
	 */
	(void) slist_map( sym->topchildren, (SListMapFn) link_destroy, NULL );

	/* Can free the patch list. We should not have to resolve off this
	 * name again.
	 */
	FREEF( g_slist_free, sym->patch );

	/* It's a ZOMBIE now.
	 */
	sym->type = SYM_ZOMBIE;

	/* Finished removing it. 
	 */
	sym->dest--;

#ifdef DEBUG
	symbol_sanity( sym );
#endif /*DEBUG*/

	return( NULL );
}

static void *
symbol_made_error_clear( Link *link )
{
	expr_error_clear( link->parent->expr );

	return( NULL );
}

/* Finish creating a symbol. Sequence is: symbol_new(), specialise ZOMBIE
 * into a particular symbol type, symbol_made(). Do any final tidying up.
 */
void
symbol_made( Symbol *sym )
{
#ifdef DEBUG_MAKE
	printf( "symbol_made: " );
	symbol_name_print( sym );
	printf( "\n" );
#endif /*DEBUG_MAKE*/

	if( is_top( sym ) ) {
		/* Remake all top-level dependencies.
		 */
		(void) symbol_link_build( sym );

		/* Clear error on every symbol that refs us, then mark dirty.
		 * This lets us replace refed-to syms cleanly.
		 */
		slist_map( sym->topparents,
			(SListMapFn) symbol_made_error_clear, NULL );

		/* Real dirrrrty.
		 */
		if( sym->expr )
			expr_dirty( sym->expr, link_serial_new() );
	}

#ifdef DEBUG
	dump_symbol( sym );
#endif /*DEBUG*/
}

/* Compile refers to sym, which is going ... mark compile as containing an 
 * error.
 */
static void *
symbol_destroy_error( Compile *compile, Symbol *sym )
{
	BufInfo buf;
	char str[80];

	buf_init_static( &buf, str, 80 );
	buf_appendf( &buf, "symbol \"" );
	symbol_qualified_name( sym, &buf );
	buf_appendf( &buf, "\" is not defined" );
	ierrors( "%s", buf_all( &buf ) );

	compile_error_set( compile );

	return( NULL );
}

/* Unlink a symbol from everything. 
 */
static void 
symbol_destroy( GtkObject *object )
{
	Symbol *sym;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_SYMBOL( object ) );

	sym = SYMBOL( object );

#ifdef DEBUG_MAKE
	printf( "symbol_destroy: " );
	symbol_name_print( sym );
	printf( "\n" );
#endif /*DEBUG_MAKE*/

	/* Note the impending destruction of this symbol. It's a state
	 * change, take off the leaf set if it's there.
	 */
	sym->dest++;
	symbol_state_change( sym );

	/* Clear state.
	 */
	if( is_top( sym ) ) {
		/* All stuff that depends on this sym is now dirty.
		 */
		symbol_dirty_intrans( sym, link_serial_new() );
		symbol_dirty_clear( sym );
	}

	/* Strip it down.
	 */
	(void) symbol_strip( sym );

	/* Junk any tool. When we destroy the tool, we don't want to destroy
	 * ourselves ... unlink first.
	 */
	if( sym->tool ) {
		Tool *tool = sym->tool;

		sym->tool = NULL;
		tool->sym = NULL;

		FREEO( tool );
	}

	/* Any exprs which refer to us must have errors.
	 */
	(void) slist_map( sym->parents, 
		(SListMapFn) symbol_destroy_error, sym );

	/* Remove links from any expr which refer to us.
	 */
	(void) slist_map( sym->parents, (SListMapFn) compile_link_break, sym );

	/* No other syms should have toplinks to us.
	 */
	(void) slist_map( sym->topparents, (SListMapFn) link_destroy, NULL );

	/* Unregister value with GC.
	 */
	reduce_unregister( sym );

	stable_unlink( sym );

	(void) slist_map( sym->ext_expr, (SListMapFn) expr_destroy, NULL );
	FREEF( expr_destroy, sym->expr );
	sym->tab = NULL;

	/* Finished removing it. 
	 */
	sym->dest--;

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

/* Final death of a symbol.
 */
static void 
symbol_finalize( GtkObject *object )
{
	Symbol *sym;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_SYMBOL( object ) );

	sym = SYMBOL( object );

#ifdef DEBUG_MAKE
	printf( "symbol_finalize: " );
	symbol_name_print( sym );
	printf( " (0x%x)\n", (unsigned int) sym );
#endif /*DEBUG_MAKE*/

	/* Free other stuff. 
	 */
	sym->type = SYM_ZOMBIE; 

	assert( !sym->tool );
	assert( !sym->parents );
	assert( !sym->topparents );
	assert( !sym->topchildren );

	FREEF( g_slist_free, sym->patch );
	FREEF( g_slist_free, sym->parents );

	GTK_OBJECT_CLASS( parent_class )->finalize( object );
}

static void
symbol_changed( Model *model )
{
	Symbol *sym = SYMBOL( model );

	/* If we have a tool, signal changed on that as well.
	 */
	if( sym->tool )
		model_changed( MODEL( sym->tool ) );

	MODEL_CLASS( parent_class )->changed( model );
}

static gboolean
symbol_load( Model *model, 
	ModelLoadState *state, Model *parent, xmlNode *xnode )
{
	Symbol *sym = SYMBOL( model );
	Symbol *psym = SYMBOL( parent );

	char name[256];

	/* "name" is really a model property ... but we need it for
	 * stable_add(), so grab it here as well.
	 */
	if( !get_sprop( xnode, "name", name, 256 ) )
		return( FALSE );
	model_set( MODEL( sym ), name, NULL );
	stable_add( psym->expr->compile->locals, sym );

	symbol_made( sym );

	if( !MODEL_CLASS( parent_class )->load( model, state, parent, xnode ) )
		return( FALSE );

	return( TRUE );
}

static void
symbol_class_init( SymbolClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	ModelClass *model_class = (ModelClass *) klass;

	parent_class = gtk_type_class( TYPE_FILEMODEL );

	object_class->destroy = symbol_destroy;
	object_class->finalize = symbol_finalize;

	model_class->changed = symbol_changed;
	model_class->load = symbol_load;

	/* Create signals.
	 */

	/* Init default methods.
	 */
}

static void
symbol_init( Symbol *sym )
{
	symbol_clear( sym );

	sym->tab = NULL;

	reduce_register( sym );

#ifdef DEBUG_MAKE
	printf( "symbol_init: (0x%x)\n", (unsigned int) sym );
#endif /*DEBUG_MAKE*/
}

GtkType
symbol_get_type( void )
{
	static GtkType symbol_type = 0;

	if (!symbol_type) {
		static const GtkTypeInfo sym_info = {
			"Symbol",
			sizeof( Symbol ),
			sizeof( SymbolClass ),
			(GtkClassInitFunc) symbol_class_init,
			(GtkObjectInitFunc) symbol_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		symbol_type = 
			gtk_type_unique( TYPE_FILEMODEL, &sym_info );
	}

	return( symbol_type );
}

void 
symbol_link( Symbol *sym, SymTable *tab, const char *name )
{
	model_set( MODEL( sym ), name, NULL );
	stable_add( tab, sym );
}

/* Make a new symbol on an expr. If it's already there and a ZOMBIE, just 
 * return it. If it's not a ZOMBIE, turn it into one. Otherwise make and 
 * link on a new symbol.
 */
Symbol *
symbol_new( SymTable *tab, const char *name )
{
	Symbol *sym;

	if( (sym = stable_find( tab, name )) ) {
		if( sym->type != SYM_ZOMBIE ) {
			/* Already exists: strip it down.
			 */
			(void) symbol_strip( sym );
		}

#ifdef DEBUG_MAKE
		printf( "symbol_new: redefining " );
		symbol_name_print( sym );
		printf( " (0x%x)\n", (unsigned int) sym );
#endif /*DEBUG_MAKE*/
	}
	else {
		sym = gtk_type_new( TYPE_SYMBOL );

		symbol_link( sym, tab, name );

#ifdef DEBUG_MAKE
		printf( "symbol_new: creating " );
		symbol_name_print( sym );
		printf( " (0x%x)\n", (unsigned int) sym );
#endif /*DEBUG_MAKE*/
	}

	return( sym );
}

/* Name in defining occurence. If this is a top-level definition, clean the
 * old symbol and get ready to attach a user function to it. If its not a top-
 * level definition, we flag an error. Consider repeated parameter names, 
 * repeated occurence of names in locals, local name clashes with parameter
 * name etc.
 * We make a ZOMBIE: our caller should turn it into a blank user definition, a
 * parameter etc.
 */
Symbol *
symbol_new_defining( Compile *compile, const char *name )
{
	Symbol *sym;

	/* Block definition of "root" anywhere ... too confusing.
	 */
	if( strcmp( name, MODEL( symbol_root )->name ) == 0 )
		yyerror( "attempt to redefine root symbol \"%s\"", name );

	/* Is this a redefinition of an existing symbol?
	 */
	if( (sym = stable_find( compile->locals, name )) ) {
		/* Yes. Check that this redefinition is legal.
		 */
		switch( sym->type ) {
		case SYM_VALUE:
			/* Redef of existing symbol? Only allowed at top
			 * level.
			 */
			if( compile->sym != symbol_root )
				/* Nameclash!
				 */
				yyerror( "name \"%s\" repeated in scope", 
					name );
			break;

		case SYM_PARAM:
			yyerror( "can't redefine parameter \"%s\"", name );
			/*NOTREACHED*/

		case SYM_ZOMBIE:
			/* This is the definition for a previously referenced
			 * symbol. Just return the ZOMBIE we made.
			 */
			break;

		case SYM_WORKSPACE:
			yyerror( "can't redefine workspace \"%s\"", name );
			/*NOTREACHED*/

		case SYM_WORKSPACEGROUP:
			yyerror( "can't redefine workspacegroup \"%s\"", name );
			/*NOTREACHED*/

		case SYM_ROOT:
			assert( FALSE );
			/*NOTREACHED*/

		case SYM_EXTERNAL:
			yyerror( "can't redefine external function \"%s\"", 
				name );
			/*NOTREACHED*/

		case SYM_BUILTIN:
			yyerror( "can't redefine built in function \"%s\"", 
				name );
			/*NOTREACHED*/

		default:
			assert( FALSE ); 
			/*NOTREACHED*/
		}

		/* This is the defining occurence ... move to the end of the
		 * traverse orser.
		 */
		stable_to_end( sym );
	}

	/* Get it ready.
	 */
	sym = symbol_new( compile->locals, name );

	return( sym );
}

/* Make a reference to a symbol. Look on the local table for the name - if
 * it's not there, make a ZOMBIE. Note that ZOMBIEs etc. need patch lists
 * attached to them for all pointers to them we make. Responsibility of
 * caller!
 */
Symbol *
symbol_new_reference( Compile *compile, const char *name )
{
	Symbol *sym = stable_find( compile->locals, name );

	if( !sym )
		sym = symbol_new( compile->locals, name );

	/* Note the new dependency.
	 */
	compile_link_make( compile, sym );

	return( sym );
}

/* Compile refers to child ... break link.
 */
void *
symbol_link_break( Symbol *child, Compile *compile )
{
	compile_link_break( compile, child );

	return( NULL );
}

/* Specialise into a user definition.
 */
gboolean
symbol_user_init( Symbol *sym )
{
	assert( sym->type == SYM_ZOMBIE );

	sym->type = SYM_VALUE;
	if( !sym->expr ) 
		sym->expr = expr_new( sym );

	/* We don't symbol_made() yet, wait until we have finished building
	 * sym->expr.
	 */

	return( TRUE );
}

/* Specialise into a parameter on an expression.
 */
gboolean
symbol_parameter_init( Symbol *sym, Compile *compile )
{
	assert( sym->type == SYM_ZOMBIE );

	sym->type = SYM_PARAM;
	compile->param = g_slist_append( compile->param, sym );
	compile->nparam = g_slist_length( compile->param );
	symbol_made( sym );

	return( TRUE );
}

/* Specialise into a builtin parameter (eg. "this").
 */
gboolean
symbol_parameter_builtin_init( Symbol *sym )
{
	assert( sym->type == SYM_ZOMBIE );

	sym->type = SYM_PARAM;
	symbol_made( sym );

	return( TRUE );
}

/* Get the next dirty leaf symbol.
 */
Symbol *
symbol_leaf_next( void ) 
{
	if( symbol_leaf_set )
		return( (Symbol *) symbol_leaf_set->data );
	else
		return( NULL ); 
}

/* Set leaf state.
 */
static void
symbol_set_leaf( Symbol *sym, gboolean leaf )
{
	if( sym->leaf != leaf ) {
		sym->leaf = leaf;

		if( leaf )
			symbol_leaf_set = 
				g_slist_prepend( symbol_leaf_set, sym );
		else
			symbol_leaf_set = 
				g_slist_remove( symbol_leaf_set, sym );

		if( sym->expr && sym->expr->row )
			model_changed( MODEL( sym->expr->row ) );
	}
}

/* State of a symbol has changed ... update! 
 */
void
symbol_state_change( Symbol *sym )
{
	assert( sym->ndirtychildren >= 0 );

	/* Used to do more ... now we just set leaf.
	 */
	symbol_set_leaf( sym, symbol_isleafable( sym ) );
}

/* Recalculate a symbol. We know we are dirty and have no dirty ancestors.
 */
static gboolean
symbol_recalculate_sub( Symbol *sym )
{
#ifdef DEBUG
	printf( "symbol_recalculate_sub: " );
	symbol_name_print( sym );
	printf( "\n" );
#endif /*DEBUG*/

	if( is_value( sym ) ) {
		/* Is this the root of a display? If yes, use that recomp
		 * mechanism.
		 */
		if( sym->expr->row ) {
			row_recomp( sym->expr->row );
			if( sym->expr->row->err )
				return( FALSE );
		}
		else {
			gboolean res;
			PElement *root = &sym->expr->root;

			res = reduce_regenerate( sym->expr, root );
			expr_value_new( sym->expr );

			if( !res )
				return( FALSE );

			symbol_dirty_clear( sym );
		}
	}
	else
		symbol_dirty_clear( sym );

	return( TRUE );
}

/* Recalc a symbol ... with error checks.
 */
void *
symbol_recalculate_check( Symbol *sym )
{
	if( !sym->dirty ) 
		return( NULL );
	if( sym->expr->err ) {
		expr_error_get( sym->expr );
		return( sym );
	}
	if( !is_value( sym ) ) {
		symbol_dirty_clear( sym );
		return( NULL );
	}

	im_clear_error_string();
	reduce_context->hi->filled = FALSE;
	if( !symbol_recalculate_sub( sym ) || reduce_context->hi->filled ) {
		expr_error_set( sym->expr );
		return( sym );
	}

	return( NULL );
}

/* Recompute the leaf set.
 */
static void
symbol_recalculate_leaves( void )
{
	Symbol *sym;

	/* Make sure all repaints have been processed.
	 */
	(void) mainw_countdown_animate( 99 );

	/* Read all text widgets off the screen.
	 */
	(void) view_scan_all();

	/* Grab stuff off the leaf set.
	 */
	while( (sym = symbol_leaf_next()) ) {
		/* Should be dirty with no dirty children. Unless it's a
		 * function, in which case dirty kids are OK.
		 */
		assert( sym->dirty );
		assert( !sym->expr->err );
		assert( is_top( sym ) );
		assert( symbol_ndirty( sym ) == 0 || is_value( sym ) );

		/* Found a symbol!
		 */
		(void) symbol_recalculate_check( sym );

		/* Process some more repaints.
		 */
		if( mainw_countdown_animate( 99 ) )
			return;
	}

	/* Note a pending GC.
	 */
	(void) heap_gc_request( reduce_context->hi );
}

/* Recalculate the symbol table, even if auto recomp is off.
 */
void
symbol_recalculate_all_force( void )
{
#ifdef DEBUG_TIME
	static GTimer *compute_timer = NULL;

	if( !compute_timer )
		compute_timer = g_timer_new();
#endif /*DEBUG_TIME*/

	/* Make sure we cannot get nested calls. 
	 */
	if( symbol_busy )
		return;

#ifdef DEBUG
	/* Walk the whole table, doing a sanity check.
	 */
	stable_map( symbol_root->expr->compile->locals, 
		(symbol_map_fn) symbol_sanity, NULL, NULL, NULL );
#endif /*DEBUG*/

	/* Lock and calculate.
	 */
	symbol_busy = TRUE;
	reduce_total_recomputations = 0;
#ifdef DEBUG_TIME
	g_timer_reset( compute_timer );
#endif /*DEBUG_TIME*/
	symbol_recalculate_leaves();
	symbol_busy = FALSE;

#ifdef DEBUG_TIME
	printf( "symbol_recalculate_all: %d reductions in %g secs\n",
		reduce_total_recomputations,
		g_timer_elapsed( compute_timer, NULL ) );
#endif /*DEBUG_TIME*/

	/* Request a save of the current workspace.
	 */
	if( main_workspacegroup->current )
		workspace_checkmark( main_workspacegroup->current );

	mainw_countdown_end();
	mainw_next_error_refresh();
}

/* Recalculate the symbol table.
 */
void
symbol_recalculate_all( void )
{
	if( CALC_RECOMP )
		symbol_recalculate_all_force();
}

/* Do a general recalc, then repeat for a particular sym to pick up any
 * errors.
 */
void *
symbol_recalculate( Symbol *sym )
{
	if( symbol_busy )
		return( NULL );

	symbol_recalculate_all();
	return( symbol_recalculate_check( sym ) );
}

static void *
symbol_help_sub( Symbol *param, BufInfo *buf )
{
	buf_appends( buf, " " );
	buf_appends( buf, MODEL( param )->name );

	return( NULL );
}

/* Fill a buf with help text for a symbol.
 */
void
symbol_help( Symbol *sym, BufInfo *buf )
{
	if( sym->expr && sym->expr->compile && sym->expr->compile->text ) {
		char *p = sym->expr->compile->text;

		/* Skip leading whitespace.
		 */
		while( isspace( (int)(*p) ) )
			p++;

		/* Skip leading comment, if any.
		 */
		if( p[0] == '/' && p[1] == '*' )
			p += 2;
		else if( p[0] == '/' && p[1] == '/' )
			p += 2;

		/* Skip more whitespace.
		 */
		while( isspace( (int)(*p) ) )
			p++;

		symbol_qualified_name( sym, buf );
		slist_map( sym->expr->compile->param,
			(SListMapFn) symbol_help_sub, buf );
		buf_appends( buf, ": " );

		buf_appendline( buf, p );
	}
}
