/* a slider with an entry widget
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

*/

/*
#define DEBUG
 */

#include "ip.h"

/* Our signals. 
 */
enum {
	CHANGED,		
	ACTIVATE,	
	SLIDER_CHANGED,
	TEXT_CHANGED,	
	LAST_SIGNAL
};

static GtkHBoxClass *parent_class = NULL;

static guint tslider_signals[LAST_SIGNAL] = { 0 };

static gint tslider_idle_running = 0;

static GSList *tslider_to_unref = NULL;

static void *
tslider_idle_unref( GtkRange *range )
{
	/* 
	
		FIXME ... workaround for bug in gtk+-1.2.x :-( 

	 */
	FREEFI( gtk_timeout_remove, range->timer );

	gtk_object_unref( GTK_OBJECT( range ) );

	return( NULL );
}

/* Refresh everything on the dirty list.
 */
static gint
tslider_idle_unref_cb( gpointer user_data )
{
	slist_map( tslider_to_unref, (SListMapFn) tslider_idle_unref, NULL );
	FREEF( g_slist_free, tslider_to_unref );

	tslider_idle_running = 0;

	return( FALSE );
}

/* Add something to the pending-unref list. Delay unref until the idle loop.
 */
static void 
tslider_pending_unref( GtkRange *range )
{
	gtk_object_ref( GTK_OBJECT( range ) );

	tslider_to_unref = g_slist_prepend( tslider_to_unref, range );

	if( !tslider_idle_running )
		tslider_idle_running = 
			gtk_idle_add( tslider_idle_unref_cb, NULL );
}

/* Are two doubles more or less equal. We need this when we check the sliders
 * for update to stop loops. The 0.0001 is a bit of a fudge :-(
 */
#define DEQ( A, B ) (ABS((A) - (B)) < 0.0001)

static void
tslider_destroy( GtkObject *object )
{
	Tslider *tslider;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_TSLIDER( object ) );

	tslider = TSLIDER( object );

#ifdef DEBUG
	printf( "tslider_destroy: 0x%x\n", (unsigned int) tslider );
#endif /*DEBUG*/

	/* My instance destroy stuff.
	 */
	if( tslider->adj ) {
		gtk_signal_disconnect_by_data( GTK_OBJECT( tslider->adj ),
			(gpointer) tslider );
		tslider->adj = NULL;
	}

	/* Yuk!!! We can sometimes destroy a tslider from the adjustment
	 * value_changed callback, causing extreme confusion. 
	 */
	tslider_pending_unref( GTK_RANGE( tslider->slider ) );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

/* Map a value to a slider position.
 */
static double
tslider_value_to_slider( Tslider *tslider, double value )
{
	/* Map our range to 0-1.
	 */
	const double scale = 1.0 / (tslider->to - tslider->from);
	const double to01 = (value - tslider->from) * scale;

	/* Pass through user fn.
	 */
	const double mapped = tslider->value_to_slider( 
		tslider->from, tslider->to, to01 );
	const double nvalue = mapped / scale + tslider->from;

#ifdef DEBUG
	printf( "tslider_value_to_slider: %g, to %g\n", value, nvalue );
#endif /*DEBUG*/

	/* Map back to main range.
	 */
	return( nvalue );
}

/* Map a slider position to a value.
 */
static double
tslider_slider_to_value( Tslider *tslider, double value )
{
	/* Map our range to 0-1.
	 */
	const double scale = 1.0 / (tslider->to - tslider->from);
	const double to01 = (value - tslider->from) * scale;

	/* Pass through user fn.
	 */
	const double mapped = tslider->slider_to_value( 
		tslider->from, tslider->to, to01 );
	const double nvalue = mapped / scale + tslider->from;

#ifdef DEBUG
	printf( "tslider_slider_to_value: %g, to %g\n", value, nvalue );
#endif /*DEBUG*/

	/* Map back to main range.
	 */
	return( nvalue );
}

/* from/to/value have changed ... update the widgets.
 */
static void
tslider_real_changed( Tslider *tslider )
{
	GtkAdjustment *adj = tslider->adj;
	GtkWidget *entry = tslider->entry;

#ifdef DEBUG
	printf( "tslider_real_changed: 0x%x, val = %g\n", 
		(unsigned int) tslider, tslider->value );
#endif /*DEBUG*/

	if( tslider->auto_link ) 
		tslider->svalue = tslider_value_to_slider( tslider, 
			tslider->value );

	gtk_signal_handler_block_by_data( GTK_OBJECT( adj ), tslider );
	gtk_signal_handler_block_by_data( GTK_OBJECT( entry ), tslider );

	/* Some libc's hate out-of-bounds precision, so clip, just in case.
	 */
	set_gentry( tslider->entry, "%.*f", 
		IM_CLIP( 0, tslider->digits, 100 ), tslider->value );
	gtk_scale_set_digits( GTK_SCALE( tslider->slider ), tslider->digits );

	if( !DEQ( tslider->from, tslider->last_from ) || 
		!DEQ( tslider->to, tslider->last_to ) ) {
		double range = tslider->to - tslider->from;

		adj->step_increment = range / 100;
		adj->page_increment = range / 10;
		adj->page_size = range / 10;

		adj->lower = tslider->from;
		adj->upper = tslider->to + adj->page_size;

		tslider->last_to = tslider->to;
		tslider->last_from = tslider->from;

		gtk_adjustment_changed( adj );
	}

	if( !DEQ( tslider->svalue, tslider->last_svalue ) ) {
		adj->value = tslider->svalue;
		tslider->last_svalue = tslider->svalue;

		gtk_adjustment_value_changed( adj );
	}

	gtk_signal_handler_unblock_by_data( GTK_OBJECT( adj ), tslider );
	gtk_signal_handler_unblock_by_data( GTK_OBJECT( entry ), tslider );
}

static void
tslider_class_init( TsliderClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;

	parent_class = gtk_type_class( GTK_TYPE_HBOX );

	object_class->destroy = tslider_destroy;

	klass->changed = tslider_real_changed;
	klass->slider_changed = NULL;
	klass->activate = NULL;

	/* Create signals.
	 */
	tslider_signals[CHANGED] = gtk_signal_new( "changed",
		GTK_RUN_FIRST,
		object_class->type,
		GTK_SIGNAL_OFFSET( TsliderClass, changed ),
		gtk_marshal_NONE__NONE,
		GTK_TYPE_NONE, 0 );
	tslider_signals[ACTIVATE] = gtk_signal_new( "activate",
		GTK_RUN_FIRST,
		object_class->type,
		GTK_SIGNAL_OFFSET( TsliderClass, activate ),
		gtk_marshal_NONE__NONE,
		GTK_TYPE_NONE, 0 );
	tslider_signals[SLIDER_CHANGED] = gtk_signal_new( "slider_changed",
		GTK_RUN_FIRST,
		object_class->type,
		GTK_SIGNAL_OFFSET( TsliderClass, slider_changed ),
		gtk_marshal_NONE__NONE,
		GTK_TYPE_NONE, 0 );
	tslider_signals[TEXT_CHANGED] = gtk_signal_new( "text_changed",
		GTK_RUN_FIRST,
		object_class->type,
		GTK_SIGNAL_OFFSET( TsliderClass, text_changed ),
		gtk_marshal_NONE__NONE,
		GTK_TYPE_NONE, 0 );

	gtk_object_class_add_signals( object_class, 
		tslider_signals, LAST_SIGNAL );

	/* Init methods.
	 */
}

/* From/to/value have changed ... tell everyone.
 */
void
tslider_changed( Tslider *tslider )
{
#ifdef DEBUG
	printf( "tslider_changed\n" );
#endif /*DEBUG*/

	gtk_signal_emit( GTK_OBJECT( tslider ), tslider_signals[CHANGED] );
}

/* Activated!
 */
static void
tslider_activate( Tslider *tslider )
{
#ifdef DEBUG
	printf( "tslider_activate\n" );
#endif /*DEBUG*/

	gtk_signal_emit( GTK_OBJECT( tslider ), tslider_signals[ACTIVATE] );
}

/* Just the slider changed.
 */
static void
tslider_slider_changed( Tslider *tslider )
{
#ifdef DEBUG
	printf( "tslider_slider_changed\n" );
#endif /*DEBUG*/

	gtk_signal_emit( GTK_OBJECT( tslider ), 
		tslider_signals[SLIDER_CHANGED] );
}

/* Text has been touched.
 */
static void
tslider_text_changed( Tslider *tslider )
{
#ifdef DEBUG
	printf( "tslider_text_changed\n" );
#endif /*DEBUG*/

	gtk_signal_emit( GTK_OBJECT( tslider ), tslider_signals[TEXT_CHANGED] );
}

/* Enter in entry widget
 */
static void
tslider_value_activate_cb( GtkWidget *entry, Tslider *tslider )
{
	double value;

	if( !get_geditable_double( entry, &value ) ) {
		box_alert( entry );
		return;
	}

	if( tslider->value != value ) {
		tslider->value = value;

		if( tslider->auto_link ) 
			tslider_changed( tslider );
		else
			tslider_activate( tslider );
	}
}

/* Drag on slider.
 */
static void
tslider_value_changed_cb( GtkAdjustment *adj, Tslider *tslider )
{
#ifdef DEBUG
	printf( "tslider_value_changed_cb\n" );
#endif /*DEBUG*/

	if( tslider->svalue != adj->value ) {
		tslider->svalue = adj->value;

		if( tslider->auto_link ) {
			tslider->value = 
				tslider_slider_to_value( tslider, adj->value );

			tslider_changed( tslider );
		}
		else 
			tslider_slider_changed( tslider );
	}
}

/* Text has changed (and may need to be scanned later).
 */
static void
tslider_text_changed_cb( GtkWidget *widget, Tslider *tslider )
{
#ifdef DEBUG
	printf( "tslider_text_changed_cb\n" );
#endif /*DEBUG*/

	tslider_text_changed( tslider );
}

/* Default identity conversion.
 */
static double
tslider_conversion_id( double from, double to, double value )
{
	return( value );
}

static void
tslider_init( Tslider *tslider )
{
#ifdef DEBUG
	printf( "tslider_init: 0x%x\n", (unsigned int) tslider );
#endif /*DEBUG*/

	/* Any old start values ... overridden later.
	 */
	tslider->from = -1;
	tslider->to = -1;
	tslider->value = -1;
	tslider->svalue = -1;
	tslider->digits = -1;
	tslider->last_to = -1;
	tslider->last_from = -1;
	tslider->last_svalue = -1;

        gtk_box_set_spacing( GTK_BOX( tslider ), 2 );

	tslider->entry = build_entry( 4 );
	gtk_entry_set_max_length( GTK_ENTRY( tslider->entry ), 10 );
        set_tooltip( tslider->entry, "Slider value ... edit!" );
        gtk_box_pack_start( GTK_BOX( tslider ), 
		tslider->entry, FALSE, FALSE, 0 );
        gtk_signal_connect( GTK_OBJECT( tslider->entry ), "activate",
                GTK_SIGNAL_FUNC( tslider_value_activate_cb ), tslider );
        gtk_signal_connect( GTK_OBJECT( tslider->entry ), "changed",
                GTK_SIGNAL_FUNC( tslider_text_changed_cb ), tslider );
	gtk_widget_show( tslider->entry );

        tslider->slider = gtk_hscale_new( NULL );
	tslider->adj = gtk_range_get_adjustment( GTK_RANGE( tslider->slider ) );
        gtk_range_set_update_policy( GTK_RANGE( tslider->slider ),
		GTK_UPDATE_CONTINUOUS );
#ifdef DEBUG
	/*
        gtk_range_set_update_policy( GTK_RANGE( tslider->slider ),
		GTK_UPDATE_DISCONTINUOUS );
	 */
#endif /*DEBUG*/
        gtk_scale_set_draw_value( GTK_SCALE( tslider->slider ), FALSE );
        gtk_box_pack_start( GTK_BOX( tslider ), 
		tslider->slider, TRUE, TRUE, 0 );
        set_tooltip( tslider->slider, "Left-drag to set number" );
        gtk_signal_connect( GTK_OBJECT( tslider->adj ), "value_changed", 
		GTK_SIGNAL_FUNC( tslider_value_changed_cb ), tslider );
	gtk_widget_show( tslider->slider );

	tslider->auto_link = TRUE;
	tslider->slider_to_value = tslider_conversion_id;
	tslider->value_to_slider = tslider_conversion_id;
}

GtkType
tslider_get_type( void )
{
	static GtkType tslider_type = 0;

	if( !tslider_type ) {
		static const GtkTypeInfo sinfo = {
			"Tslider",
			sizeof( Tslider ),
			sizeof( TsliderClass ),
			(GtkClassInitFunc) tslider_class_init,
			(GtkObjectInitFunc) tslider_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		tslider_type = gtk_type_unique( GTK_TYPE_HBOX, &sinfo );
	}

	return( tslider_type );
}

Tslider *
tslider_new()
{
	Tslider *tslider = gtk_type_new( TYPE_TSLIDER );

	return( tslider );
}

void
tslider_set_conversions( Tslider *tslider, 
	tslider_fn value_to_slider, tslider_fn slider_to_value )
{
	tslider->value_to_slider = value_to_slider;
	tslider->slider_to_value = slider_to_value;

	tslider->auto_link = value_to_slider && slider_to_value;
}

double
tslider_log_value_to_slider( double from, double to, double value )
{
	/* What does 1.0 map to on our [0,1] scale?
	 */
	const double mapped1 = (1.0 - from) / (to - from);

	/* We want an exponent which maps the mid point on the slider to 1.
	 */
	const double a = log( mapped1 ) / log( 0.5 );

	const double nvalue = pow( value, 1.0 / a );

	return( nvalue );
}

double
tslider_log_slider_to_value( double from, double to, double value )
{
	/* What does 1.0 map to on our [0,1] scale?
	 */
	const double mapped1 = (1.0 - from) / (to - from);

	/* We want an exponent which maps the mid point on the slider to 1.
	 */
	const double a = log( mapped1 ) / log( 0.5 );

	const double nvalue = pow( value, a );

	return( nvalue );
}
