/* main() ... start everything up. See mainw.c for main window stuff.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#include "ip.h"

/* 
#define DEBUG
 */

/* Stop startup creation of externs for all VIPS functions etc.
#define DEBUG_NOAUTO
 */

/* stop on any gtk error/warning/whatever
#define DEBUG_FATAL
 */

/* Time startup.
#define DEBUG_TIME
 */

/* On quit, make sure we free stuff we can free.
#define DEBUG_LEAK
 */

/* General stuff. 
 */
GdkWindow *main_window_gdk;			/* Top GdkWindow */
GtkWidget *main_window_top = NULL;		/* Secret top window */

Workspacegroup *main_workspacegroup = NULL;	/* All the workspaces */
Toolkitgroup *main_toolkitgroup = NULL;		/* All the toolkits */
Symbol *main_symbol_root = NULL;		/* Root of symtable */
Watchgroup *main_watchgroup = NULL;		/* All of the watches */
Imageinfogroup *main_imageinfogroup = NULL;	/* All of the images */

void *main_c_stack_base = NULL;			/* Base of C stack */

gboolean main_starting = TRUE;			/* In startup */

/* Action on main_quit()
 */
typedef enum {
	EXIT_ACTION_NONE,			/* Just quit */
	EXIT_ACTION_PRINT_MAIN			/* Print symbol "main" */
} ExitAction;

static ExitAction action_on_exit = EXIT_ACTION_NONE;

static const char *main_argv0 = NULL;		/* argv[0] */
static const char *main_name = NULL;		/* Name invoked as */
static iOpenFile *main_stdin = NULL;		/* stdin as an iOpenFile */
static gboolean main_interactive = TRUE;	/* In GUI mode */
static GtkIconFactory *main_icon_factory = NULL;/* Add stocks to this */
static Splash *main_splash = NULL;

/* Accumulate startup errors here.
 */
static char main_start_error_txt[MAX_STRSIZE];
static BufInfo main_start_error = 
	BUF_STATIC( main_start_error_txt, MAX_STRSIZE );

static void
main_log_add( const char *fmt, ... )
{
	va_list ap;

        va_start( ap, fmt );
	buf_vappendf( &main_start_error, fmt, ap );
        va_end( ap );
}

static const char *
main_log_get( void )
{
	return( buf_all( &main_start_error ) );
}

static gboolean
main_log_isempty( void )
{
	return( buf_isempty( &main_start_error ) );
}

static void
main_usage( void )
{
	printf( _(
"%s: usage:\n"
"%s filename1 filename2 ...\n"
"\tstart in GUI mode, loading the named files" ),
		main_name, main_name );
	printf( "\n" );

	printf( _(
"%s -script filename arg1 arg2 ...\n"
"\tread in filename as a set of definitions, set list argv to\n"
"\t[\"filename\", \"arg1\", \"arg2\", ...], set argc to length of list;\n"
"\tprint the value of symbol \"main\" to stdout; exit; useful for\n"
"\trunning %s as an interpreter on unix" ),
		main_name, main_name );
	printf( "\n" );

	printf( _(
"%s -main arg1 arg2 ...\n"
"\tas -script, but read the definitions from stdin rather than from a\n"
"\tfile; useful for here-is shell scripts" ),
		main_name );
	printf( "\n" );

	printf( _(
"%s -workspace arg1 arg2 ...\n"
"\tas -main, but read a workspace save file instead; run in GUI mode\n"
"\tprint the value of the final row in the save file on exit" ),
		main_name );
	printf( "\n" );

	printf( _(
"%s -benchmark\n"
"\tload all start objects and quit; useful for bechmarking the compiler" ),
		main_name );
	printf( "\n" );

	printf( _( "%s version: %s" ), PACKAGE, VERSION );
	printf( "\n" );

	printf( _( "linked to vips library: %s" ), im_version_string() );
	printf( "\n" );
}

/* NULL log handler. Used to suppress output on win32 without DEBUG_FATAL.
 */
#ifndef DEBUG_FATAL
#ifdef HAVE_WINDOWS_H 
static void
main_log_null( const char *log_domain, GLogLevelFlags log_level,
	const char *message, void *user_data )
{
}
#endif /*HAVE_WINDOWS_H*/ 
#endif /*!DEBUG_FATAL*/

/* Clean up our application and quit. Not interactive! Do any "has been
 * modified, OK to quit?" stuff before this.
 */
static void
main_quit( void )
{
	Symbol *sym;
#if HAVE_FFTW || HAVE_FFTW3
	iOpenFile *of;
#endif /*HAVE_FFTW || HAVE_FFTW3*/

#ifdef DEBUG
	printf( "main_quit: cleaning up ...\n" );
#endif/*DEBUG*/

	/* Any exit actions?
	 */
	switch( action_on_exit ) {
	case EXIT_ACTION_NONE:	
		break;

	case EXIT_ACTION_PRINT_MAIN:
		sym = compile_lookup( symbol_root->expr->compile, "main" );

		if( !sym )
			error( _( "symbol \"main\" not found" ) );
		if( symbol_recalculate( sym ) ||
			!reduce_pelement( reduce_context, reduce_spine_strict, 
				&sym->expr->root ) ) {
			BufInfo buf;
			char txt[MAX_STRSIZE];

			buf_init_static( &buf, txt, MAX_STRSIZE );
			expr_error_print_all( &buf );
			fprintf( stderr, "%s\n", buf_all( &buf ) );
			error( _( "error calculating \"main\"" ) );
		}

		(void) set_output( &sym->expr->root );
		(void) graph_value( &sym->expr->root );
		break;

	default:
		assert( FALSE );
	}

	/* Force all windows down. 
	
		FIXME 
		
		this can cause a recursive call to us from mainw_destroy()

	 */
	iwindow_map_all( (iWindowMapFn) iwindow_kill, NULL );
	mainw_shutdown();

	/* Dump wisdom back again.
	 */
#if HAVE_FFTW || HAVE_FFTW3
	if( (of = file_open_write( "%s", 
		IP_SAVEDIR IM_DIR_SEP_STR "wisdom" )) ) {
		fftw_export_wisdom_to_file( of->fp );
		file_close( of );
	}
#endif /*HAVE_FFTW*/

	/* Remove any ws retain files.
	 */
	workspace_retain_clean();

	/* Junk all symbols. This may remove a bunch of intermediate images
	 * too.
	 */
	UNREF( main_symbol_root );
	UNREF( main_toolkitgroup );
	UNREF( main_watchgroup );
	UNREF( main_workspacegroup );

	/* Junk reduction machine ... this should remove all image temps.
	 */
	reduce_destroy( reduce_context );

#ifdef DEBUG_LEAK
	printf( "DEBUG_LEAK: testing for leaks ...\n" );

	/* Free other GTK stuff.
	 */
	if( main_icon_factory )
		gtk_icon_factory_remove_default( main_icon_factory );

	/* Should have freed everything now.
	 */

	/* Make sure!

		FIXME ... #ifdef this lot out at some point

	 */
	imageinfo_check_all_destroyed( main_imageinfogroup );
	UNREF( main_imageinfogroup );
	heap_check_all_destroyed();
	im__print_all();
	util_check_all_destroyed();
	vips_check_all_destroyed();
#endif /*DEBUG_LEAK*/

#ifdef DEBUG
	printf( "main_quit: exit( 0 )\n" );
#endif/*DEBUG*/

	/* And exit.
	 */
	exit( 0 );
}

static void
main_quit_test_cb( void *sys, iWindowResult result )
{
	if( result == IWINDOW_TRUE )
		/* No return from this.
		 */
		main_quit();
}

/* Check before quitting.
 */
void
main_quit_test( void )
{
	/* Close registered models.
	 */
	filemodel_inter_close_registered_cb( IWINDOW( main_window_top ), NULL,
		main_quit_test_cb, NULL );
}

gboolean
main_splash_enabled( void )
{
	return( !existsf( "%s", NO_SPLASH ) );
}

void
main_splash_update( const char *fmt, ... )
{
	if( main_splash ) {
		va_list ap;

		va_start( ap, fmt );
		splash_updatev( main_splash, fmt, ap );
		va_end( ap );
	}

	animate_hourglass();
}

static void
main_watchgroup_changed_cb( void )
{
	static int last_max_cpus = -1;

	/* Try to avoid setting too often ... we can get leaks.
	 */
	if( last_max_cpus != VIPS_CPUS ) {
		last_max_cpus = VIPS_CPUS;
		putenvf( "IM_CONCURRENCY=%d", VIPS_CPUS );
	}
}

/* Try to load a thing, anything at all. Actually, we don't load plugins
 * experimentally: win32 pops up an annoying error dialog if you try that.
 */
gboolean
main_load( Workspace *ws, const char *filename )
{
	BufInfo buf;
	char txt[MAX_STRSIZE];
	Workspace *new_ws;

	main_splash_update( _( "Loading \"%s\"" ), im_skip_dir( filename ) );

	if( toolkit_new_from_file( main_toolkitgroup, filename ) )
		return( TRUE );
	error_clear();

	if( (new_ws = 
		workspace_new_from_file( main_workspacegroup, filename )) ) {
		Mainw *new_mainw;

		new_mainw = mainw_new( new_ws );
		gtk_widget_show( GTK_WIDGET( new_mainw ) );
		mainw_recent_add( &mainw_recent_workspace, filename );

		return( TRUE );
	}
	error_clear();

	/* Try as matrix or image. Have to do these via definitions.
	 */
	buf_init_static( &buf, txt, MAX_STRSIZE );
	buf_appends( &buf, "Matrix_file \"" );
	buf_appendsc( &buf, TRUE, filename );
	buf_appends( &buf, "\"" );
	if( workspace_add_def( ws, buf_all( &buf ) ) ) {
		mainw_recent_add( &mainw_recent_matrix, filename );
		return( TRUE );
	}
	error_clear();

	buf_init_static( &buf, txt, MAX_STRSIZE );
	buf_appends( &buf, "Image_file \"" );
	buf_appendsc( &buf, TRUE, filename );
	buf_appends( &buf, "\"" );
	if( workspace_add_def( ws, buf_all( &buf ) ) ) {
		mainw_recent_add( &mainw_recent_image, filename );
		return( TRUE );
	}
	error_clear();

	error_top( _( "Unknown file type." ) );
	error_sub( _( "Unable to load \"%s\"." ), filename );

	return( FALSE );
}

#ifndef DEBUG_NOAUTO
static void *
main_load_plug( char *name )
{
	if( !call_string_filename( (call_string_fn) im_load_plugin,
		name, NULL, NULL, NULL ) ) {
		error_top( _( "Unable to load." ) );
		error_sub( _( "Error loading plug-in \"%s\"." ), name );
		error_vips();
		box_alert( NULL );
	}

	return( NULL );
}
#endif /*!DEBUG_NOAUTO*/

static void *
main_load_def( const char *filename )
{	
	Toolkit *kit;

	/* If we're in command-line mode, only load toolkits which 
	 * start with '_'. For interactive mode, load everything.
	 */
	if( main_interactive || im_skip_dir( filename )[0] == '_' ) {
		main_splash_update( _( "Loading toolkit \"%s\"" ), 
			im_skip_dir( filename ) );

		if( !(kit = 
			toolkit_new_from_file( main_toolkitgroup, filename )) )
			box_alert( NULL );
		else 
			filemodel_set_auto_load( FILEMODEL( kit ) );
	}

	return( NULL );
}

static void *
main_load_ws( const char *filename )
{
	Workspace *ws;

#ifdef DEBUG
	printf( "main_load_ws: %s\n", filename );
#endif/*DEBUG*/

	main_splash_update( _( "Loading workspace \"%s\"" ), 
		im_skip_dir( filename ) );

	if( !(ws = workspace_new_from_file( main_workspacegroup, filename )) ) 
		box_alert( NULL );
	else {
		filemodel_set_auto_load( FILEMODEL( ws ) );
	}

	return( NULL );
}

#ifndef DEBUG_NOAUTO
/* Link all the packages in a function.
 */
static void *
main_link_package( im_package *pack)
{
	char name[MAX_STRSIZE];
	Toolkit *kit;
        int i;

	im_snprintf( name, MAX_STRSIZE, "_%s", pack->name );
	kit = toolkit_new( main_toolkitgroup, name );

        for( i = 0; i < pack->nfuncs; i++ ) 
		if( vips_is_callable( pack->table[i] ) ) {
			Symbol *sym;

			sym = symbol_new( symbol_root->expr->compile,
				pack->table[i]->name );
			assert( sym->type == SYM_ZOMBIE );
			sym->type = SYM_EXTERNAL;
			sym->function = pack->table[i];
			sym->fn_nargs = vips_n_args( pack->table[i] );
			(void) tool_new_sym( kit, -1, sym );
			symbol_made( sym );
		}

	filemodel_set_auto_load( FILEMODEL( kit ) );
	filemodel_set_modified( FILEMODEL( kit ), FALSE );
	kit->pseudo = TRUE;

        return( NULL );
}
#endif /*!DEBUG_NOAUTO*/

/* Load all plugins and defs.
 */
static void
main_load_all( void )
{
	const char *prefix;

	mainw_recent_freeze();

/* Stop load of builtins, plugs and vips ... handy for debugging if you're
 * tracing symbol.c
 */
#ifdef DEBUG_NOAUTO
	printf( "*** DEBUG_NOAUTO set, not loading builtin, plugs and vips\n" );
#else /*!DEBUG_NOAUTO*/

#ifdef DEBUG
	printf( "built-ins init\n" );
#endif/*DEBUG*/

	/* Add builtin toolkit.
	 */
	builtin_init();

#ifdef DEBUG
	printf( "plug-ins init\n" );
#endif/*DEBUG*/

	main_splash_update( "%s", _( "Loading plugins" ) );

	/* Load up any standard plugs.
	 */
	if( (prefix = im_guess_prefix( main_argv0, "VIPSHOME" )) &&
		im_load_plugins( "%s" IM_DIR_SEP_STR "lib", prefix ) ) {
		error_top( _( "Unable to load plugins." ) );
		error_sub( _( "Error loading plug-ins in "
			"directory \"%s" IM_DIR_SEP_STR "lib\"" ),
			NN( prefix ) );
		error_vips();
		box_alert( NULL );
	}

	/* Load any plug-ins on PATH_START. 
	 */
	(void) path_map_exact( PATH_START, "*.plg", 
		(path_map_fn) main_load_plug, NULL );

	/* Link all VIPS functions as SYM_EXTERNAL.
	 */
        (void) im_map_packages( (VSListMap2Fn) main_link_package, NULL );
#endif /*!DEBUG_NOAUTO*/

	/* Load up all defs and wses.
	 */
#ifdef DEBUG
	printf( "definitions init\n" );
#endif/*DEBUG*/
	(void) path_map_exact( PATH_START, "*.def", 
		(path_map_fn) main_load_def, NULL );

#ifdef DEBUG
	printf( "ws init\n" );
#endif/*DEBUG*/
	(void) path_map_exact( PATH_START, "*.ws", 
		(path_map_fn) main_load_ws, NULL );

	mainw_recent_thaw();
}

static void *
main_junk_auto_load( Filemodel *filemodel )
{
	assert( IS_FILEMODEL( filemodel ) );

	if( filemodel->auto_load )
		IDESTROY( filemodel );

	return( NULL );
}

/* Remove and reload all menus/plugins/workspaces.
 */
void
main_reload( void )
{
	set_hourglass();
	program_freeze();

	/* Remove.
	 */
	toolkitgroup_map( main_toolkitgroup, 
		(toolkit_map_fn) main_junk_auto_load, NULL, NULL );
	workspace_map( (workspace_map_fn) main_junk_auto_load, NULL, NULL );
	im_close_plugins();

	/* Reload.
	 */
	main_load_all();

	/* We may have changed our prefs ... link the watches to the
	 * new prefs workspace.
	 */
	watch_relink_all();

	set_pointer();
	program_thaw();
}

/* Use a file to paint a named stock item.
 */
static void
main_file_for_stock( GtkIconFactory *icon_factory,
	const char *stock, const char *file )
{
	GtkIconSource *icon_source;
	GtkIconSet *icon_set;
	char buf[FILENAME_MAX];
	char buf2[FILENAME_MAX];

	im_snprintf( buf2, FILENAME_MAX, 
		"$VIPSHOME/share/$PACKAGE/data/%s", file );
	expand_variables( buf2, buf );
        nativeize_path( buf );
	icon_source = gtk_icon_source_new(); 
	gtk_icon_source_set_filename( icon_source, buf );
	icon_set = gtk_icon_set_new();
	gtk_icon_set_add_source( icon_set, icon_source );
	gtk_icon_source_free( icon_source );
	gtk_icon_factory_add( icon_factory, stock, icon_set );
	gtk_icon_set_unref( icon_set );
}

/* Make our custom icon sets.
 */
static void
main_register_icons( void )
{
	static const GtkStockItem stock_item[] = {
/* Can be (eg.) 
 *
 *   { GTK_STOCK_COPY, N_("_Copy"), GDK_CONTROL_MASK, 'c', GETTEXT_PACKAGE },
 *
 */
		{ STOCK_DROPPER, N_( "Ink dropper" ), 
			0, 0, GETTEXT_PACKAGE },
		{ STOCK_DUPLICATE, N_( "D_uplicate" ), 
			GDK_CONTROL_MASK, 'u', GETTEXT_PACKAGE },
		{ STOCK_PAINTBRUSH, N_( "Pen" ), 
			0, 0, GETTEXT_PACKAGE },
		{ STOCK_LINE, N_( "Line" ), 
			0, 0, GETTEXT_PACKAGE },
		{ STOCK_TEXT, N_( "Text" ), 
			0, 0, GETTEXT_PACKAGE },
		{ STOCK_SMUDGE, N_( "Smudge" ), 
			0, 0, GETTEXT_PACKAGE },
		{ STOCK_FLOOD, N_( "Flood" ), 
			0, 0, GETTEXT_PACKAGE },
		{ STOCK_FLOOD_BLOB, N_( "Flood Blob" ), 
			0, 0, GETTEXT_PACKAGE },
		{ STOCK_RECT, N_( "Fill Rectangle" ), 
			0, 0, GETTEXT_PACKAGE },
		{ STOCK_MOVE, N_( "Pan" ), 
			0, 0, GETTEXT_PACKAGE },
		{ STOCK_SELECT, N_( "Select" ), 
			0, 0, GETTEXT_PACKAGE }
	};

	GtkIconSet *icon_set;

	gtk_stock_add_static( stock_item, IM_NUMBER( stock_item ) );
	main_icon_factory = gtk_icon_factory_new();

	/* Make a colour picker stock ... take the stock icon and add our own
	 * text (gtk defines no text for the standard version of this stock
	 * icon).
	 */
	icon_set = gtk_icon_factory_lookup_default( GTK_STOCK_COLOR_PICKER );
	gtk_icon_factory_add( main_icon_factory, STOCK_DROPPER, icon_set );

	/* For clone, use the DND_MULTIPLE icon (close enough).
	 */
	icon_set = gtk_icon_factory_lookup_default( GTK_STOCK_DND_MULTIPLE );
	gtk_icon_factory_add( main_icon_factory, STOCK_DUPLICATE, icon_set );

	/* Link to our stock .pngs.
	 */
	main_file_for_stock( main_icon_factory, 
		STOCK_PAINTBRUSH, "stock-tool-ink-22.png" );
	main_file_for_stock( main_icon_factory, 
		STOCK_LINE, "stock-tool-path-22.png" );
	main_file_for_stock( main_icon_factory, 
		STOCK_TEXT, "stock-tool-text-22.png" );
	main_file_for_stock( main_icon_factory, 
		STOCK_SMUDGE, "stock-tool-smudge-22.png" );
	main_file_for_stock( main_icon_factory, 
		STOCK_FLOOD, "stock-tool-bucket-fill-22.png" );
	main_file_for_stock( main_icon_factory, 
		STOCK_FLOOD_BLOB, "stock-tool-bucket-fill-22.png" );
	main_file_for_stock( main_icon_factory, 
		STOCK_RECT, "stock-tool-rect-select-22.png" );
	main_file_for_stock( main_icon_factory, 
		STOCK_MOVE, "stock-tool-move-22.png" );
	main_file_for_stock( main_icon_factory, 
		STOCK_SELECT, "stock-tool-select-22.png" );

	gtk_icon_factory_add_default( main_icon_factory );
	g_object_unref( main_icon_factory );
}

/* Init the display connection stuff.
 */
static void
main_x_init( int *argc, char ***argv )
{
	char buf[FILENAME_MAX];

	if( main_window_gdk )
		return;

#ifdef DEBUG
	printf( "X11 init\n" );
#endif/*DEBUG*/

	gtk_set_locale();
	setlocale( LC_NUMERIC, "C" );
	(void) call_string_filename( (call_string_fn) gtk_rc_add_default_file, 
		"$VIPSHOME" IM_DIR_SEP_STR "share" IM_DIR_SEP_STR 
		PACKAGE IM_DIR_SEP_STR "rc" IM_DIR_SEP_STR 
		"ipgtkrc", NULL, NULL, NULL );
	gtk_init( argc, argv );

	/* Set the default icon. gtk+-2.2.x onwards, ought to check for this
	 * really.
	 */
	expand_variables( "$VIPSHOME/share/$PACKAGE/data/vips-128.png", buf );
        nativeize_path( buf );
	gtk_window_set_default_icon_from_file( buf, NULL );

	/* Turn off startup notification. Startup is done when we pop our
	 * first window, not when we make this secret window or display the
	 * splash screen.
	 */
	gtk_window_set_auto_startup_notification( FALSE );

        /* Make invisible top level window ... used to get stuff for
         * build. Realize it, but don't display it.
         */
        main_window_top = iwindow_new( GTK_WINDOW_TOPLEVEL );
        gtk_widget_realize( main_window_top );
	main_window_gdk = main_window_top->window;

#ifdef DEBUG
	printf( "*** debug updates is on\n" );
	gdk_window_set_debug_updates( TRUE );
#endif /*DEBUG*/

	main_register_icons();

	if( main_splash_enabled() ) {
		main_splash = splash_new();
		gtk_widget_show( GTK_WIDGET( main_splash ) );
	}

	/* Next window we make is end of startup.
	 */
	gtk_window_set_auto_startup_notification( TRUE );

	/* Load up any saved accelerators.
	 */
	call_string_filename( (call_string_fn) gtk_accel_map_load,
		IP_SAVEDIR IM_DIR_SEP_STR "accel_map", NULL, NULL, NULL );
}

static void *
main_toobig_done_sub( const char *filename )
{
	unlinkf( "%s", filename );

	return( NULL );
}

/* OK in "flush temps" yesno.
 */
static void
main_toobig_done( iWindow *iwnd, 
	void *client, iWindowNotifyFn nfn, void *sys )
{
	path_map_dir( PATH_TMP, "*",
		(path_map_fn) main_toobig_done_sub, NULL );

	/* Tells space-free indicators to update.
	 */
	if( main_imageinfogroup )
		iobject_changed( IOBJECT( main_imageinfogroup ) );

	nfn( sys, IWINDOW_TRUE );
}

/* Test for a bunch of stuff in the TMP area. Need to do this before
 * we load args in case there are large JPEGs there. Only bother in
 * interactive mode: we won't be able to question the user without an
 * X connection.
 */
static void
main_check_temp( double total )
{
	if( total > 10 * 1024 * 1024 ) {
		char txt[256];
		BufInfo buf;
		char tmp[FILENAME_MAX];

		expand_variables( PATH_TMP, tmp );
		nativeize_path( tmp );
		absoluteize_path( tmp );

		buf_init_static( &buf, txt, 256 );
		to_size( &buf, total );

		box_yesno( NULL,
			main_toobig_done, iwindow_true_cb, NULL,
			NULL, NULL,
			_( "Empty temp area" ),
			_( "Many files in temp area." ),
			_( "The temp area \"%s\" contains %s of files.\n"
			"Would you like to empty the temp area?\n"
			"This will delete any workspace backups and "
			"cannot be undone." ),
			tmp, buf_all( &buf ) );
	}
}

/* Start here!
 */
int
main( int argc, char *argv[] )
{
	/* Flags which trigger the usage message.
	 */
	static const char *usage_flags[] = {
		"-?", "--?", 
		"-help", "--help", 
		"-v", "-version", "--version", "--v"
	};
	
	gboolean welcome_message = FALSE;
	Workspace *ws;
	int i, j;
	double total = 0.0;
#ifdef HAVE_GETRLIMIT
	struct rlimit rlp;
#endif /*HAVE_GETRLIMIT*/
	const char *prefix;
	char name[256];
#if HAVE_FFTW || HAVE_FFTW3
	iOpenFile *of;
#endif /*HAVE_FFTW*/

#ifdef DEBUG_TIME
	static GTimer *startup_timer = NULL;

	if( !startup_timer )
		startup_timer = g_timer_new();

	g_timer_reset( startup_timer );

	/* In startup phase.
	 */
	main_starting = TRUE;

	printf( "%s: startup timer zeroed ...\n", argv[0] );
#endif /*DEBUG_TIME*/

	/* Init object system. Need to call this explicitly in case we don't
	 * start the GUI.
	 */
	g_type_init();

	/* Want numeric locale to be "C", so we have C rules for doing 
	 * double <-> string (ie. no "," for decimal point).
	 */
	setlocale( LC_NUMERIC, "C" );

#ifdef DEBUG
	printf( "main: sizeof( struct heap_node ) == %d\n",
		sizeof( struct heap_node ) );

	/* Should be 3 pointers, hopefully.
	 */
	if( sizeof( struct heap_node ) != 3 * sizeof( void * ) )
		printf( "*** struct packing problem!\n" );
#endif/*DEBUG*/

	/* Yuk .. shouldn't really write to argv0. This can't change the
	 * string length.
	 *
	 * On win32 we will sometimes get paths with mixed '/' and '\' which 
	 * confuses vips's prefix guessing. Make sure we have one or the other.
	 */
	nativeize_path( argv[0] );

	main_argv0 = argv[0];
	main_name = im_skip_dir( main_argv0 );
	main_c_stack_base = &argc;

	/* Pass config.h stuff down to .ws files.
	 */
	putenvf( "PACKAGE=%s", PACKAGE );
	putenvf( "VERSION=%s", VERSION );

#ifdef HAVE_WINDOWS_H
        /* No HOME on windows ... make one from HOMEDRIVE and HOMEDIR (via
         * glib).
         */
        if( !getenv( "HOME" ) )
                putenvf( "HOME=%s", g_get_home_dir() );
#endif /*HAVE_WINDOWS_H*/

	/* Start up vips.
	 */
	if( im_init_world( main_argv0 ) )
		error_exit( "unable to start VIPS" );

	/* Try to discover our data files.
	 */
	if( !(prefix = im_guess_prefix( main_argv0, "VIPSHOME" )) ) 
		error_exit( "unable to guess VIPSHOME" );

	/* Init i18n ... get catalogues from $VIPSHOME/share/locale so we're
	 * relocatable.
	 */
	im_snprintf( name, 256, 
		"%s" IM_DIR_SEP_STR "share" IM_DIR_SEP_STR "locale", prefix );
#ifdef DEBUG
	printf( "bindtextdomain: %s\n", name );
#endif /*DEBUG*/
	bindtextdomain( GETTEXT_PACKAGE, name );
	bind_textdomain_codeset( GETTEXT_PACKAGE, "UTF-8" );
	textdomain( GETTEXT_PACKAGE );

	/* Look for any usage args.
	 */
	for( i = 1; i < argc; i++ )
		for( j = 0; j < IM_NUMBER( usage_flags ); j++ )
			if( strcmp( argv[i], usage_flags[j] ) == 0 ) {
				main_usage();
				exit( 0 );
			}

#ifdef DEBUG_FATAL
	/* Set masks for debugging ... stop on any problem. 
	 */
	g_log_set_always_fatal( 
		G_LOG_FLAG_RECURSION |
		G_LOG_FLAG_FATAL |
		G_LOG_LEVEL_ERROR |
		G_LOG_LEVEL_CRITICAL |
		G_LOG_LEVEL_WARNING );

	printf( "*** DEBUG_FATAL is on ... will abort() on first warning\n" );
#else /*!DEBUG_FATAL*/
#ifdef HAVE_WINDOWS_H 
	/* No logging output ... on win32, log output pops up a very annoying
 	 * console text box.
	 */
	g_log_set_handler( "GLib", 
		G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION, 
		main_log_null, NULL );
	g_log_set_handler( "Gtk", 
		G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION, 
		main_log_null, NULL );
	g_log_set_handler( NULL,
		G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION, 
		main_log_null, NULL );
#endif /*HAVE_WINDOWS_H*/ 
#endif /*DEBUG_FATAL*/

	main_stdin = file_open_read_stdin();

#ifdef HAVE_GETRLIMIT
	/* Make sure we have lots of file descriptors.
	 */
	if( getrlimit( RLIMIT_NOFILE, &rlp ) == 0 ) {
		rlp.rlim_cur = rlp.rlim_max;
		if( setrlimit( RLIMIT_NOFILE, &rlp ) == 0 ) {
#ifdef DEBUG
			printf( "set max file descriptors to %d\n", 
				(int) rlp.rlim_max );
#endif /*DEBUG*/
		}
		else {
			g_warning( _( "unable to change max file "
				"descriptors\n"
				"max file descriptors still set to %d" ),
				(int) rlp.rlim_cur );
		}
	}
	else {
		g_warning( _( "unable to read max file descriptors" ) );
	}
#endif /*HAVE_GETRLIMIT*/

	/* Set default values for paths.
	 */
	path_init();

	/* Do we have a save dir? make one if not.
	 */
	if( !existsf( "%s", IP_SAVEDIR ) ) {
		if( !mkdirf( "%s", IP_SAVEDIR ) || 
			!mkdirf( "%s", IP_SAVEDIR IM_DIR_SEP_STR "tmp" ) || 
			!mkdirf( "%s", IP_SAVEDIR IM_DIR_SEP_STR "start" ) || 
			!mkdirf( "%s", IP_SAVEDIR IM_DIR_SEP_STR "data" ) ) 
			error_exit( _( "unable to make %s area: %s" ),
				IP_SAVEDIR, g_strerror( errno ) );

		/* Can't pop up the welcome message now, wait until we start X.
		 */
		welcome_message = TRUE;
	}

	/* Init other stuff.
	 */
#ifdef HAVE_FFTW3
	fftw_import_system_wisdom();
#endif /*HAVE_FFTW3*/
#if HAVE_FFTW || HAVE_FFTW3
	if( (of = file_open_read( "%s", 
		IP_SAVEDIR IM_DIR_SEP_STR "wisdom" )) ) {
		fftw_import_wisdom_from_file( of->fp );
		file_close( of );
	}
#endif /*HAVE_FFTW*/
	mainw_startup();
	reduce_context = reduce_new();
	main_symbol_root = symbol_root_init();
	g_object_ref( G_OBJECT( main_symbol_root ) );
	iobject_sink( IOBJECT( main_symbol_root ) );
	model_base_init();
	main_workspacegroup = workspacegroup_new( "Workspaces" );
	g_object_ref( G_OBJECT( main_workspacegroup ) );
	iobject_sink( IOBJECT( main_workspacegroup ) );
	main_watchgroup = watchgroup_new( main_workspacegroup, "Preferences" );
	g_object_ref( G_OBJECT( main_watchgroup ) );
	iobject_sink( IOBJECT( main_watchgroup ) );
	main_toolkitgroup = toolkitgroup_new( symbol_root );
	g_object_ref( G_OBJECT( main_toolkitgroup ) );
	iobject_sink( IOBJECT( main_toolkitgroup ) );
	main_imageinfogroup = imageinfogroup_new();
	g_object_ref( G_OBJECT( main_imageinfogroup ) );
	iobject_sink( IOBJECT( main_imageinfogroup ) );

	/* Set IM_CONCURRENCY if a watch changes.
	 */
	g_signal_connect( main_watchgroup, "watch_changed", 
		G_CALLBACK( main_watchgroup_changed_cb ), NULL );

#ifdef DEBUG
	printf( "arg processing\n" );
#endif/*DEBUG*/

	/* Might make this from stdin/whatever if we have a special
	 * command-line flag.
	 */
	ws = NULL;

	/* Look for non-interactive modes first. Just look at argv[1].
	 * Interactive modes may have random extra X or toolkit arguments.
	 */
	if( argc >= 2 ) {
		if( strcmp( argv[1], "-main" ) == 0 ) {
			Toolkit *kit;

			main_interactive = FALSE;
			action_on_exit = EXIT_ACTION_PRINT_MAIN;

			/* Load stdin as a load of defs. Good for HEREIS 
			 * shell scripts.
			 */
			if( !(kit = toolkit_new_from_openfile( 
				main_toolkitgroup, main_stdin )) )
				main_log_add( "%s\n", error_get_sub() );
		}
		else if( strcmp( argv[1], "-workspace" ) == 0 ) {
			/* Load stdin as if it were a workspace. On dispose(),
			 * print the value of the last loaded symbol.
			 */
			if( !(ws = workspace_new_from_openfile( 
				main_workspacegroup, main_stdin )) ) 
				main_log_add( "%s\n", error_get_sub() );
			else {
				filemodel_set_filename( FILEMODEL( ws ), NULL );
				FILEMODEL( ws )->nosave = TRUE;
				ws->print_last = TRUE;
			}
		}
		else if( strcmp( argv[1], "-script" ) == 0 ) {
			action_on_exit = EXIT_ACTION_PRINT_MAIN;
			main_interactive = FALSE;

			if( argc < 3 )
				error( _( "-script needs an argument" ) );
			if( !toolkit_new_from_file( main_toolkitgroup,
				argv[2] ) )
				main_log_add( "%s\n", error_get_sub() );

			/* Shift args. We want argv[0] to be the name of the
			 * script we are interpreting.
			 */
			for( i = 0; i < argc; i++ )
				argv[i] = argv[i + 1];
			argv[0] = argv[1];
			argc -= 1;
		}
		else if( strcmp( argv[1], "-benchmark" ) == 0 ) {
			/* Just start up and shutdown, no X.
			 */
			main_interactive = FALSE;
		}
	}

	/* Make sure we have a start workspace.
	 */
	if( !ws ) {
		workspacegroup_name_new( main_workspacegroup, name );
		ws = workspace_new_blank( main_workspacegroup, name );
	}

#ifdef DEBUG
	if( !main_interactive ) 
		printf( "non-interactive mode\n" );
#endif /*DEBUG*/

	/* Start the X connection. We need this before _load_all(), so that
	 * we can pop up error dialogs.
	 */
	if( main_interactive ) 
		main_x_init( &argc, &argv );

	/* Load start-up stuff.
	 */
	main_load_all();

	/* Recalc to build all classes.
	 */
	main_splash_update( "%s", _( "First recalculation ..." ) );
	symbol_recalculate_all();

	/* Measure amount of stuff in temp area ... need this for checking
	 * temps later.
	 */
	if( main_interactive ) 
		total = directory_size( PATH_TMP );

	if( main_interactive ) {
		/* Load args as files, if we can.
		 */
		for( i = 1; i < argc; i++ )
			if( argv[i][0] != '-' && !main_load( ws, argv[i] ) ) 
				main_log_add( "%s\n", error_get_sub() );
	}
	else {
		Toolkit *kit = toolkit_new( main_toolkitgroup, "args" );
		char txt[MAX_STRSIZE];
		BufInfo buf;

		/* Make argc/argv[].
		 */
		buf_init_static( &buf, txt, MAX_STRSIZE );
		buf_appendf( &buf, "argc = %d;", argc - 1 );
		attach_input_string( buf_all( &buf ) );
		(void) parse_onedef( kit, -1 );

		buf_init_static( &buf, txt, MAX_STRSIZE );
		buf_appendf( &buf, "argv = [ \"%s\"", argv[0] );
		for( i = 2; i < argc; i++ ) 
			buf_appendf( &buf, ", \"%s\"", argv[i] );
		buf_appendf( &buf, " ];" );

		attach_input_string( buf_all( &buf ) );
		if( !parse_onedef( kit, -1 ) ) 
			main_log_add( "%s\n", error_get_sub() );
	}

	/* Recalc to load any args.
	 */
	main_splash_update( "%s", _( "Final recalculation ..." ) );
	symbol_recalculate_all();

	/* Make sure our start ws doesn't have modified set. We may have
	 * loaded some images or whatever into it.
	 */
	filemodel_set_modified( FILEMODEL( ws ), FALSE );

#ifdef DEBUG_TIME
	printf( "%s: main init in %gs\n",  
		main_argv0,
		g_timer_elapsed( startup_timer, NULL ) );
#endif /*DEBUG_TIME*/

	/* Are we running interactively? Start the main window and loop.
	 */
	if( main_interactive ) {
		/* Only display our initial ws if it's not blank, or if it is
		 * blank, if there are no other windows up.
		 */
		if( !workspace_is_empty( ws ) || mainw_number() == 0 ) {
			Mainw *mainw;
			
			main_splash_update( _( "Building main window" ) );
			mainw = mainw_new( ws );
			gtk_widget_show( GTK_WIDGET( mainw ) );
		}

		/* Process a few events ... we want the window to be mapped so
		 * that log/welcome/clean? messages we pop appear in the right
		 * place on the srceen.
		 */
		while( g_main_context_iteration( NULL, FALSE ) )
			;

		if( main_splash ) {
			gtk_widget_destroy( GTK_WIDGET( main_splash ) );
			main_splash = NULL;
		}

		if( !main_log_isempty() ) {
			error_top( _( "Startup error." ) );
			error_sub( _( "Startup error log:\n%s" ), 
				main_log_get() );
			box_alert( NULL );
		}

		if( welcome_message ) {
			char save_dir[FILENAME_MAX];
			char buf[256];

			im_snprintf( buf, 256, 
				_( "Welcome to %s-%s!" ), PACKAGE, VERSION );
			expand_variables( IP_SAVEDIR, save_dir );
			nativeize_path( save_dir );
			box_info( NULL, 
				buf,
_( "A new directory has been created in your home directory to hold startup,\n"
"data and temporary files:\n"
"\n"
"     %s\n"
"\n"
"If you've used previous versions of %s, you will probably want\n"
"to move any files over from your old work area and remove any old temps." ),
				save_dir, PACKAGE );
		}

		/* Offer to junk temps.
		 */
		main_check_temp( total );

#ifdef DEBUG
		printf( "starting event dispatch loop\n" );
#endif/*DEBUG*/

		/* Through startup.
		 */
		main_starting = FALSE;

		gtk_main();
	}

	main_quit();

	return( 0 );
}

#ifdef HAVE_WINDOWS_H 
/* Get non-cmd line args on win32.
 */
static int 
breakargs( char *program, char *line, char **argv ) 
{ 
	int argc = 1; 

	argv[0] = program; 

	while( *line && argc < MAX_SYSTEM - 1 ) { 
		while( *line && isspace( *line ) ) 
			line++; 

		if( *line == '"' ) {
			/* Windows-95 quoted arguments 
			 */ 
			char *start = line + 1; 
			char *end = start; 

			while( *end && *end != '"' ) 
				end++; 

			if( *end == '"' ) { 
				*end = '\0'; 
				argv[argc++] = start; 
				line = end + 1; 
				continue; 
			} 
		} 

		if( *line ) { 
			argv[argc++] = line; 
			while( *line && !isspace( *line ) ) 
				line++; 

			if( *line ) 
				*line++ = '\0'; 
		} 
	} 

	/* add trailing NULL pointer to argv 
	 */ 
	argv[argc] = NULL; 

	return( argc ); 
} 

int WINAPI 
WinMain( HINSTANCE hInstance, HINSTANCE hPrevInstance, 
	LPSTR lpszCmdLine, int nShowCmd ) 
{ 
	char *argv[MAX_SYSTEM];
	int  argc;                                               
	TCHAR program[MAXPATHLEN];                               

	GetModuleFileName( hInstance, program, sizeof(program) );  
	argc = breakargs( (char *) program, lpszCmdLine, argv );    

	return( main( argc, argv ) );
} 
#endif /*HAVE_WINDOWS_H*/ 
