/** 
 * @file libc_syms.c
 * 
 * Resololves symbols we need from libc.
 * 
 * Even when not using NJAMD_PROT=none, NJAMD still needs some symbols from
 * libc to be able to handle pthreads, and to properly install signal
 * handlers in a manner transparent to the user.
 * 
 * Copyright (C) 2000 by Mike Perry.
 * Distributed WITHOUT WARRANTY under the GPL. See COPYING for details.
 */
#include <lib/libc_syms.h>
#include <lib/output.h>
#ifdef HAVE_SANE_DLOPEN
# include <dlfcn.h>
#endif
#include <stdio.h>
#include <stdlib.h>

/**
 * Symbol to handle the case when dlopen is broken
 */
static void bad_dlopen(void)
{
	/* FIXME */
	__nj_output_fatal_internal_error(0, 0, NULL, 0, "dlopen broken on this platform. You can't use PROT=NONE");

	exit(1);
}

/** 
 * Handles initialization of dynamically loaded functions.
 *
 * This function uses dlopen to resolve all the functions we need if we plan
 * to use NJAMD_PROT=none, as well as to keep track of the threads and when
 * they exit (by wrapping exit and pthread_exit)
 *
 * @returns The error string reported by dlerror
 * @see pthread_exit() exit() signal() sigaction()
 */ 
int __nj_libc_syms_init(struct nj_libc_syms *syms)
{

	syms->libc_handle = syms->pthread_handle = NULL;
#ifdef HAVE_SANE_DLOPEN
	TR;
	/* If we have a libpthread, then standard libc functions are threadsafe.
	   If not, then we get the libc functions from the threadsafe lib if we're
	   building threadsafe version, or the standard path if not. Currently we
	   never define _THREAD_SAFE */
# ifdef HAVE_LIBPTHREAD
	if((syms->libc_handle = dlopen(LIBC_SO_PATH, RTLD_LAZY)) == NULL)
		__nj_eprintf("libc.so not found: %s\n", dlerror());
	if((syms->pthread_handle = dlopen(LIBPTHREAD_PATH, RTLD_LAZY)) == NULL)
		__nj_eprintf("libpthread.so not found: %s\n", dlerror());
# elif defined(_THREAD_SAFE) 
	if((syms->libc_handle = dlopen(LIBC_R_PATH, RTLD_LAZY)) == NULL)
		__nj_eprintf("libc_r.so not found: %s\n", dlerror());
# else
	if((syms->libc_handle = dlopen(LIBC_SO_PATH, RTLD_LAZY)) == NULL)
		__nj_eprintf("libc.so not found: %s\n", dlerror());
# endif
#endif
	if(syms->libc_handle && syms->pthread_handle)
		return 0;
	else
		return -1;
}

/** 
 * Resolve a symbol from libc
 *
 * @param syms the libc symbol object
 * @param symbol The symbol string
 * @returns The pointer to the libc symbol
 */
void *__nj_libc_syms_resolve_libc(struct nj_libc_syms *syms, char *symbol)
{
	void *ret = NULL;
#ifdef HAVE_SANE_DLOPEN
	if(syms->libc_handle)
		ret = dlsym(syms->libc_handle, symbol);
#endif

	if(!ret)
		ret = bad_dlopen;

	return ret;
}
/**
 * Resolve pthread symbol
 * 
 * @param syms The symbol source object
 * @param symbol The symbol to resolve
 * @returns The pointer to the libc symbol
 */ 
void *__nj_libc_syms_resolve_pthread(struct nj_libc_syms *syms, char *symbol)
{
	void *ret = NULL;
#ifdef HAVE_SANE_DLOPEN
	if(syms->pthread_handle)
		ret = dlsym(syms->pthread_handle, symbol);
#endif

	if(!ret)
		ret = bad_dlopen;

	return ret;
}

/**
 * Finish using the symbol source
 * 
 * @param syms the symbol source
 */
void __nj_libc_syms_fini(struct nj_libc_syms *syms)
{
#ifdef HAVE_SANE_DLOPEN
	if(syms->libc_handle)
		dlclose(syms->libc_handle);
# ifdef _THREAD_SAFE
	if(syms->pthread_handle)
		dlclose(syms->pthread_handle);
# endif
#endif
}
// vim:ts=4
