/**
 * @file stack.c
 *
 * Stack routines.
 *
 * Simple threadsafe linked list stack that uses externally managed memory.  

 * Copyright (C) 2000 by Mike Perry.
 * Distributed WITHOUT WARRANTY under the GPL. See COPYING for details.
 */
#include <stdio.h>
#include <lib/stack.h>

/**
 * Initialize a new stack.
 *
 * Creates a new stack and initialized the mutex.
 * @param new_stack The stack to init.
 */
void __nj_stack_bootstrap_init(struct nj_stack *new_stack)
{
	new_stack->top = NULL;
}

/**
 * Initialize a new stack for the user
 *
 * Just sets up pthread mutex
 * @param new_stack The stack
 */
void __nj_stack_user_init(struct nj_stack *new_stack)
{
#ifdef _THREAD_SAFE
	pthread_mutex_init(&new_stack->lock, NULL);
#endif
}

/**
 * Finish using the stack
 */
void __nj_stack_fini(struct nj_stack *stack)
{
#ifdef _THREAD_SAFE
	pthread_mutex_destroy(&stack->lock);
#endif
	
}

/**
 * Pops an item off the stack.
 *
 * @param stack The stack to pop from
 * @returns A stack item from the stack itself.
 */
struct nj_stack_item *__nj_stack_pop(struct nj_stack *stack)
{
	struct nj_stack_item *item = NULL;
#ifdef _THREAD_SAFE
	pthread_mutex_lock(&stack->lock);
#endif
	if(stack->top)
	{		
		item = stack->top;
		stack->top = stack->top->next;
	}

#ifdef _THREAD_SAFE
	pthread_mutex_unlock(&stack->lock);
#endif
	return item;
}

/**
 * Push an item onto the stack.
 *
 * @param stack The stack to push onto.
 * @param item A stack item who's storage comes from some renewable pool, 
 * managed completely independant of the stack
 * 
 */ 
void __nj_stack_push(struct nj_stack *stack, struct nj_stack_item *item)
{

#ifdef _THREAD_SAFE
	pthread_mutex_lock(&stack->lock);
#endif

	item->next = stack->top;
	stack->top = item;

#ifdef _THREAD_SAFE
	pthread_mutex_unlock(&stack->lock);
#endif

}

// vim:ts=4
