#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
/* 
  Filename: users.cc
  keeps list of users
*/

#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <pwd.h>
#include <grp.h>
#include <cstdlib>
#include <arc/FileUtils.h>
#include <arc/StringConv.h>
#include <arc/Logger.h>
#include <arc/Utils.h>

#include <string>
#include <list>

#include "../conf/conf.h"
#include "../run/run_parallel.h"
#include "../misc/escaped.h"
#include "../jobs/states.h"
#include "../jobs/job_config.h"
#include "../conf/environment.h"
#include "users.h"

static Arc::Logger& logger = Arc::Logger::getRootLogger();
static std::string empty_string("");

JobUser::JobUser(const GMEnvironment& env):gm_env(env) {
  control_dir="";
  unix_name=""; unix_group=""; home=""; uid=0; gid=0;
  valid=false; jobs=NULL;
  keep_finished=DEFAULT_KEEP_FINISHED;
  keep_deleted=DEFAULT_KEEP_DELETED;
  cred_plugin=NULL;
  strict_session=false;
  fixdir=fixdir_always;
  share_uid=0;
  reruns = 0;
  diskspace = 0;
}

void JobUser::SetHeadNode(const std::string &head_node) {
  headnode=head_node;
}

void JobUser::SetLRMS(const std::string &lrms_name,const std::string &queue_name) {
  default_lrms=lrms_name;
  default_queue=queue_name;
}

void JobUser::SetControlDir(const std::string &dir) {
  if(dir.length() == 0) {
    control_dir=home + "/.jobstatus";
  }
  else { control_dir=dir; };
}

void JobUser::SetSessionRoot(const std::string &dir) {
  session_roots.clear();
  if(dir.length() == 0 || dir == "*") { session_roots.push_back(home + "/.jobs"); }
  else { session_roots.push_back(dir); };
}

void JobUser::SetSessionRoot(const std::vector<std::string> &dirs) {
  session_roots.clear();
  if (dirs.empty()) {
    std::string dir;
    SetSessionRoot(dir);
  } else {
    for (std::vector<std::string>::const_iterator i = dirs.begin(); i != dirs.end(); i++) {
      if (*i == "*") session_roots.push_back(home + "/.jobs");
      else session_roots.push_back(*i);
    }
  }
}

const std::string & JobUser::SessionRoot(std::string job_id) const {
  if (session_roots.empty()) return empty_string;
  if (session_roots.size() == 1 || job_id.empty()) return session_roots[0];
  // search for this jobid's session dir
  struct stat st;
  for (std::vector<std::string>::const_iterator i = session_roots.begin(); i != session_roots.end(); i++) {
    std::string sessiondir(*i + '/' + job_id);
    if (stat(sessiondir.c_str(), &st) == 0 && S_ISDIR(st.st_mode))
      return *i;
  }
  return empty_string; // not found
}

void JobUser::SetCacheParams(CacheConfig params) {
  std::vector<std::string> cache_dirs = params.getCacheDirs();
  for (std::vector<std::string>::iterator i = cache_dirs.begin(); i != cache_dirs.end(); i++) {
    substitute(*i);
  }
  params.setCacheDirs(cache_dirs);
  std::vector<std::string> drain_cache_dirs = params.getDrainingCacheDirs();
  for (std::vector<std::string>::iterator i = drain_cache_dirs.begin(); i != drain_cache_dirs.end(); i++) {
    substitute(*i);
  }
  params.setDrainingCacheDirs(drain_cache_dirs);
  cache_params = params;
}

static bool fix_directory(const std::string& path, JobUser::fixdir_t fixmode, mode_t mode, uid_t uid, gid_t gid) {
  if(fixmode == JobUser::fixdir_never) {
    struct stat st;
    if(!Arc::FileStat(path,&st,true)) return false;
    if(!S_ISDIR(st.st_mode)) return false;
    return true;
  } else if(fixmode == JobUser::fixdir_missing) {
    struct stat st;
    if(Arc::FileStat(path,&st,true)) {
      if(!S_ISDIR(st.st_mode)) return false;
      return true;
    };
  };
  // JobUser::fixdir_always
  if(!Arc::DirCreate(path,mode,true)) return false;
  // Only can switch owner if running as root
  if(::getuid() == 0) (::chown(path.c_str(),uid,gid) != 0);
  (::chmod(path.c_str(),mode) != 0);
  return true;
}

bool JobUser::CreateDirectories(void) {
  bool res = true;
  if(!control_dir.empty()) {
    mode_t mode = 0;
    if((uid == 0) && (getuid() == 0)) {
      // This control dir serves multiple users and running
      // as root (hence really can serve multiple users)
      mode = S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;
    } else {
      mode = S_IRWXU;
    };
    if(!fix_directory(control_dir,fixdir,mode,uid,gid)) res = false;
    // Structure inside control dir is important - *always* create it
    // Directories containing logs and job states may need access from 
    // information system, etc. So allowing them to be more open.
    // Delegation is only accessed by service itself.
    if(!fix_directory(control_dir+"/logs",fixdir_always,mode,uid,gid)) res = false;
    if(!fix_directory(control_dir+"/accepting",fixdir_always,mode,uid,gid)) res = false;
    if(!fix_directory(control_dir+"/restarting",fixdir_always,mode,uid,gid)) res = false;
    if(!fix_directory(control_dir+"/processing",fixdir_always,mode,uid,gid)) res = false;
    if(!fix_directory(control_dir+"/finished",fixdir_always,mode,uid,gid)) res = false;
    std::string deleg_dir = DelegationDir();
    if(!fix_directory(deleg_dir,fixdir_always,S_IRWXU,uid,gid)) res = false;
  };
  for(std::vector<std::string>::iterator i = session_roots.begin(); i != session_roots.end(); i++) {
    mode_t mode = 0;
    if((uid == 0) && (getuid() == 0)) {
      if(strict_session) {
        // For multiple users creating immediate subdirs using own account
        // dangerous permissions, but there is no other option
        mode = S_IRWXU | S_IRWXG | S_IRWXO | S_ISVTX;
      } else {
        // For multiple users not creating immediate subdirs using own account
        mode = S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;
      }
    } else {
      // For single user
      mode = S_IRWXU;
    };
    if(!fix_directory(*i,fixdir,mode,uid,gid)) res = false;
  };
  return res;
}

std::string JobUser::DelegationDir(void) const {
  std::string deleg_dir = control_dir+"/delegations";
  uid_t u = ::getuid();
  if(u == 0) return deleg_dir;
  struct passwd pwbuf;
  char buf[4096];
  struct passwd* pw;
  if(::getpwuid_r(u,&pwbuf,buf,sizeof(buf),&pw) == 0) {
    if(pw && pw->pw_name) {
      deleg_dir+=".";
      deleg_dir+=pw->pw_name;
    };
  };
  return deleg_dir;
}

/*
  %R - session root
  %r - list of session roots
  %C - control dir
  %c - list of control dirs
  %U - username
  %u - userid
  %g - groupid
  %H - home dir
  %Q - default queue
  %L - default lrms
  %W - installation path
  %G - globus path
  %F - configuration file
*/

bool JobUser::substitute(std::string& param) const {
  std::string::size_type curpos=0;
  for(;;) {
    if(curpos >= param.length()) break;
    std::string::size_type pos = param.find('%',curpos);
    if(pos == std::string::npos) break;
    pos++; if(pos>=param.length()) break;
    if(param[pos] == '%') { curpos=pos+1; continue; };
    std::string to_put;
    switch(param[pos]) {
      case 'R': to_put=SessionRoot(); break;
      case 'C': to_put=ControlDir(); break;
      case 'U': to_put=UnixName(); break;
      case 'H': to_put=Home(); break;
      case 'Q': to_put=DefaultQueue(); break;
      case 'L': to_put=DefaultLRMS(); break;
      case 'u': to_put=Arc::tostring(get_uid()); break;
      case 'g': to_put=Arc::tostring(get_gid()); break;
      case 'W': to_put=gm_env.nordugrid_loc(); break;
      case 'F': to_put=gm_env.nordugrid_config_loc(); break;
      case 'G': 
        logger.msg(Arc::ERROR,"Globus location variable substitution is not supported anymore. Please specify path directly.");
        break;
      default: to_put=param.substr(pos-1,2);
    };
    curpos=pos+1+(to_put.length() - 2);
    param.replace(pos-1,2,to_put);
  };
  return true;
}

bool JobUsers::substitute(std::string& param) const {
  std::string session_roots = "";
  std::string control_dirs = "";
  for(JobUsers::const_iterator i = begin();i!=end();++i) {
    std::string tmp_s;
    tmp_s = i->SessionRoot();
    tmp_s = Arc::escape_chars(tmp_s, " \\", '\\', false);
    tmp_s=tmp_s+" ";
    if(session_roots.find(tmp_s) == std::string::npos) session_roots+=tmp_s;
    tmp_s = i->ControlDir();
    tmp_s = Arc::escape_chars(tmp_s, " \\", '\\', false);
    tmp_s=tmp_s+" ";
    if(control_dirs.find(tmp_s) == std::string::npos) control_dirs+=tmp_s;
  };
  std::string::size_type curpos=0;
  for(;;) {
    if(curpos >= param.length()) break;
    std::string::size_type pos = param.find('%',curpos);
    if(pos == std::string::npos) break;
    pos++; if(pos>=param.length()) break;
    if(param[pos] == '%') { curpos=pos+1; continue; };
    std::string to_put;
    switch(param[pos]) {
      case 'r': to_put=session_roots; break;
      case 'c': to_put=control_dirs; break;
      default: to_put=param.substr(pos-1,2);
    };
    curpos=pos+1+(to_put.length() - 2);
    param.replace(pos-1,2,to_put);
  };
  return true;
}

JobUser::JobUser(const GMEnvironment& env,uid_t uid_,gid_t gid_,RunPlugin* cred):gm_env(env) {
  struct passwd pw_;
  struct passwd *pw;
  char buf[BUFSIZ];
  uid=uid_;
  gid=gid_;
  valid=false;
  cred_plugin=cred;
  // resolve name
  if(uid_ == 0) {
    unix_name="";  
    gid=0;
    home="/tmp";
    valid=true;
  }
  else {
    getpwuid_r(uid_,&pw_,buf,BUFSIZ,&pw);
    if(pw != NULL) {
      unix_name=pw->pw_name;  
      if(gid_ == 0) gid=pw->pw_gid;
      home=pw->pw_dir;
      valid=true;
    };
  };
  jobs=NULL;
  SetControlDir("");
  SetSessionRoot("");
  SetLRMS("","");
  keep_finished=DEFAULT_KEEP_FINISHED;
  keep_deleted=DEFAULT_KEEP_DELETED;
  strict_session=false;
  fixdir=fixdir_always;
  share_uid=0;
  reruns = 0;
  diskspace = 0;
}

JobUser::JobUser(const GMEnvironment& env,const std::string &u_name,RunPlugin* cred):gm_env(env) {
  char buf[BUFSIZ];
  unix_name=u_name;  
  std::string::size_type p = unix_name.find(':');
  if(p != std::string::npos) {
    unix_group = unix_name.substr(p+1);
    unix_name.resize(p);
  };
  cred_plugin=cred;
  valid=false;
  /* resolve name */
  if(unix_name.length() == 0) {
    uid=0;  
    gid=0;
    home="/tmp";
    valid=true;
  }
  else {
    struct passwd pw_;
    struct passwd *pw = NULL;
    getpwnam_r(unix_name.c_str(),&pw_,buf,BUFSIZ,&pw);
    if(pw != NULL) {
      uid=pw->pw_uid;
      gid=pw->pw_gid;
      home=pw->pw_dir;
      valid=true;
      if(!unix_group.empty()) {
        struct group gr_;
        struct group *gr = NULL;
        getgrnam_r(unix_group.c_str(),&gr_,buf,BUFSIZ,&gr);
        if(gr != NULL) {
          gid=gr->gr_gid;
        };
      };
    };
  };
  SetControlDir("");
  SetSessionRoot("");
  SetLRMS("","");
  jobs=NULL;
  keep_finished=DEFAULT_KEEP_FINISHED;
  keep_deleted=DEFAULT_KEEP_DELETED;
  strict_session=false;
  fixdir=fixdir_always;
  share_uid=0;
  diskspace=0;
  reruns = 0;
}

JobUser::JobUser(const JobUser &user):gm_env(user.gm_env) {
  uid=user.uid; gid=user.gid;
  unix_name=user.unix_name;  
  unix_group=user.unix_group;  
  control_dir=user.control_dir;
  home=user.home;
  jobs=user.jobs;
  session_roots=user.session_roots;
  default_lrms=user.default_lrms;
  default_queue=user.default_queue;
  valid=user.valid;
  keep_finished=user.keep_finished;
  keep_deleted=user.keep_deleted;
  cache_params=user.cache_params;
  cred_plugin=user.cred_plugin;
  strict_session=user.strict_session;
  fixdir=user.fixdir;
  share_uid=user.share_uid;
  share_gids=user.share_gids;
  diskspace=user.diskspace;
  reruns=user.reruns;
  helpers=user.helpers;
}

JobUser::~JobUser(void) { 
}

void JobUser::SetShareID(uid_t suid) {
  share_uid=suid;
  share_gids.clear();
  if(suid <= 0) return;
  struct passwd pwd_buf;
  struct passwd* pwd = NULL;
#ifdef _SC_GETPW_R_SIZE_MAX
  int buflen = sysconf(_SC_GETPW_R_SIZE_MAX);
  if (buflen <= 0) buflen = 16384;
#else
  int buflen = 16384;
#endif
  char* buf = (char*)malloc(buflen);
  if(!buf) return;
  if(getpwuid_r(suid, &pwd_buf, buf, buflen, &pwd) == 0) {
    if(pwd) {
#ifdef HAVE_GETGROUPLIST
#ifdef _MACOSX
      int groups[100];
#else
      gid_t groups[100];
#endif
      int ngroups = 100;
      if(getgrouplist(pwd->pw_name, pwd->pw_gid, groups, &ngroups) >= 0) {
        for(int n = 0; n<ngroups;++n) {
          share_gids.push_back((gid_t)(groups[n]));
        };
      };
#endif
      share_gids.push_back(pwd->pw_gid);
    };
  };
  free(buf);
}

bool JobUser::match_share_gid(gid_t sgid) const {
  for(std::list<gid_t>::const_iterator i = share_gids.begin();
                       i != share_gids.end();++i) {
    if(sgid == *i) return true;
  };
  return false;
}

JobUsers::JobUsers(GMEnvironment& env):gm_env(env) {
}

JobUsers::~JobUsers(void) {
}

JobUsers::iterator JobUsers::AddUser(const std::string &unix_name,RunPlugin* cred_plugin,const std::string &control_dir, const std::vector<std::string> *session_roots) {
  JobUser user(gm_env,unix_name,cred_plugin);
  user.SetControlDir(control_dir);
  if(session_roots) user.SetSessionRoot(*session_roots);
  if(user.is_valid()) { return users.insert(users.end(),user); };
  return users.end();
}

std::string JobUsers::ControlDir(iterator user) {
  if(user == users.end()) return std::string("");
  return (*user).ControlDir();
}

JobUsers::iterator JobUsers::find(const std::string& user) {
  iterator i;
  for(i=users.begin();i!=users.end();++i) {
    if((*i) == user) break;
  };
  return i;
}

std::string JobUsers::ControlDir(const std::string& user) {
  for(iterator i=users.begin();i!=users.end();++i) {
    if((*i) == user) return (*i).ControlDir();
  };
  return std::string("");
}

#ifndef NO_GLOBUS_CODE

/* Change effective user - real switch is done only if running as root.
   This method is caled in single-threaded environment and in 
   post-multi-threaded right after fork. So it must avoid 
   using anything that could internally inlcude calls to thread 
   functions. */
bool JobUser::SwitchUser(bool su) const {
  //std::string uid_s = Arc::tostring(uid);
  //if(!Arc::SetEnv("USER_ID",uid_s)) if(!su) return false;
  //if(!Arc::SetEnv("USER_NAME",unix_name)) if(!su) return false;

  static char uid_s[64];
  static char gid_s[64];
  snprintf(uid_s,63,"%llu",(long long unsigned int)uid);
  snprintf(gid_s,63,"%llu",(long long unsigned int)gid);
  uid_s[63]=0;
  gid_s[63]=0;
#ifdef HAVE_SETENV
  if(setenv("USER_ID",uid_s,1) != 0) if(!su) return false;
  if(setenv("USER_GID",gid_s,1) != 0) if(!su) return false;
  if(setenv("USER_NAME",unix_name.c_str(),1) != 0) if(!su) return false;
  if(setenv("USER_GROUP",unix_group.c_str(),1) != 0) if(!su) return false;
#else
  static char user_id_s[64];
  static char user_gid_s[64];
  static char user_name_s[64];
  static char user_group_s[64];
  strncpy(user_id_s,"USER_ID",63); user_id_s[63]=0;
  strncat(user_id_s,uid_s,63-strlen(user_id_s)); user_id_s[63]=0;
  strncpy(user_gid_s,"USER_GID",63); user_gid_s[63]=0;
  strncat(user_gid_s,gid_s,63-strlen(user_gid_s)); user_gid_s[63]=0;
  strncpy(user_name_s,"USER_NAME",63); user_name_s[63]=0;
  strncat(user_name_s,unix_name.c_str(),63-strlen(user_name_s)); user_name_s[63]=0;
  strncpy(user_group_s,"USER_GROUP",63); user_group_s[63]=0;
  strncat(user_group_s,unix_group.c_str(),63-strlen(user_group_s)); user_group_s[63]=0;
  if(putenv(user_id_s) != 0) if(!su) return false;
  if(putenv(user_gid_s) != 0) if(!su) return false;
  if(putenv(user_name_s) != 0) if(!su) return false;
  if(putenv(user_group_s) != 0) if(!su) return false;
#endif

  /* set proper umask */
  umask(0077);
  if(!su) return true;
  uid_t cuid;
  if(((cuid=getuid()) != 0) && (uid != 0)) {
    if(cuid != uid) return false;
  };
  if(uid != 0) {
    setgid(gid); /* this is not an error if group failed, not a big deal */
    if(setuid(uid) != 0) return false;
  };
  return true;
}

bool JobUser::run_helpers(void) {
//  if(unix_name.length() == 0) { /* special users can not run helpers */
//    return true;  
//  };
  bool started = true;
  for(std::list<JobUserHelper>::iterator i=helpers.begin();i!=helpers.end();++i) {
    started &= i->run(*this);
  };
  return started;
}

void JobUser::PrepareToDestroy() {
  // TODO: send signals to helpers and stop started threads
}

bool JobUsers::run_helpers(void) {
  for(iterator i=users.begin();i!=users.end();++i) {
    i->run_helpers();
  };
  return true;
}
#endif // NO_GLOBUS_CODE

JobUserHelper::JobUserHelper(const std::string &cmd) {
  command=cmd;
  proc=NULL;
}
  
JobUserHelper::~JobUserHelper(void) {
#ifndef NO_GLOBUS_CODE
    if(proc != NULL) {
      delete proc;
      proc=NULL;
    };
#endif
}

#ifndef NO_GLOBUS_CODE
bool JobUserHelper::run(JobUser &user) {
    if(proc != NULL) {
      if(proc->Running()) {
        return true; /* it is already/still running */
      };
      delete proc;
      proc=NULL;
    };
    /* start/restart */
    if(command.length() == 0) return true;  /* has anything to run ? */
    char* args[100]; /* up to 98 arguments should be enough */
    std::string args_s = command;
    std::string arg_s;
    int n;
    for(n=0;n<99;n++) {
      arg_s=config_next_arg(args_s);
      if(arg_s.length() == 0) break;
      args[n]=strdup(arg_s.c_str());
    };
    args[n]=NULL;
    logger.msg(Arc::VERBOSE,"Starting helper process (%s): %s",
               user.UnixName().c_str(),command.c_str());
    std::string helper_id="helper."+user.UnixName();
    bool started=RunParallel::run(user,helper_id.c_str(),args,&proc);
    for(n=0;n<99;n++) {
      if(args[n] == NULL) break;
      free(args[n]);
    };
    if(started) return true;
    if(proc && (*proc)) return true;
    if(proc) { delete proc; proc=NULL; };
    logger.msg(Arc::ERROR,"Helper process start failed (%s): %s",
               user.UnixName().c_str(),command.c_str());
    /* start failed */
    /* doing nothing - maybe in the future */
    return false;
}
#endif

