/*
 *  Copyright (C) 2004 Mathias Andre <mathias@openbrookes.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>
#include <glade/glade.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>

#include "callbacks.h"
#include "interface.h"
#include "../nb_global.h"

#define _(text) gettext(text)

void
on_new_activate	(GtkMenuItem *menuitem, gpointer user_data)
{
	extern global * g;
	nb_global_new_note (g);
}


void
on_save_activate	(GtkMenuItem *menuitem, gpointer user_data)
{
	extern global * g;
	nb_global_update_note (g);
}


void
on_delete_activate	(GtkMenuItem *menuitem, gpointer user_data)
{
	extern global * g;
	nb_global_delete_note (g);
}


void
on_import_activate	(GtkMenuItem *menuitem, gpointer user_data)
{
	import_notes_from_file ();
}


void
on_export_activate	(GtkMenuItem *menuitem, gpointer user_data)
{
	export_notes_to_file ();
}


void
on_about_activate	(GtkMenuItem *menuitem, gpointer user_data)
{
	create_about_dialog ();
}

void
on_optionmenutype_note_activate	(GtkMenuItem *menuitem, gpointer user_data)
{
	extern global * g;
	nb_note_set_type (g->current_note, NOTE);
}


void
on_optionmenutype_personal_activate	(GtkMenuItem *menuitem, gpointer user_data)
{
	extern global * g;
	nb_note_set_type (g->current_note, PERSONAL);

}


void
on_optionmenutype_problem_activate	(GtkMenuItem *menuitem, gpointer user_data)
{
	extern global * g;
	nb_note_set_type (g->current_note, PROBLEM);

}


void
on_optionmenutype_trick_activate	(GtkMenuItem *menuitem, gpointer user_data)
{
	extern global * g;
	nb_note_set_type (g->current_note, TRICK);

}


void
on_optionmenutype_bookmark_activate	(GtkMenuItem *menuitem, gpointer user_data)
{
	extern global * g;
	nb_note_set_type (g->current_note, BOOKMARK);

}


void
on_optionmenustatus_none_activate	(GtkMenuItem *menuitem, gpointer user_data)
{
	extern global * g;
	nb_note_set_status (g->current_note, NONE);

}


void
on_optionmenustatus_open_activate	(GtkMenuItem *menuitem, gpointer user_data)
{
	extern global * g;
	nb_note_set_status (g->current_note, OPEN);

}


void
on_optionmenustatus_solved_activate	(GtkMenuItem *menuitem, gpointer user_data)
{
	extern global * g;
	nb_note_set_status (g->current_note, SOLVED);

}


void
on_note_edited	(GtkEditable *editable, gpointer user_data)
{
	extern global * g;
	g->modified_note = EDITED;
}

void
tree_selection_changed	(GtkTreeSelection *selection, gpointer user_data)
{
	extern global * g;
	int count;

	count = gtk_tree_selection_count_selected_rows (selection);

	if ( count > 1 )
		return;

	if ( g->modified_note == EDITED )
	{
		if ( (g->inserted_note == INSERTED) || (is_empty_note () == 0) )
		{
			if ( !nb_global_save_note (g) )
				return;
			else
				push_statusbar ((gchar *) _("Successfully saved note to file."));
		}

		nb_note_clear_text (g->current_note);
	}

	gtk_tree_selection_selected_foreach (selection, (GtkTreeSelectionForeachFunc) update_fields_with_selection, NULL);

	{
		set_fields_value (g->current_note->title,
				  g->current_note->text,
				  g->current_note->type,
				  g->current_note->status,
				  nb_note_get_string_from_date (g->current_note->date));

		g->inserted_note = INSERTED;
		g->modified_note = CLEAN;
	}
}

void
update_fields_with_selection (GtkTreeModel *model,
			      GtkTreePath  *path,
			      GtkTreeIter  *iter,
			      gpointer      user_data)
{
	extern global * g;
	int index;

	gtk_tree_model_get (model, iter, NOTE_POINTER_COLUMN, &(g->current_note), -1);

	index = nb_notelist_get_note_index (g->nl, g->current_note);

	if ( index == -1 )
	{
		nb_note_set_text (g->current_note, NULL);
		fprintf (stderr, "Invalid note! Note is not present in the array!\n");
	}
	else
		nb_note_set_text (g->current_note, nb_notelist_get_text_from_index (g->nl, index));
}

void
on_search_edited	(GtkEntry *selection, gpointer user_data)
{
	update_tree_with_result ((char *) gtk_entry_get_text (selection));
}

void
on_clear_search_clicked	(GtkButton * selection, gpointer user_data)
{
	extern GladeXML * xml;
	GtkWidget * entryfind = glade_xml_get_widget (xml, "entryfind");

	gtk_entry_set_text ( GTK_ENTRY (entryfind), "");
}

void
export_notes_validate	(GtkWidget * widget, gpointer user_data)
{
	extern GladeXML * xmlexport;
	extern global * g;
	GtkWidget * exportdialog, * filechooser;
	GtkWidget * check_export_note, * check_export_personal, * check_export_problem, * check_export_trick, * check_export_bookmark;
	exporttype etype;

	exportdialog = glade_xml_get_widget (xmlexport, "exportdialog");
	filechooser = g_object_get_data ( G_OBJECT (exportdialog), "filechooser");
	check_export_note = glade_xml_get_widget (xmlexport, "check_export_note");
	check_export_personal = glade_xml_get_widget (xmlexport, "check_export_personal");
	check_export_problem = glade_xml_get_widget (xmlexport, "check_export_problem");
	check_export_trick = glade_xml_get_widget (xmlexport, "check_export_trick");
	check_export_bookmark = glade_xml_get_widget (xmlexport, "check_export_bookmark");

	/* get the toggle buttons status */
	etype.note = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON (check_export_note));
	etype.personal = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON (check_export_personal));
	etype.problem = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON (check_export_problem));
	etype.trick = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON (check_export_trick));
	etype.bookmark = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON (check_export_bookmark));

	nb_global_export_notelist (g, &etype, gtk_file_chooser_get_filename ( GTK_FILE_CHOOSER (filechooser)));

	gtk_widget_destroy (exportdialog);
}

void
on_link_activate (GtkTextView * textview, GdkEvent * event)
{
	GtkTextIter start, end;
	GtkTextBuffer *buffer;
	GdkEventButton *ev;
	GtkTextTagTable * table;
	GtkTextTag * tag;
	char c;
	gint x, y;

	if ( event->type != GDK_BUTTON_RELEASE )
		return ;	

	 ev= (GdkEventButton *)event;

	if ( ev->button!= 1)
		return ;

	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview));

	/* don't follow link if user selected some text */
	gtk_text_buffer_get_selection_bounds (buffer, &start, &end);
	if ( gtk_text_iter_get_offset (&start) != gtk_text_iter_get_offset (&end) )
		return ;
	
	gtk_text_view_window_to_buffer_coords (GTK_TEXT_VIEW (textview), GTK_TEXT_WINDOW_WIDGET, ev->x, ev->y, &x, &y);
	gtk_text_view_get_iter_at_location (GTK_TEXT_VIEW (textview), &start, x, y);

	/* check if it has a link tag */
	/* TODO: we may do a text_iter_has_tag_by_name as we also do the check for cursor change in interface.c */
	table = gtk_text_buffer_get_tag_table (buffer);
	tag = gtk_text_tag_table_lookup (table, "links");
	
	if ( !gtk_text_iter_has_tag (&start, tag) )
		return ;

	do
	{
		gtk_text_iter_backward_char (&start);
		c = gtk_text_iter_get_char (&start); 
	}
	while ( (c != ' ')  && (c != 0) && (!gtk_text_iter_starts_line (&start)) );

	if ( !gtk_text_iter_starts_line (&start) )
		gtk_text_iter_forward_char (&start);

	end = start;

	/* look for the end of the link */
	do
	{
		gtk_text_iter_forward_char (&end);
		c = gtk_text_iter_get_char (&end);
	}
	while ( (c != EOF) && (c != 0) && (c != ' ') && (c != '\n') );

	start_browser (gtk_text_buffer_get_text (buffer, &start, &end, FALSE));
}

gboolean 
on_main_quit (GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
	extern global * g;

	if ( !nb_global_quit (g) )
		return;

	store_interface_positions ();

	gtk_main_quit ();
	
	return TRUE;
}

void 
start_browser (char * url)
{
	extern global * g;
	gchar ** split;
	gchar * command;

	if ( fork() == 0 )
	{
		if (  g_strrstr (g->default_browser, "%s") != NULL )
		{
			split = g_strsplit (g->default_browser, "%s", 2);
			command = g_strdup_printf ("%s%s%s", split[0], url, split[1]);
			g_strfreev(split);

			split = g_strsplit (command, " ", 0);
			g_strstrip(split[0]);

			execvp (split[0], split);
			perror ("exec: ");
			exit (1);
		}
		else
		{
			execlp (g->default_browser, g->default_browser, url, NULL);
			perror ("exec: ");
			exit (1);
		}
	}
}

void 
scan_url_tag_from_textview (GtkTextBuffer * buffer, gpointer user_data)
{
	GtkTextIter start;
	GtkTextIter end;

	gtk_text_buffer_get_start_iter (buffer,&start);

	while ( gtk_text_iter_forward_search (&start, "http://", GTK_TEXT_SEARCH_TEXT_ONLY, &start, &end, NULL) )
	{
		char c;

		/* look for the end of the link */
		do 
		{
			gtk_text_iter_forward_char (&end);
			c = gtk_text_iter_get_char (&end);
		}
		while ( (c != EOF) && (c != 0) && (c != ' ') && (c != '\n') );

		/* set the links tag */
		gtk_text_buffer_apply_tag_by_name (buffer, "links", &start, &end);

		start = end;
	}

	on_note_edited	(NULL, NULL);
}

gboolean 
motion_notify_event (GtkWidget * textview, GdkEventMotion *event)
{
	gint x, y;
 	
	gtk_text_view_window_to_buffer_coords (GTK_TEXT_VIEW (textview), 
					       GTK_TEXT_WINDOW_WIDGET,
					       event->x, event->y, &x, &y);

	set_cursor_if_appropriate (GTK_TEXT_VIEW (textview), x, y);

	gdk_window_get_pointer (textview->window, NULL, NULL, NULL);

	/* propagate the event */
	return FALSE;
}

gboolean
visibility_notify_event (GtkWidget * textview, GdkEventVisibility * event)
{
	  gint wx, wy, bx, by;

	  gdk_window_get_pointer (textview->window, &wx, &wy, NULL);

	  gtk_text_view_window_to_buffer_coords (GTK_TEXT_VIEW (textview), 
			  GTK_TEXT_WINDOW_WIDGET,
			  wx, wy, &bx, &by);

	  set_cursor_if_appropriate (GTK_TEXT_VIEW (textview), bx, by);

	/* propagate the event */
	  return FALSE;
}

gint 
sort_iter_compare_string	(GtkTreeModel * model, GtkTreeIter * a, GtkTreeIter * b, gpointer userdata)
{
	gint column = GPOINTER_TO_INT (userdata);
	gchar * val1, * val2;
	gint ret;

	switch (column)
	{
		case TITLE_COLUMN:
		{
			gtk_tree_model_get (model, a, TITLE_COLUMN, &val1, -1);
			gtk_tree_model_get (model, b, TITLE_COLUMN, &val2, -1);
			break;
		}

		case TYPE_COLUMN:
		{
			gtk_tree_model_get (model, a, TYPE_COLUMN, &val1, -1);
			gtk_tree_model_get (model, b, TYPE_COLUMN, &val2, -1);
			break;
		}

		case STATUS_COLUMN:
		{
			gtk_tree_model_get (model, a, STATUS_COLUMN, &val1, -1);
			gtk_tree_model_get (model, b, STATUS_COLUMN, &val2, -1);
			break;
		}
	}

	if ( (val1 == NULL) || (val2 == NULL) )
	{
		if ( val1 == NULL )
		{
			if ( val2 == NULL )
				ret = 0;
			else 
				ret = -1;
		}
		else
			ret = 1;
	}
	else
	{
		ret = g_utf8_collate (val1,val2);
	}

	g_free (val1);
	g_free (val2);

	return ret;
}

gint 
sort_iter_compare_date	(GtkTreeModel * model, GtkTreeIter * a, GtkTreeIter * b, gpointer userdata)
{
	gint column = GPOINTER_TO_INT (userdata);
	note * n1, * n2;
	gint ret;

	gtk_tree_model_get (model, a, NOTE_POINTER_COLUMN, &n1, -1);
	gtk_tree_model_get (model, b, NOTE_POINTER_COLUMN, &n2, -1);

	if ( column == DATE_COLUMN )
		ret = g_date_compare (n1->date, n2->date);
	else
		ret = g_date_compare (n1->lastUpdate, n2->lastUpdate);

	return ret;
}

void
on_drag_received	(GtkWidget * wgt, GdkDragContext * context, int x, int y, GtkSelectionData *seldata, guint info, guint time, gpointer userdata)
{
	printf ("drag and drop:%s\n", (gchar *)seldata->data);
}
