////////////////////////////////////////////////////////////////////////////
// NoteCase notes manager project <http://notecase.sf.net>
//
// This code is licensed under BSD license.See "license.txt" for more details.
//
// File: Implements NoteCase HTML format (and NoteCenter format) I/O
////////////////////////////////////////////////////////////////////////////

#include "FormatHtml.h"
#include "File64.h"
#include "DocumentIterator.h"
#include <stdio.h>	//sprintf
#ifndef _WIN32
 #include <strings.h>	//strcasecmp
#else
 #define strcasecmp stricmp
#endif
#include "debug.h"

int InternalIcon_Name2Index(const char *szName);
const char *InternalIcon_Index2Name(int nIndex);

#define  DOC_PARSE_BLANK		0
#define  DOC_PARSE_EXPECT_TITLE	1
#define  DOC_PARSE_EXPECT_TEXT	2

void replaceall(std::string &strData, const char *szFind, const char *szReplace);
void WriteLevel(DocumentIterator &it, File64 &file, int nParentID);

FormatHTML::FormatHTML()
{
	m_nCurParentID = -1;
	m_nCurNodeID = -1;
	m_nDocParseState = DOC_PARSE_BLANK;
	m_bNoteCenterMode = false;
	m_bHtmlExport = false;
	m_nTagsParsedCount = 0;
}

FormatHTML::~FormatHTML()
{
}

int FormatHTML::Load(const char *szFile, NoteDocument &doc)
{
	m_objDoc = &doc;  //store pointer

	//clear old document contents
	doc.Clear();

	File64 objFile;	
	if(!objFile.Open(szFile, F64_READ|F64_SHARE_READ|F64_OPEN_EXISTING))
		return DOC_LOAD_NOT_FOUND;

	char szBuffer[1024];
	int nRead;

	//TOFIX better error handling
	while((nRead = objFile.Read(szBuffer, sizeof(szBuffer))) > 0)
		Parse(szBuffer, nRead);

	//detect errors in file format (should have at least some HTML tags inside)
	if(0 == m_nTagsParsedCount)
		return DOC_LOAD_FORMAT_ERROR;

	return DOC_LOAD_OK;
}

bool FormatHTML::Save(const char *szFile, NoteDocument &doc)
{
	File64 objFile;	
	if(!objFile.Open(szFile, F64_WRITE|F64_SHARE_READ|F64_OPEN_NEW))
		return false;

	//write header
	if(!m_bHtmlExport)
	{
		if(m_bNoteCenterMode)
			objFile.WriteString("<!DOCTYPE NoteCenter-File>\r\n");
		else
			objFile.WriteString("<!DOCTYPE NoteCase-File>\r\n");
	
		//calc and write selected node "index"
		char szLastNote[100];
		sprintf(szLastNote, "<!--LastNote:%d-->\r\n", doc.m_nCurrentNode);
		objFile.WriteString(szLastNote);
	}

	//write inital HTML tags including HEAD section
	objFile.WriteString("<HTML>\r\n<HEAD>\r\n");
	//define used charset
	objFile.WriteString("<meta content=\"text/html;charset=UTF-8\" http-equiv=\"Content-Type\">");
	objFile.WriteString("<TITLE>");
	//TOFIX write file title as document title???
	objFile.WriteString("</TITLE>\r\n</HEAD>\r\n<BODY BGCOLOR=\"#FFFFFF\">\r\n");

	//recursively iterate through document node tree
	DocumentIterator it(doc);
	WriteLevel(it, objFile, -1);

	//write footer
	objFile.WriteString("</BODY>\r\n</HTML>\r\n");

	return true;
}

// HTMLParser event handlers
void FormatHTML::OnTagBegin(const char *szTag, const char *szParams)
{
	//TRACE("FormatHTML:[%s] tag start\n", szTag);

	m_nTagsParsedCount ++;

	//if proper tag, create new node
	if(0 == strcasecmp(szTag, "DL"))	//DL indicates sublevel branching
	{
		m_nCurParentID = m_nCurNodeID;
		m_nCurNodeID   = -1;	
	}
	else if(0 == strcasecmp(szTag, "DT"))	//DT indicates new node title coming
	{
		m_objDoc->NodeInsert(m_nCurParentID, -1);  //TOFIX error report
		m_nCurNodeID = m_objDoc->GetNodeByIdx(m_objDoc->GetNodeCount()-1).m_nID;
		m_nDocParseState = DOC_PARSE_EXPECT_TITLE;
	}
	else if(0 == strcasecmp(szTag, "DD"))	//DD indicates new node text coming
	{
		m_nDocParseState = DOC_PARSE_EXPECT_TEXT;
	}
}

void FormatHTML::OnTagEnd(const char *szTag)	
{
	//TRACE("FormatHTML:[%s] tag end\n", szTag);

	//if proper tag, position on the parent node ?
	if(0 == strcasecmp(szTag, "DL"))	// </DL> indicates end of sublevel branching
	{
		int nIdx = m_objDoc->GetIdxFromID(m_nCurParentID);
		if(nIdx >= 0)
			m_nCurParentID = m_objDoc->GetNodeByIdx(nIdx).m_nParentID;
		//m_nCurNodeID   = -1;
	}
	else if(0 == strcasecmp(szTag, "DT"))	//DT indicates new node title coming
	{
	}
}

void FormatHTML::OnComment(const char *szText)
{
	//TRACE("FormatHTML:[%s] HTML comment\n", szText);

	//extract info from comments <!--LastNote:4-->
	if(0 == strncmp(szText, "LastNote:", strlen("LastNote:")))
	{
		m_objDoc->m_nCurrentNode = atoi(szText+strlen("LastNote:"));
	}
	else if (0 == strncmp(szText, "property:icon_file=", strlen("property:icon_file=")))
	{
		//example tag: "<!--property:icon_file=/root/aaa.xpm-->"
		//printf("icon_file->%s\n",szText);
		DocumentIterator it(*m_objDoc);
		it.GetNodeByID(m_nCurNodeID).m_nIconType = ICON_CUSTOM;
		it.GetNodeByID(m_nCurNodeID).m_strIconFile = szText + strlen("property:icon_file=");
	}
	else if (0 == strncmp(szText, "property:icon_internal=", strlen("property:icon_internal=")))
	{
		//printf("icon_internal->%s\n",szText);
		//example tag: "<!--property:icon_internal=blank-->"
		DocumentIterator it(*m_objDoc);
		int nIdx = InternalIcon_Name2Index(szText + strlen("property:icon_internal="));		
		if(nIdx >= 0)
			it.GetNodeByID(m_nCurNodeID).m_nIconType = nIdx;
	}
}

void FormatHTML::OnText(const char *szText)
{
	std::string data(szText);

	//TRACE("FormatHTML:[%s] HTML text\n", szText);

	//set text or title for current node
	if(DOC_PARSE_EXPECT_TITLE == m_nDocParseState)
	{
		//TRACE("FormatHTML:[%s] is node title \n", szText);

		if(!m_bNoteCenterMode)
			HTMLParser::UnescapeChars(data);

		//set node title
		int nIdx = m_objDoc->GetIdxFromID(m_nCurNodeID);
		if(nIdx >= 0)
			m_objDoc->GetNodeByIdx(nIdx).SetTitle(data.c_str());

		m_nDocParseState = DOC_PARSE_BLANK;
	}
	else if(DOC_PARSE_EXPECT_TEXT == m_nDocParseState)
	{
		//TRACE("FormatHTML:[%s] is node text\n", szText);

		HTMLParser::UnescapeChars(data);	//only for node text (NC format)

		//set node text
		int nIdx = m_objDoc->GetIdxFromID(m_nCurNodeID);
		if(nIdx >= 0){
			//NoteCenter Text format: remove \r\n from text start and text end
			if(0 == strncmp(data.c_str(), "\r\n", 2))
				data = data.substr(2, data.size()-2);
			if(0 == strncmp(data.c_str()+data.size()-2, "\r\n", 2))
				data = data.substr(0, data.size()-2);

			m_objDoc->GetNodeByIdx(nIdx).SetText(data.c_str());
		}

		m_nDocParseState = DOC_PARSE_BLANK;
	}
}

void WriteLevel(DocumentIterator &it, File64 &file, int nParentID)
{
	std::string strData;
	int nSiblingIdx = 0;

	int nIdx = it.GetChildIdx(nParentID, 0);  //first child
	if(nIdx < 0)
		return;

	file.WriteString("\r\n<DL>\r\n");

	while(nIdx >= 0)
	{
		//write node title
		strData = it.GetNodeByIdx(nIdx).GetTitle();
		//TOFIX if(!m_bNoteCenterMode)
			HTMLParser::EscapeChars(strData);

		file.WriteString("<DT><H3>");
		file.WriteString(strData.c_str());
		file.WriteString("</H3></DT>\r\n");

		//write node parameters (icon settings)
		if(ICON_NONE != it.GetNodeByIdx(nIdx).m_nIconType)
		{
			if(ICON_CUSTOM == it.GetNodeByIdx(nIdx).m_nIconType)
			{
				file.WriteString("<!--property:icon_file=");
				file.WriteString(it.GetNodeByIdx(nIdx).m_strIconFile.c_str());
				file.WriteString("-->\r\n");
			}
			else
			{
				file.WriteString("<!--property:icon_internal=");
				file.WriteString(InternalIcon_Index2Name(it.GetNodeByIdx(nIdx).m_nIconType));
				file.WriteString("-->\r\n");
			}
		}

		//write node text
		strData = it.GetNodeByIdx(nIdx).GetText();
		//TOFIX if(!m_bNoteCenterMode)
			HTMLParser::EscapeChars(strData);

		//change all occurences "\r" to "\r\n"
		replaceall(strData, "\r\n", "\r");
		replaceall(strData, "\r", "\r\n");

		file.WriteString("<DD><PRE>\r\n");
		file.WriteString(strData.c_str());
		file.WriteString("\r\n</PRE></DD>\r\n");

		//recursively write children of this node
		WriteLevel(it, file, it.GetNodeByIdx(nIdx).m_nID);

		//go to the next sibling node
		nSiblingIdx ++;
		nIdx = it.GetChildIdx(nParentID, nSiblingIdx);
	}

	//
	file.WriteString("</DL>\r\n");
}

void replaceall(std::string &strData, const char *szFind, const char *szReplace)
{
//	std::replace(strData.begin(), strData.end(), szFind, szReplace);

	int nSrcSize = strlen(szFind);
	if(nSrcSize > 0)
	{
		int nDstSize = strlen(szReplace);

		size_t nPos = strData.find(szFind);
		while(nPos != std::string::npos)
		{
			//strData.replace(nPos, nSrcSize, szReplace);
			strData.erase(nPos, nSrcSize);
			strData.insert(nPos, szReplace);

			nPos = strData.find(szFind, nPos+nDstSize); //keep looking forward
		}
	}
}

int InternalIcon_Name2Index(const char *szName)
{
	if(0 == strcmp("blank", szName))
		return 0;
	else if(0 == strcmp("folder", szName))
		return 1;
	else if(0 == strcmp("help", szName))
		return 2;
	else if(0 == strcmp("lock", szName))
		return 3;
	else if(0 == strcmp("new_dir", szName))
		return 4;
	else if(0 == strcmp("recycle", szName))
		return 5;

	return -1;
}

const char *InternalIcon_Index2Name(int nIndex)
{
	switch (nIndex)
	{
		case 0:	return "blank";
		case 1:	return "folder";
		case 2:	return "help";
		case 3:	return "lock";
		case 4:	return "new_dir";
		case 5:	return "recycle";
		default: return NULL;
	}
}

