/*
 * Copyright 2013 Canonical Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.0
import Ubuntu.Components 0.1
import Ubuntu.Components.ListItems 0.1 as ListItem
import "Constants.js" as Const

FocusScope {
    id: noteItem

    property string dateText
    property string noteText

    onNoteTextChanged: noteContent.setData(noteText);

    property alias dirty: noteContent.dirty
    property real collapsedHeight
    property real expandedHeight
    property real actualExpandedHeight: Math.max(noteContent.contentHeight + units.gu(2), expandedHeight)
    property bool isExpanded: false

    signal noteDeleted
    signal activated
    signal deactivated
    signal collapsed

    state: isExpanded ? "expanded" : "collapsed"

    onIsExpandedChanged: {
        // When collapsing, update the content of the collapsed text to match the expanded one.
        // When expanding, if the note is empty focus it immediately instead of requiring another click.
        // This is because there's nothing to read on an empty note, so the only action possible is edit.
        if (!isExpanded) collapsedContent.fill(noteContent.model);
        else if (readNoteData().length === 0) noteContent.startEditAtEnd()
    }

    Component.onCompleted: removableMenu.itemRemoved.connect(noteDeleted)

    function readNoteData() { return noteContent.readData() }

    ListItem.Empty {
        id: removableMenu
        showDivider: false
        removable: !noteItem.isExpanded

        onClicked: noteItem.activated()

        backgroundIndicator: RemovableBG {
            anchors.fill: parent
            state: removableMenu.swipingState
        }

        anchors {
            fill: parent
            leftMargin: units.gu(0.5)
            rightMargin: units.gu(0.5)
        }

        MouseArea {
            anchors.fill: parent
            enabled: !removableMenu.removable
            onPressAndHold: {
                if (noteContent.activeTextAreaIndex != -1) {
                    noteContent.activateClipboard("text", noteContent.activeTextArea,
                                                  noteContent.activeTextAreaIndex)
                }
            }

            onClicked: {
                noteContent.startEditAtEnd();
                mouse.accepted = false;
            }

            propagateComposedEvents: true
        }

        BorderImage {
            anchors.fill: noteContent
            source: "graphics/note_bg.sci"
        }

        MixedEdit {
            id: noteContent
            width: parent.width
            height: noteItem.height

            opacity: noteItem.isExpanded ? 1.0 : 0.0
            visible: opacity > 0.0

            onModelUpdated: collapsedContent.fill(noteContent.model);

            onActivated: { focus = true; noteItem.activated(); }

            function startEditAtEnd() {
                noteContent.focus = true;
                noteContent.editAtIndex(-1, false)
            }
        }

        CollapsedNote {
            id: collapsedContent
            anchors.fill: parent
            anchors.margins: units.gu(0.5)

            height: noteItem.collapsedHeight
            opacity: noteItem.isExpanded ? 0.0 : 1.0
        }

        CollapsedNoteDecorations {
            anchors.fill: parent
            anchors.margins: units.gu(0.5)

            date: noteItem.dateText
            opacity: collapsedContent.opacity
        }
    }

    InverseMouseArea {
        anchors.fill: parent
        onClicked: {
            /* We want the first click on the noteItem to expand the item but
              not focus anything, and the second click to focus. Therefore,
              when collapsing the item, we need to defocus is, otherwise it
              will retain focus and the next first click will refocus it again */
            noteItem.deactivated()
            noteContent.focus = false;
            mouse.accepted = false;
        }
        enabled: noteItem.isExpanded
        propagateComposedEvents: true
    }

    states: [
        State { name: "collapsed"
            PropertyChanges { target: noteItem; height: collapsedHeight }
        },
        State { name: "expanded"
            PropertyChanges { target: noteItem; height: actualExpandedHeight }
        }
    ]

    transitions: [
        Transition { from: "collapsed"
            PropertyAnimation { target: noteItem; property: "height"; duration: Const.animationDuration }
        },
        Transition { from: "expanded"
            SequentialAnimation {                  
                PropertyAnimation { target: noteItem; property: "height"; duration: Const.animationDuration }
                ScriptAction { script: noteItem.collapsed() }
            }
        }
    ]
}
