# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
# Copyright 2012-2013 Canonical
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.

"""notes-app autopilot tests."""

import BaseHTTPServer
import os.path
import glob
import threading
import base64

from autopilot.input import Mouse, Touch, Pointer
from autopilot.matchers import Eventually
from autopilot.platform import model
from autopilot.testcase import AutopilotTestCase
from testtools.matchers import Equals

from ubuntuuitoolkit.emulators import UbuntuUIToolkitEmulatorBase

from notes_app.emulators.notesapp import NotesApp
import imagedata

HTTP_SERVER_PORT = 8129


class NotesAppTestCase(AutopilotTestCase):

    """A common test case class that provides several useful methods for notes
    app tests."""

    TYPING_DELAY = 0.02

    if model() == 'Desktop':
        scenarios = [('with mouse', dict(input_device_class=Mouse))]
    else:
        scenarios = [('with touch', dict(input_device_class=Touch))]

    local_location = "../../notes-app"

    def setUp(self):
        self.pointing_device = Pointer(self.input_device_class.create())
        super(NotesAppTestCase, self).setUp()
        self.launch_app()
        self.main_window.visible.wait_for(True)

    """Workaround to find the qmlscene binary via shell globbing.
       This is needed since we can't rely on qt5-default being installed on
       devices to make qmlscene available in the path"""
    def qmlscene(self):
        return glob.glob("/usr/lib/*/qt5/bin/qmlscene")[0]

    def launch_app(self):
        if os.path.exists(self.local_location):
            self.launch_app_local()
        elif os.path.exists('/usr/share/notes-app/NotesApp.qml'):
            self.launch_app_installed()
        else:
            self.launch_click_installed()

    def launch_app_local(self):
        self.app = self.launch_test_application(
            self.qmlscene(),
            "-I",
            "../../src",
            "../../NotesApp.qml",
            emulator_base=UbuntuUIToolkitEmulatorBase)

    def launch_app_installed(self):
        self.app = self.launch_test_application(
            self.qmlscene(),
            "/usr/share/notes-app/NotesApp.qml",
            "--desktop_file_hint=/usr/share/applications/notes-app.desktop",
            app_type='qt',
            emulator_base=UbuntuUIToolkitEmulatorBase)

    def launch_click_installed(self):
        self.app = self.launch_click_package(
            "com.ubuntu.notes")

    def launch_and_quit_app(self):
        self.launch_app()
        self.main_window.visible.wait_for(True)

        # When calling launch_app an instance of the spawned process
        # control object will be stored in self.app.process, and a cleanup
        # handler will be registered that essentially kills the process.
        # Therefore, by triggering the cleanup handler here we're killing the
        # process and removing the handler, which allows a clean launch of
        # the process during regular test setup.
        self.doCleanups()

    def assert_osk_eventually_shown(self):
        if model() != 'Desktop':
            keyboardRectangle = self.main_window.get_keyboard_rectangle()
            self.assertThat(keyboardRectangle.state,
                            Eventually(Equals("shown")))

    def assert_osk_eventually_hidden(self):
        if model() != 'Desktop':
            keyboardRectangle = self.main_window.get_keyboard_rectangle()
            self.assertThat(keyboardRectangle.state,
                            Eventually(Equals("hidden")))

    @property
    def main_window(self):
        return self.app.select_single(NotesApp)


class HTTPRequestHandler(BaseHTTPServer.BaseHTTPRequestHandler):

    """
    A custom HTTP request handler that serves GET resources.
    """

    def send_image(self, data):
        self.send_header("Content-Type", "image/png")
        self.end_headers()
        self.wfile.write(data)

    def do_GET(self):
        if self.path == "/image.png":
            self.send_response(200)
            self.send_image(base64.b64decode(imagedata.IMAGE_DATA))
        elif self.path == "/large_wide.png":
            self.send_response(200)
            self.send_image(base64.b64decode(imagedata.IMAGE_LARGE_WIDE))
        elif self.path == "/small_wide.png":
            self.send_response(200)
            self.send_image(base64.b64decode(imagedata.IMAGE_SMALL_WIDE))
        elif self.path == "/large_high.png":
            self.send_response(200)
            self.send_image(base64.b64decode(imagedata.IMAGE_LARGE_HIGH))
        elif self.path == "/small_high.png":
            self.send_response(200)
            self.send_image(base64.b64decode(imagedata.IMAGE_SMALL_HIGH))
        else:
            self.send_error(404)


class HTTPServerInAThread(threading.Thread):

    """
    A simple custom HTTP server run in a separate thread.
    """

    def __init__(self, port):
        super(HTTPServerInAThread, self).__init__()
        self.server = BaseHTTPServer.HTTPServer(("", port), HTTPRequestHandler)
        self.server.allow_reuse_address = True

    def run(self):
        self.server.serve_forever()

    def shutdown(self):
        self.server.shutdown()
        self.server.server_close()


class NotesTestCaseBaseWithHTTPServer(NotesAppTestCase):

    """
    A specialization of the common test case class that runs
    a simple custom HTTP server in a separate thread.
    """

    def setUp(self):
        self.server = HTTPServerInAThread(HTTP_SERVER_PORT)
        self.server.start()
        super(NotesTestCaseBaseWithHTTPServer, self).setUp()

    def tearDown(self):
        super(NotesTestCaseBaseWithHTTPServer, self).tearDown()
        self.server.shutdown()


class DatabaseMixin(object):

    """
    Helper functions for dealing with sqlite databases
    """

    def _get_db_path(self):
        db_path_list = [
            "~/.local/share/notes-app/Databases",
            "~/.local/share/Qt Project/QtQmlViewer/QML/OfflineStorage/"
            "Databases/"]
        for path in db_path_list:
            path = os.path.expanduser(path)
            if os.path.exists(path):
                return path
        return None

    def find_db(self):
        dbs_path = self._get_db_path()
        if not dbs_path:
            return None
        files = [f for f in os.listdir(dbs_path)
                 if os.path.splitext(f)[1] == ".ini"]
        for f in files:
            ini_path = os.path.join(dbs_path, f)
            with open(ini_path) as ini:
                for line in ini:
                    if "=" in line:
                        key, val = line.strip().split("=")
                        if key == "Name" and val == "notes":
                            try:
                                return ini_path.replace(".ini", ".sqlite")
                            except OSError:
                                pass
        return None
