# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
# Copyright 2013 Canonical
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.

"""Tests for Notes app"""

from __future__ import absolute_import

from testtools.matchers import Equals
from autopilot.matchers import Eventually

from notes_app.tests import NotesAppTestCase, DatabaseMixin

import sqlite3


class TestExpandCollapse(NotesAppTestCase, DatabaseMixin):
    """Tests deleting notes"""

    def setUp(self):
        # Setup the database before starting the app
        self.setup_db()
        super(TestExpandCollapse, self).setUp()

    def setup_db(self):
        notes = [
            """{ "elements" : [
                {"content":"This is a note.","type":"text"}
            ]}""",
            """{ "elements" : [
                {"content":"This is another note.","type":"text"}
            ]}"""
        ]

        path = self.find_db()
        if path is None:
            self.launch_and_quit_app()
            path = self.find_db()
            if path is None:
                self.assertNotEquals(path, None)

        conn = sqlite3.connect(path)
        cursor = conn.cursor()
        cursor.execute("DELETE FROM notes")
        for note in notes:
            cursor.execute("INSERT INTO notes (date, note) VALUES ('2013-04-07', '" + note + "')")
        conn.commit()
        conn.close()

    def test_expand_and_collapse_many(self):
        first = self.main_window.get_notes()[0]
        second = self.main_window.get_notes()[1]

        self.assertThat(first.isExpanded, Equals(False))
        self.assertThat(second.isExpanded, Equals(False))

        self.pointing_device.click_object(first)
        self.assertThat(first.isExpanded, Eventually(Equals(True)))

        # Here and in further tests we have to wait for height to equal actualExpandedHeight
        # because that is the point when the expanding animation is finished and the note
        # reaches its final expanded geometry. Just testing for isExpanded does not work
        # as the property changes immediately and then the animation is triggered.
        self.assertThat(first.height, Eventually(Equals(first.actualExpandedHeight)))

        self.pointing_device.click_object(second)
        self.assertThat(first.isExpanded, Eventually(Equals(False)))
        self.assertThat(second.isExpanded, Eventually(Equals(True)))
        self.assertThat(second.height, Eventually(Equals(second.actualExpandedHeight)))

        self.pointing_device.click_object(first)
        self.assertThat(first.isExpanded, Eventually(Equals(True)))
        self.assertThat(second.isExpanded, Eventually(Equals(False)))

    def test_collapse_header(self):
        first = self.main_window.get_notes()[0]
        header = self.main_window.get_header()

        self.pointing_device.click_object(first)
        self.assertThat(first.isExpanded, Eventually(Equals(True)))
        self.assertThat(first.height, Eventually(Equals(first.actualExpandedHeight)))

        self.pointing_device.click_object(header)
        self.assertThat(first.isExpanded, Eventually(Equals(False)))

    def test_collapse_bottom(self):
        first = self.main_window.get_notes()[0]
        second = self.main_window.get_notes()[1]

        self.pointing_device.click_object(first)
        self.assertThat(first.isExpanded, Eventually(Equals(True)))
        self.assertThat(first.height, Eventually(Equals(first.actualExpandedHeight)))

        # click in the empty space after all the notes
        note_x, note_y, note_w, note_h = second.globalRect
        self.pointing_device.move(note_x + note_w / 2, note_y + note_h + 20)
        self.pointing_device.click()

        self.assertThat(first.isExpanded, Eventually(Equals(False)))

    def test_collapse_between(self):
        first = self.main_window.get_notes()[0]

        self.pointing_device.click_object(first)
        self.assertThat(first.isExpanded, Eventually(Equals(True)))
        self.assertThat(first.height, Eventually(Equals(first.actualExpandedHeight)))

        # click in the empty space between notes
        note_x, note_y, note_w, note_h = first.globalRect
        self.pointing_device.move(note_x + note_w / 2, note_y + note_h + 2)
        self.pointing_device.click()

        self.assertThat(first.isExpanded, Eventually(Equals(False)))
