# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
# Copyright 2013 Canonical
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.

"""Tests for Notes app"""

from __future__ import absolute_import

from testtools.matchers import Equals
from autopilot.matchers import Eventually
from autopilot.introspection.dbus import StateNotFoundError

from notes_app.tests import NotesAppTestCase, DatabaseMixin

import sqlite3
import json
import unittest


class UtilsMixin(object):
    def focus_note_for_editing(self, index):
        notes = self.main_window.get_notes()
        note = notes[index]

        self.pointing_device.click_object(note)

        # wait for note to expand
        note.height.wait_for(note.actualExpandedHeight)

        # click outside of the text area
        note_x, note_y, note_w, note_h = note.globalRect
        self.pointing_device.move(note_x + note_w / 2, note_y + note_h - 1)
        self.pointing_device.click()

        return note

    # When the application quits autopilot will fail to connect to
    # the process and therefore an exception will be raised. There is
    # no other way that I could find to check if the app did actually quit.
    def app_has_quit(self):
        try:
            self.main_window
        except StateNotFoundError:
            return True
        return False

    def get_db_path(self):
        path = self.find_db()
        if path is None:
            self.launch_and_quit_app()
            path = self.find_db()
            if path is None:
                self.assertNotEquals(path, None)
        return path


class TestQuit(NotesAppTestCase, DatabaseMixin, UtilsMixin):
    """Tests save on quit"""

    """ This is needed to wait for the application to start.
        In the testfarm, the application may take some time to show up."""
    def setUp(self):
        # Setup the database before starting the app
        self.setup_db()
        super(TestQuit, self).setUp()

    def setup_db(self):
        note_data = """{ "elements" : [
            {"content":"foobar","type":"text"}
        ]}"""

        path = self.get_db_path()
        conn = sqlite3.connect(path)
        cursor = conn.cursor()
        cursor.execute("DELETE FROM notes")
        cursor.execute("INSERT INTO notes (date, note) VALUES ('2013-04-07', '" + note_data + "')")
        conn.commit()
        conn.close()

    @unittest.skip("The current method of closing the app does not work \
        on jenkins")
    def test_save_before_quit(self):
        # Verify that after typing something in the note and quitting will save
        # the note to db
        self.focus_note_for_editing(0)
        self.keyboard.type("xxx", delay=self.TYPING_DELAY)
        self.keyboard.press_and_release("Alt+F4")

        self.assertThat(self.app_has_quit, Eventually(Equals(True)))

        path = self.find_db()
        conn = sqlite3.connect(path)
        cursor = conn.cursor()

        cursor.execute("SELECT note FROM NOTES")
        result = cursor.fetchone()
        self.assertEquals(len(result), 1)

        data = json.loads(result[0])
        self.assertEquals(len(data["elements"]), 1)
        self.assertEquals(data["elements"][0]["content"], "foobarxxx")


class TestQuitNoNote(NotesAppTestCase, DatabaseMixin, UtilsMixin):
    """Tests save on quit with no notes"""

    """ This is needed to wait for the application to start.
        In the testfarm, the application may take some time to show up."""
    def setUp(self):
        # Setup the database before starting the app
        self.setup_db()
        super(TestQuitNoNote, self).setUp()

    def setup_db(self):
        path = self.get_db_path()
        conn = sqlite3.connect(path)
        cursor = conn.cursor()
        cursor.execute("DELETE FROM notes")
        conn.commit()
        conn.close()

    @unittest.skip("The current method of closing the app does not work \
        on jenkins")
    def test_save_before_quit(self):
        # Verify that after typing something in the note and quitting will
        # save the note to db
        self.focus_note_for_editing(0)
        self.keyboard.type("xxx", delay=self.TYPING_DELAY)
        self.keyboard.press_and_release("Alt+F4")

        self.assertThat(self.app_has_quit, Eventually(Equals(True)))

        path = self.find_db()
        conn = sqlite3.connect(path)
        cursor = conn.cursor()

        cursor.execute("SELECT note FROM NOTES")
        result = cursor.fetchone()
        self.assertEquals(len(result), 1)

        data = json.loads(result[0])
        self.assertEquals(len(data["elements"]), 1)
        self.assertEquals(data["elements"][0]["content"], "xxx")
