/****************************************************************************************/
/*											*/
/* This program is free software; you can redistribute it and/or modify it under the	*/
/* terms of the GNU General Public License as published by the Free Software		*/
/* Foundation; either version 2 of the License, or (at your option) any later version.	*/
/*											*/
/* This program is distributed in the hope that it will be useful, but WITHOUT ANY	*/
/* WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A	*/
/* PARTICULAR PURPOSE. See the GNU General Public License for more details.		*/
/*											*/
/* You should have received a copy of the GNU General Public License along with this	*/
/* program; (See "COPYING"). If not, If not, see <http://www.gnu.org/licenses/>.        */
/*											*/
/*--------------------------------------------------------------------------------------*/
/*											*/
/*  Copyright   Joerg Anders, TU Chemnitz, Fakultaet fuer Informatik, GERMANY           */
/*		ja@informatik.tu-chemnitz.de						*/
/*											*/
/*											*/
/****************************************************************************************/

#include <math.h>
#include "note.h"
#include "staff.h"
#include "system.h"
#include "page.h"
#include "voice.h"
#include "resource.h"
#include "system.h"
#include "mainwindow.h"
#include "commandlist.h"
#include "commandhistory.h"
#include "tienotescommand.h"
#include "untieforwardcommand.h"


#define X_POS_PAGE_REL(p) ((getPage()->getContetXpos() + (p)) * zoom_factor - leftx)
#define Y_POS_STAFF_REL(p) (((p) + getSystem()->getYPos() + getStaff()->getBottomPos()) * zoom_factor - topy)
#define X_POS_INVERS_PAGE_REL(p) ((leftx + (p)) / zoom_factor - getPage()->getContetXpos())
#define Y_POS_INVERS_STAFF_REL(p) ((topy + (p)) / zoom_factor - (getSystem()->getYPos() + getStaff()->getBottomPos()))

#define X_PS_POS(p) ((DEFAULT_BORDER + LEFT_RIGHT_BORDER + (p)) * PS_ZOOM)
#define Y_PS_POS_STAFF_REL(p) ((height - ((p) + getSystem()->getYPos() + getStaff()->getBottomPos())) * PS_ZOOM)

#define TREBLE_PITCH_OFFS 35
#define BASS_PITCH_OFFS   23
#define ALTO_PITCH_OFFS   29
#define TENOR_PITCH_OFFS  27
#define SOPRAN_PITCH_OFFS 33

struct pitch_descr {
	char pitch, min_sharp, min_flat;
};

pitch_descr NedNote::lineToPitch[] = {
                                          {  4 /* E */, 6, 2}, {  5 /* F */, 1, 100}, {  7 /* G */, 3, 5}, {  9 /* A */, 5, 3}, { 11 /* B */, 100, 1},
{ 12 /* C */, 2, 6}, { 14 /* D */, 4, 4}, { 16 /* E */, 6, 2}, { 17 /* F */, 1, 100}, { 19 /* G */, 3, 5}, { 21 /* A */, 5, 3}, { 23 /* B */, 100, 1},
{ 24 /* C */, 2, 6}, { 26 /* D */, 4, 4}, { 28 /* E */, 6, 2}, { 29 /* F */, 1, 100}, { 31 /* G */, 3, 5}, { 33 /* A */, 5, 3}, { 35 /* B */, 100, 1},
{ 36 /* C */, 2, 6}, { 38 /* D */, 4, 4}, { 40 /* E */, 6, 2}, { 41 /* F */, 1, 100}, { 43 /* G */, 3, 5}, { 45 /* A */, 5, 3}, { 47 /* B */, 100, 1},
{ 48 /* C */, 2, 6}, { 50 /* D */, 4, 4}, { 52 /* E */, 6, 2}, { 53 /* F */, 1, 100}, { 55 /* G */, 3, 5}, { 57 /* A */, 5, 3}, { 59 /* B */, 100, 1}, 
{ 60 /* C */, 2, 6}, { 62 /* D */, 4, 4}, { 64 /* E */, 6, 2}, { 65 /* F */, 1, 100}, { 67 /* G */, 3, 5}, { 69 /* A */, 5, 3}, { 71 /* B */, 100, 1},
{ 72 /* C */, 2, 6}, { 74 /* D */, 4, 4}, { 76 /* E */, 6, 2}, { 77 /* F */, 1, 100}, { 79 /* G */, 3, 5}, { 81 /* A */, 5, 3}, { 83 /* B */, 100, 1},
{ 84 /* C */, 2, 6}, { 86 /* D */, 4, 4}, { 88 /* E */, 6, 2}, { 89 /* F */, 1, 100}, { 91 /* G */, 3, 5}, { 93 /* A */, 5, 3}, { 95 /* B */, 100, 1},
{ 96 /* C */, 2, 6}, { 98 /* D */, 4, 4}, {100 /* E */, 6, 2}, {101 /* F */, 1, 100}, {103 /* G */, 3, 5}, {105 /* A */, 5, 3}, {107 /* B */, 100, 1},
{108 /* C */, 2, 6}, {110 /* D */, 4, 4}, {112 /* E */, 6, 2}, {114 /* F */, 1, 100}, {115 /* G */, 3, 5}, {117 /* A */, 5, 3}, {119 /* B */, 100, 1}};

NedNote::NedNote(NedChordOrRest *chord_or_rest, int line, int head, unsigned int status) :
m_active(FALSE), m_acc_dist(ACCIDENTAL_BASE_DIST), m_line(line), m_tmp_line(UNKNOWN_LINE), m_status(status), m_head(head), m_chord_or_rest(chord_or_rest),
m_tie_forward(NULL), m_tie_backward(NULL){
	m_ypos = - line * LINE_DIST/2.0;
}

NedMainWindow *NedNote::getMainWindow() {return m_chord_or_rest->getStaff()->getSystem()->getPage()->getMainWindow();}
NedPage *NedNote::getPage() {return m_chord_or_rest->getStaff()->getSystem()->getPage();}
NedSystem *NedNote::getSystem() {return m_chord_or_rest->getStaff()->getSystem();}
NedStaff *NedNote::getStaff() {return m_chord_or_rest->getStaff();}
NedVoice *NedNote::getVoice() {return m_chord_or_rest->getVoice();}

void NedNote::draw(cairo_t *cr) {
	double leftx = getMainWindow()->getLeftX();
	double topy = getMainWindow()->getTopY();
	double zoom_factor = getMainWindow()->getCurrentZoomFactor();
	cairo_glyph_t glyph;
	double yl;
	double yy = (m_tmp_line == UNKNOWN_LINE) ? m_ypos : m_tmp_ypos;
	double px0, px1;
	int line;
	double tie_y_offs1, tie_y_offs2;

	if (getMainWindow()->m_selected_note == this || m_active) {
		if (getMainWindow()->doPaintColored()) {
			switch(getVoice()->getVoiceNumber()) {
				case 0: cairo_set_source_rgb (cr, DSCOLOR(V1RED), DSCOLOR(V1GREEN), DSCOLOR(V1BLUE)); break;
				case 1: cairo_set_source_rgb (cr, DSCOLOR(V2RED), DSCOLOR(V2GREEN), DSCOLOR(V2BLUE)); break;
				case 2: cairo_set_source_rgb (cr, DSCOLOR(V3RED), DSCOLOR(V3GREEN), DSCOLOR(V3BLUE)); break;
				default: cairo_set_source_rgb (cr, DSCOLOR(V4RED), DSCOLOR(V4GREEN), DSCOLOR(V4BLUE)); break;
			}
		}
		else {
			cairo_set_source_rgb (cr, 1.0, 0.0, 0.0);
		}
	}
	else if (getMainWindow()->doPaintColored()) {
		switch(getVoice()->getVoiceNumber()) {
			case 0: cairo_set_source_rgb (cr, DCOLOR(V1RED), DCOLOR(V1GREEN), DCOLOR(V1BLUE)); break;
			case 1: cairo_set_source_rgb (cr, DCOLOR(V2RED), DCOLOR(V2GREEN), DCOLOR(V2BLUE)); break;
			case 2: cairo_set_source_rgb (cr, DCOLOR(V3RED), DCOLOR(V3GREEN), DCOLOR(V3BLUE)); break;
			default: cairo_set_source_rgb (cr, DCOLOR(V4RED), DCOLOR(V4GREEN), DCOLOR(V4BLUE)); break;
		}
	}
	switch (m_head) {
		case NORMAL_NOTE:
			switch(m_chord_or_rest->getLength()) {
				case WHOLE_NOTE:
					glyph.index = BASE + 6; break;
				case NOTE_2:
					glyph.index = BASE + 7; break;
				default:
					glyph.index = BASE + 4; break;
			}
			break;
		case CROSS_NOTE1:
			glyph.index = BASE + 21; break;
		case CROSS_NOTE2:
			glyph.index = BASE + 22; break;
		case RECT_NOTE1:
			glyph.index = BASE + 23; break;
		case RECT_NOTE2:
			glyph.index = BASE + 24; break;
		case TRIAG_NOTE1:
			glyph.index = BASE + 25; break;
		case TRIAG_NOTE2:
			glyph.index = BASE + 26; break;
	}
	glyph.x = X_POS_PAGE_REL(m_xpos + m_chord_or_rest->getXPos());
	glyph.y = Y_POS_STAFF_REL(yy);



	cairo_show_glyphs(cr, &glyph, 1);

	switch (m_status & ACCIDENTAL_MASK) {
		case STAT_DFLAT: glyph.index = BASE + 29;
				glyph.x = X_POS_PAGE_REL(m_chord_or_rest->getXPos() - m_acc_dist);
				cairo_show_glyphs(cr, &glyph, 1);
				break;
		case STAT_FLAT: glyph.index = BASE + 16;
				glyph.x = X_POS_PAGE_REL(m_chord_or_rest->getXPos() - m_acc_dist);
				cairo_show_glyphs(cr, &glyph, 1);
				break;
		case STAT_SHARP: glyph.index = BASE + 0;
				glyph.x = X_POS_PAGE_REL(m_chord_or_rest->getXPos() - m_acc_dist);
				cairo_show_glyphs(cr, &glyph, 1);
				break;
		case STAT_DSHARP: glyph.index = BASE + 30;
				glyph.x = X_POS_PAGE_REL(m_chord_or_rest->getXPos() - m_acc_dist);
				cairo_show_glyphs(cr, &glyph, 1);
				break;
		case STAT_NATURAL: glyph.index = BASE + 17;
				glyph.x = X_POS_PAGE_REL(m_chord_or_rest->getXPos() - m_acc_dist);
				cairo_show_glyphs(cr, &glyph, 1);
				break;
	}

	if (m_line < -1) {
		cairo_new_path(cr);
		cairo_set_line_width(cr, zoom_factor * AUX_LINE_THICK);
		line = m_line / 2;
		line *= 2;
		for (; line < -1; line += 2) {
			yl = - line * LINE_DIST/2.0;
			cairo_move_to(cr, X_POS_PAGE_REL(m_chord_or_rest->getXPos() + m_xpos - HEAD_THICK_2 - AUX_LINE_OVER),
				Y_POS_STAFF_REL(yl));
			cairo_line_to(cr, X_POS_PAGE_REL(m_chord_or_rest->getXPos() + m_xpos + HEAD_THICK_2 + AUX_LINE_OVER),
				Y_POS_STAFF_REL(yl));
		}
		cairo_stroke(cr);
	}
	if (m_line > 9) {
		cairo_new_path(cr);
		cairo_set_line_width(cr, zoom_factor * AUX_LINE_THICK);
		line = m_line / 2;
		line *= 2;
		for (; line > 9; line -= 2) {
			yl = - line * LINE_DIST/2.0;
			cairo_move_to(cr, X_POS_PAGE_REL(m_chord_or_rest->getXPos() + m_xpos - HEAD_THICK_2 - AUX_LINE_OVER),
				Y_POS_STAFF_REL(yl));
			cairo_line_to(cr, X_POS_PAGE_REL(m_chord_or_rest->getXPos() + m_xpos + HEAD_THICK_2 + AUX_LINE_OVER),
				Y_POS_STAFF_REL(yl));
		}
		cairo_stroke(cr);
	}

	if (m_chord_or_rest->getDotCount() > 0) {
		cairo_new_path(cr);
		cairo_arc(cr, X_POS_PAGE_REL(m_xpos + m_chord_or_rest->getXPos() + HEAD_THICK + DOT1POS), Y_POS_STAFF_REL(yy),
				zoom_factor * DOT_RAD, 0, 2.0 * M_PI);
		cairo_fill(cr);
		if (m_chord_or_rest->getDotCount() > 1) {
			cairo_new_path(cr);
			cairo_arc(cr, X_POS_PAGE_REL(m_xpos + m_chord_or_rest->getXPos() + HEAD_THICK + DOT2POS), Y_POS_STAFF_REL(yy),
					zoom_factor * DOT_RAD, 0, 2.0 * M_PI);
			cairo_fill(cr);
		}
			
	}
	if (getMainWindow()->m_selected_note == this || m_active || getMainWindow()->doPaintColored()) {
		cairo_set_source_rgb (cr, 0.0, 0.0, 0.0);
	}
	if (m_tie_forward != NULL) {
		if (m_chord_or_rest->getVoice() == m_tie_forward->getVoice()) {
			if (!m_chord_or_rest->hasUpDir() && !m_tie_forward->getChord()->hasUpDir()) {
				tie_y_offs1 = -TIE_Y_OFFS1;
				tie_y_offs2 = -TIE_Y_OFFS2;
			}
			else  {
				tie_y_offs1 = TIE_Y_OFFS1;
				tie_y_offs2 = TIE_Y_OFFS2;
			}
			px0 = m_chord_or_rest->getXPos() + m_xpos + TIE_X_START;
			px1 = m_tie_forward->m_chord_or_rest->getXPos() + m_tie_forward->m_xpos - TIE_X_START;
			cairo_new_path(cr);
			cairo_move_to(cr, X_POS_PAGE_REL(px0), Y_POS_STAFF_REL(m_ypos));
			cairo_curve_to(cr,  X_POS_PAGE_REL(px0), Y_POS_STAFF_REL(m_ypos),
						X_POS_PAGE_REL(px0 + (px1 - px0)/ 2.0), Y_POS_STAFF_REL(m_ypos + tie_y_offs1),
						X_POS_PAGE_REL(px1), Y_POS_STAFF_REL(m_ypos));
			cairo_curve_to(cr,  X_POS_PAGE_REL(px1), Y_POS_STAFF_REL(m_ypos),
						X_POS_PAGE_REL(px0 + (px1 - px0)/ 2.0), Y_POS_STAFF_REL(m_ypos + tie_y_offs2),
						X_POS_PAGE_REL(px0), Y_POS_STAFF_REL(m_ypos));
			cairo_fill(cr);
		}
		else {
			px1 = getPage()->getContentWidth() + TIE_X_RIGHT_OVER;
			px0 = m_chord_or_rest->getXPos() + m_xpos + TIE_X_START;
			cairo_new_path(cr);
			cairo_move_to(cr, X_POS_PAGE_REL(px0), Y_POS_STAFF_REL(m_ypos));
			cairo_curve_to(cr,  X_POS_PAGE_REL(px0), Y_POS_STAFF_REL(m_ypos),
						X_POS_PAGE_REL(px0 + (px1 - px0)/ 2.0), Y_POS_STAFF_REL(m_ypos + TIE_Y_OFFS1),
						X_POS_PAGE_REL(px1), Y_POS_STAFF_REL(m_ypos));
			cairo_curve_to(cr,  X_POS_PAGE_REL(px1), Y_POS_STAFF_REL(m_ypos),
						X_POS_PAGE_REL(px0 + (px1 - px0)/ 2.0), Y_POS_STAFF_REL(m_ypos + TIE_Y_OFFS2),
						X_POS_PAGE_REL(px0), Y_POS_STAFF_REL(m_ypos));
			cairo_fill(cr);
		}
			
	}
	if (m_tie_backward != NULL && m_chord_or_rest->getVoice() != m_tie_backward->getVoice()) {
		px0 = getSystem()->m_system_start - TIE_X_OFFS;
		px1 = m_chord_or_rest->getXPos() + m_xpos - TIE_X_START;
		cairo_new_path(cr);
		cairo_move_to(cr, X_POS_PAGE_REL(px0), Y_POS_STAFF_REL(m_ypos));
		cairo_curve_to(cr,  X_POS_PAGE_REL(px0), Y_POS_STAFF_REL(m_ypos),
					X_POS_PAGE_REL(px0 + (px1 - px0)/ 2.0), Y_POS_STAFF_REL(m_ypos + TIE_Y_OFFS1),
					X_POS_PAGE_REL(px1), Y_POS_STAFF_REL(m_ypos));
		cairo_curve_to(cr,  X_POS_PAGE_REL(px1), Y_POS_STAFF_REL(m_ypos),
					X_POS_PAGE_REL(px0 + (px1 - px0)/ 2.0), Y_POS_STAFF_REL(m_ypos + TIE_Y_OFFS2),
					X_POS_PAGE_REL(px0), Y_POS_STAFF_REL(m_ypos));
		cairo_fill(cr);
	}
};

bool NedNote::testForTiesToDelete(NedCommandList *command_list, unsigned int dir /* = BREAK_TIE_BACKWARD */, bool execute /* = false */) {
	bool deleted_ties = FALSE;
	NedUnTieForwardCommand *untie_cmd;
	if (m_tie_backward != NULL && ((dir & BREAK_TIE_BACKWARD) != 0)) {
		deleted_ties = TRUE;
		untie_cmd = new NedUnTieForwardCommand(m_tie_backward, this);
		if (execute) {
			untie_cmd->execute();
		}
		command_list->addCommand(untie_cmd);
	}
	if (m_tie_forward != NULL && ((dir & BREAK_TIE_FORWARD) != 0)) {
		deleted_ties = TRUE;
		untie_cmd = new NedUnTieForwardCommand(this, m_tie_forward);
		if (execute) {
			untie_cmd->execute();
		}
		command_list->addCommand(untie_cmd);
	}
	return deleted_ties;
}

void NedNote::correctTieForward() {
	if (m_tie_forward != NULL) {
		m_tie_forward->m_tie_backward = this;
	}
}

void NedNote::setOffset(char offs_array[115]) {
	int offs;
	switch (getMainWindow()->m_staff_contexts[getStaff()->getStaffNumber()].m_clef_number) {
		case NEUTRAL_CLEF1:
		case TREBLE_CLEF: offs = TREBLE_PITCH_OFFS; break;
		case NEUTRAL_CLEF2:
		case BASS_CLEF: offs = BASS_PITCH_OFFS; break;
		case ALTO_CLEF: offs = ALTO_PITCH_OFFS; break;
		case TENOR_CLEF: offs = TENOR_PITCH_OFFS; break;
		case SOPRAN_CLEF: offs = SOPRAN_PITCH_OFFS; break;
	}
	switch (m_status & ACCIDENTAL_MASK) {
		case STAT_DFLAT: offs_array[m_line+offs] = -2;
				break;
		case STAT_FLAT: offs_array[m_line+offs] = -1;
				break;
		case STAT_SHARP: offs_array[m_line+offs] = 1;
				break;
		case STAT_DSHARP: offs_array[m_line+offs] = 2;
				break;
		case STAT_NATURAL: offs_array[m_line+offs] = VIRTUAL_NATURAL_OFFS;
				break;
	}
}
	

int NedNote::getPitch(int clef, int keysig, int octave_shift) {
	int p;
	int offs;
	char offs_array[115];

	if (m_tie_backward != 0) {
		return m_tie_backward->getPitch(clef, keysig, octave_shift);
	}

	switch (clef) {
		case NEUTRAL_CLEF1:
		case TREBLE_CLEF: offs = TREBLE_PITCH_OFFS; break;
		case NEUTRAL_CLEF2:
		case BASS_CLEF: offs = BASS_PITCH_OFFS; break;
		case ALTO_CLEF: offs = ALTO_PITCH_OFFS; break;
		case TENOR_CLEF: offs = TENOR_PITCH_OFFS; break;
		case SOPRAN_CLEF: offs = SOPRAN_PITCH_OFFS; break;
	}
	p = lineToPitch[m_line+offs].pitch; 
	memset(offs_array, 0, sizeof(offs_array));
	getStaff()->findAccidentals(offs_array, getSystem()->getMeasure(m_chord_or_rest->getMidiTime()), m_chord_or_rest->getMidiTime());
	switch (m_status & ACCIDENTAL_MASK) {
		case STAT_DFLAT: p -= 2;
				break;
		case STAT_FLAT: p--;
				break;
		case STAT_SHARP: p++;
				break;
		case STAT_DSHARP: p += 2;
				break;
		case STAT_NATURAL: break;
		default:
			if (offs_array[m_line+offs] == VIRTUAL_NATURAL_OFFS) { //natural
				break;
			}
			if (keysig < 0) {
				if (lineToPitch[m_line+offs].min_flat <= -keysig && offs_array[m_line+offs] == 0) {
					p--;
					break;
				}
			}
			if (lineToPitch[m_line+offs].min_sharp <= keysig && offs_array[m_line+offs] == 0) {
				p++;
				break;
			}
			p += offs_array[m_line+offs];
			break;
	} 
	return p + octave_shift;
}

void NedNote::removeUnneededAccidental(int clef, int keysig, int octave_shift) {
	int p;
	int offs;
	char offs_array[115];

	switch (clef) {
		case NEUTRAL_CLEF1:
		case TREBLE_CLEF: offs = TREBLE_PITCH_OFFS; break;
		case NEUTRAL_CLEF2:
		case BASS_CLEF: offs = BASS_PITCH_OFFS; break;
		case ALTO_CLEF: offs = ALTO_PITCH_OFFS; break;
		case TENOR_CLEF: offs = TENOR_PITCH_OFFS; break;
		case SOPRAN_CLEF: offs = SOPRAN_PITCH_OFFS; break;
	}
	p = lineToPitch[m_line+offs].pitch + octave_shift; 
	memset(offs_array, 0, sizeof(offs_array));
	getStaff()->findAccidentals(offs_array, getSystem()->getMeasure(m_chord_or_rest->getMidiTime()), m_chord_or_rest->getMidiTime());
	switch (m_status & ACCIDENTAL_MASK) {
		case STAT_FLAT: if (offs_array[m_line+offs] == -1) {
					 m_status &= (~(ACCIDENTAL_MASK));
					 break;
				}
				if (keysig < 0) {
					if (lineToPitch[m_line+offs].min_flat <= -keysig && offs_array[m_line+offs] != 2 /* natural */) {
						m_status &= (~(ACCIDENTAL_MASK));
					}
				}
				break;
		case STAT_SHARP: if (offs_array[m_line+offs] == 1) {
					m_status &= (~(ACCIDENTAL_MASK));
					break;
				 }
				if (lineToPitch[m_line+offs].min_sharp <= keysig && offs_array[m_line+offs] != 2 /* natural */) {
					m_status &= (~(ACCIDENTAL_MASK));
				}
				break;
		case STAT_NATURAL: 
				if (offs_array[m_line+offs] == 2) { //natural
					m_status &= (~(ACCIDENTAL_MASK));
					break;
				}
				if (offs_array[m_line+offs] != 0) break;
				if (keysig < 0) {
					if (lineToPitch[m_line+offs].min_flat > -keysig) {
						m_status &= (~(ACCIDENTAL_MASK));
					}
					break;
				}
				if (lineToPitch[m_line+offs].min_sharp > keysig) {
					m_status &= (~(ACCIDENTAL_MASK));
				}
				break;
	} 
}
	

void NedNote::prepareReplay(int clef, int keysig, int octave_shift, int grace_time, bool is_grace, bool is_stacc) {
	int pitch;

	pitch = getPitch(clef, keysig, octave_shift);
	if (is_grace) {
		NedResource::addMidiEvent(getMainWindow()->m_staff_contexts[getStaff()->getStaffNumber()].m_midi_channel,
			m_chord_or_rest->getMidiTime() - grace_time, SND_SEQ_EVENT_NOTEON, pitch, getMainWindow()->m_staff_contexts[getStaff()->getStaffNumber()].m_midi_volume, this);
		NedResource::addMidiEvent(getMainWindow()->m_staff_contexts[getStaff()->getStaffNumber()].m_midi_channel,
			m_chord_or_rest->getMidiTime() + m_chord_or_rest->getDuration(), SND_SEQ_EVENT_NOTEOFF, pitch, 0, this);
	}
	else if (is_stacc && m_chord_or_rest->getLength() > NOTE_32 && grace_time == 0) {
		NedResource::addMidiEvent(getMainWindow()->m_staff_contexts[getStaff()->getStaffNumber()].m_midi_channel,
			m_chord_or_rest->getMidiTime(), SND_SEQ_EVENT_NOTEON, pitch, getMainWindow()->m_staff_contexts[getStaff()->getStaffNumber()].m_midi_volume, this);
		NedResource::addMidiEvent(getMainWindow()->m_staff_contexts[getStaff()->getStaffNumber()].m_midi_channel,
			m_chord_or_rest->getMidiTime() + m_chord_or_rest->getDuration() / 2, SND_SEQ_EVENT_NOTEOFF, pitch, 0, this);
	}
	else {
		NedResource::addMidiEvent(getMainWindow()->m_staff_contexts[getStaff()->getStaffNumber()].m_midi_channel,
			m_chord_or_rest->getMidiTime(), SND_SEQ_EVENT_NOTEON, pitch, getMainWindow()->m_staff_contexts[getStaff()->getStaffNumber()].m_midi_volume, this);
		NedResource::addMidiEvent(getMainWindow()->m_staff_contexts[getStaff()->getStaffNumber()].m_midi_channel,
			m_chord_or_rest->getMidiTime() + m_chord_or_rest->getDuration() - grace_time, SND_SEQ_EVENT_NOTEOFF, pitch, 0, this);
	}
}

void NedNote::setTieSimple(NedNote *other_note) {
	m_tie_forward = other_note;
	other_note->m_tie_backward = this;
}
	
void NedNote::removeBackwardTie() {
	if (m_tie_backward == NULL) return;
	m_tie_backward->m_tie_forward = NULL;
	m_tie_backward = NULL;
}
	

void NedNote::do_tie(NedNote *other_note) {
	int sorting;
	NedNote *n0, *n1;
	if (getChord()->getType() == TYPE_GRACE || other_note->getChord()->getType() == TYPE_GRACE) return;
	if (other_note == this) return;
	if (other_note->m_line != m_line) return;
	sorting = m_chord_or_rest->getSorting(other_note->m_chord_or_rest);
	switch (sorting) {
		case SORTING_NONE: return;
		case SORTING_LESS: n0 = other_note; n1 = this; break;
		default: n0 = this; n1 = other_note; break;
	}
	if (n0->m_tie_forward != NULL || n1->m_tie_backward != NULL) return;

	NedCommandList *command_list = new NedCommandList(getMainWindow(), getSystem());
	command_list->addCommand(new NedTieNotesCommand(n0, n1));
	command_list->execute();
	getMainWindow()->getCommandHistory()->addCommandList(command_list);
}

	
void NedNote::setTies(NedNote *n1, NedNote *n2) {
	n1->m_tie_forward = n2;
	n2->m_tie_backward = n1;
}



bool NedNote::noConflict() {
	return m_chord_or_rest->noConflict(m_tmp_line);
}


bool NedNote::trySelect(double x, double y) {
	int zoom_level = getMainWindow()->getCurrentZoomLevel();
	double leftx = getMainWindow()->getLeftX();
	double topy = getMainWindow()->getTopY();
	double zoom_factor = getMainWindow()->getCurrentZoomFactor();
	cairo_text_extents_t *extention = &(NedResource::fontextentions[zoom_level][4]);

	double xl = X_POS_INVERS_PAGE_REL(x);
	double yl = Y_POS_INVERS_STAFF_REL(y);
	double xnote = m_xpos + m_chord_or_rest->getXPos();

	if (m_ypos + extention->y_bearing / zoom_factor < yl &&
	    m_ypos + (extention->y_bearing  +  extention->height) / zoom_factor > yl &&
	    xnote + extention->x_bearing / zoom_factor < xl &&
	    xnote + (extention->x_bearing  +  extention->width) / zoom_factor > xl) {
		getMainWindow()->m_selected_note = this;
		return TRUE;
	}
	return FALSE;
}

bool NedNote::testYShift(double y) {
	double topy = getMainWindow()->getTopY();
	double zoom_factor = getMainWindow()->getCurrentZoomFactor();

	double yl = Y_POS_INVERS_STAFF_REL(y);
	int old_line = m_tmp_line;
	m_tmp_line = (int) (m_line + (m_ypos - yl) / (LINE_DIST/2.0));
	
	if (m_tmp_line != old_line) {
		m_tmp_ypos =  - m_tmp_line * LINE_DIST/2.0;
		return TRUE;
	}
	return FALSE;
}

void NedNote::setStatus(unsigned int status) {
	int clef, keysig, octave_shift;
	staff_context_str *context;
	m_status = status;
	context = &(getMainWindow()->m_staff_contexts[getStaff()->getStaffNumber()]);
	getStaff()->getCurrentClefAndKeysig(m_chord_or_rest->getMidiTime(), &clef, &keysig, &octave_shift);
	if (NedResource::m_midi_echo) {
		NedResource::playImmediately(context->m_midi_channel, context->m_midi_program, getPitch(clef, keysig, octave_shift), context->m_midi_volume);
	}
}
void NedNote::moveUpDown(int line) {
	staff_context_str *context;
	int clef, keysig, octave_shift;
	getStaff()->getCurrentClefAndKeysig(m_chord_or_rest->getMidiTime(), &clef, &keysig, &octave_shift);
	m_line = line;
	m_ypos = - m_line * LINE_DIST/2.0;
	m_chord_or_rest->sortNotes();
	getMainWindow()->reposit(NULL, getPage());
	context = &(getMainWindow()->m_staff_contexts[getStaff()->getStaffNumber()]);
	if (NedResource::m_midi_echo) {
		NedResource::playImmediately(context->m_midi_channel, context->m_midi_program, getPitch(clef, keysig, octave_shift), context->m_midi_volume);
	}
	m_tmp_line = UNKNOWN_LINE;
}


bool NedNote::testRelativeMove(int dist) {
	return m_chord_or_rest->noConflict(m_line + dist);
}

void NedNote::shiftNote(bool shift) {
	switch(m_chord_or_rest->getLength()) {
		case WHOLE_NOTE:
			m_xpos = -HEAD_THICK_HALF_2; break;
		case NOTE_2:
			m_xpos = -HEAD_THICK_HALF_2; break;
		default:
			m_xpos = -HEAD_THICK_2; break;
	}
	if (shift) {
		m_xpos = -m_xpos;
	}
}

void NedNote::placeAccidental(bool all_shifted, int acc_places[4]) {
	bool placed;
	int  i;
	if ((m_status & ACCIDENTAL_MASK) != 0) {
		placed = false;
		for (i = 0; !placed && i < 4; i++) {
			if (m_line - acc_places[i] > 3) {
				placed = true;
				m_acc_dist = ACCIDENTAL_BASE_DIST + i * NedResource::m_acc_bbox.width - (all_shifted ?  HEAD_THICK : 0);
				acc_places[i] = m_line;
			}
		}
		if (!placed) {
			NedResource::Warning("Cannot place accidental");
		}
	}
}

void NedNote::computeBounds(int len, double *minx, double *miny, double *maxx, double *maxy) {
	double yy0, xx0, xx1, yy1;

	switch(len) {
		case WHOLE_NOTE:
			xx0 = m_xpos - HEAD_THICK_2;
			yy0 = m_ypos - HEAD_HIGHT_2;
			xx1 = m_xpos + HEAD_THICK_2;
			yy1 = m_ypos + HEAD_HIGHT_2;
			break;
		case NOTE_2:
			xx0 = m_xpos - HEAD_THICK_HALF_2;
			yy0 = m_ypos - HEAD_HIGHT_HALF_2;
			xx1 = m_xpos + HEAD_THICK_HALF_2;
			yy1 = m_ypos + HEAD_HIGHT_HALF_2;
			break;
		default:
			xx0 = m_xpos - HEAD_THICK_2;
			yy0 = m_ypos - HEAD_HIGHT_2;
			xx1 = m_xpos + HEAD_THICK_2;
			yy1 = m_ypos + HEAD_HIGHT_2;
			break;
	}

	if (m_line < -1 || m_line > 9) {
		xx0 -= AUX_LINE_OVER;
		xx1 += AUX_LINE_OVER;
	}

	if ((m_status & ACCIDENTAL_MASK) != 0) {
		if (-m_acc_dist /* + NedResource::m_acc_bbox.x */ < xx0) xx0 = -m_acc_dist /* + NedResource::m_acc_bbox.x */;
		if (m_ypos + NedResource::m_acc_bbox.y < yy0) yy0 = m_ypos + NedResource::m_acc_bbox.y;
		if ((m_status & ACCIDENTAL_MASK) != STAT_FLAT && (m_status & ACCIDENTAL_MASK) != STAT_DFLAT) {
			if (yy0 + NedResource::m_acc_bbox.height > yy1) yy1 = yy0 + NedResource::m_acc_bbox.height;
		}
	}
	switch (m_chord_or_rest->getDotCount()) {
		case 1: if (xx1 < m_xpos + HEAD_THICK + DOT1POS) {xx1 =  m_xpos + HEAD_THICK +  DOT1POS;} break;
		case 2: if (xx1 < m_xpos + HEAD_THICK + DOT2POS) {xx1 =  m_xpos + HEAD_THICK +  DOT2POS;} break;
	}
	if (xx0 < *minx)  *minx = xx0;
	if (yy0 < *miny)  *miny = yy0;
	if (xx1 > *maxx)  *maxx = xx1;
	if (yy1 > *maxy)  *maxy = yy1;
}

void NedNote::saveNote(FILE *fp) {
	int marker;

	if (m_tie_forward || m_tie_backward) {
		marker = NedResource::addAddr(this);
		fprintf(fp, "|%d| ", marker);
	}
	switch (m_status & ACCIDENTAL_MASK) {
		case STAT_DFLAT: fprintf(fp, "p");
				break;
		case STAT_FLAT: fprintf(fp, "b");
				break;
		case STAT_SHARP: fprintf(fp, "#");
				break;
		case STAT_DSHARP: fprintf(fp, "c");
				break;
		case STAT_NATURAL:fprintf(fp, "=");
				break;
		default: fprintf(fp, "_");
			break;
	}
	fprintf(fp, "%d ", m_line);
	if (m_head > NORMAL_NOTE) {
		fprintf(fp, " [  head :  %d  ]", m_head);
	}
}

void NedNote::saveTies(FILE *fp, bool *ties_written) {
	int marker1, marker2;
	if (m_tie_backward != NULL) {
		if (!(*ties_written)) {
			fprintf(fp, "TIES\n");
			*ties_written = TRUE;
		}
		marker1 = NedResource::getMarkerOfAddress(m_tie_backward);
		marker2 = NedResource::getMarkerOfAddress(this);
		fprintf(fp, "( %d, %d )\n", marker1, marker2);
	}
}

gint NedNote::compare(NedNote *a, NedNote *b) {
	if (a->m_line < b->m_line) return -1;
	if (a->m_line > b->m_line) return  1;
	return 0;
}

void NedNote::adjust_pointers(struct addr_ref_str *addrlist) {
	struct addr_ref_str *addr_ptr;
	void *ptr;
	if (m_tie_forward != NULL) {
		ptr = NULL;
		for (addr_ptr = addrlist; addr_ptr; addr_ptr = addr_ptr->next) {
			if (addr_ptr->ori == m_tie_forward) {
				ptr = addr_ptr->copy;
				break;
			}
		}
		m_tie_forward = (NedNote *) ptr;
	}
	if (m_tie_backward != NULL) {
		ptr = NULL;
		for (addr_ptr = addrlist; addr_ptr; addr_ptr = addr_ptr->next) {
			if (addr_ptr->ori == m_tie_backward) {
				ptr = addr_ptr->copy;
				break;
			}
		}
		m_tie_backward = (NedNote *) ptr;
	}
}
		
	

