/*
 ** Copyright(C) 2005-2006 INL
 ** written by  Eric Leblond <regit@inl.fr>
 **
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, version 2 of the License.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software
 ** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 **
 ** In addition, as a special exception, the copyright holders give
 ** permission to link the code of portions of this program with the
 ** Cyrus SASL library under certain conditions as described in each
 ** individual source file, and distribute linked combinations
 ** including the two.
 ** You must obey the GNU General Public License in all respects
 ** for all of the code used other than Cyrus SASL.  If you modify
 ** file(s) with this exception, you may extend this exception to your
 ** version of the file(s), but you are not obligated to do so.  If you
 ** do not wish to do so, delete this exception statement from your
 ** version.  If you delete this exception statement from all source
 ** files in the program, then also delete it here.
 **
 ** This product includes software developed by Computing Services
 ** at Carnegie Mellon University (http://www.cmu.edu/computing/).
 **
 */

#include <auth_srv.h>

/**
 * \addtogroup NuauthCore
 * @{
 */

/** \file client_mngr.c
 * \brief Manage client related structure
 *
 * Provide a set of functions that are used to interact with client related structure.
 * This aims to provide an abstraction to avoid change in other parts of the code.
 */

/** global lock for client hash. */
GMutex* client_mutex;
/** Client structure */
GHashTable* client_conn_hash = NULL;
GHashTable* client_ip_hash = NULL;

void clean_session(user_session_t * c_session)
{
    if (c_session->tls){
        gnutls_deinit(*(c_session->tls));
        g_free(c_session->tls);
    }
    g_free(c_session->user_name);
    g_slist_free(c_session->groups);

    g_free(c_session->sysname);
    g_free(c_session->release);
    g_free(c_session->version);

    g_mutex_free(c_session->tls_lock);

    if (c_session){
        g_free(c_session);
    }
}

static void hash_clean_session(user_session_t * c_session)
{
    int socket = (int)gnutls_transport_get_ptr(*c_session->tls);
    clean_session(c_session);
    shutdown(socket, SHUT_RDWR);
    close(socket);
}


void init_client_struct()
{
	/* build client hash */
    client_conn_hash = g_hash_table_new_full(NULL, NULL, NULL,
			(GDestroyNotify)hash_clean_session);

	/* build client hash */
    client_ip_hash = g_hash_table_new(NULL, NULL);
    client_mutex = g_mutex_new();
}

void add_client(int socket, gpointer datas)
{
	user_session_t * c_session = (user_session_t *) datas;
	GSList * ipsockets;

    g_mutex_lock (client_mutex);

	g_hash_table_insert(client_conn_hash,GINT_TO_POINTER(socket),datas);
	/* need to create entry in ip hash */
	ipsockets = g_hash_table_lookup(client_ip_hash,GINT_TO_POINTER(c_session->addr));
	ipsockets = g_slist_prepend(ipsockets,c_session);
	g_hash_table_replace (client_ip_hash, GINT_TO_POINTER(c_session->addr), ipsockets);

    g_mutex_unlock (client_mutex);
}

void delete_client_by_socket(int socket)
{
	GSList * ipsockets;
	user_session_t * session;

    g_mutex_lock(client_mutex);

	/* get addr of of client
	 *  get element
	 *  get addr field
	 */
	session = (user_session_t*)(g_hash_table_lookup(client_conn_hash ,GINT_TO_POINTER(socket)));
    if (session) {
        /* walk on IP based struct to find the socket */
        ipsockets = g_hash_table_lookup(client_ip_hash ,GINT_TO_POINTER(session->addr));
        /* destroy entry */
        ipsockets = g_slist_remove(ipsockets , session);
        if (ipsockets != NULL) {
            g_hash_table_replace (client_ip_hash, GINT_TO_POINTER(session->addr), ipsockets);
        } else {
            g_hash_table_remove(client_ip_hash, GINT_TO_POINTER(session->addr));
        }

        /* remove entry from hash */
        g_hash_table_remove(client_conn_hash,GINT_TO_POINTER(socket));
    } else {
        log_message(WARNING,AREA_USER,"Could not found user session in hash");
    }

    g_mutex_unlock(client_mutex);
}

inline user_session_t * get_client_datas_by_socket(int socket)
{
  void * ret;

  g_mutex_lock(client_mutex);
  ret = g_hash_table_lookup(client_conn_hash ,GINT_TO_POINTER(socket));
  g_mutex_unlock(client_mutex);
  return ret;
}

inline GSList * get_client_sockets_by_ip(uint32_t ip)
{
  void * ret;

  g_mutex_lock(client_mutex);
  ret = g_hash_table_lookup(client_ip_hash ,GINT_TO_POINTER(ip));
  g_mutex_unlock(client_mutex);
  return ret;
}

inline guint get_number_of_clients()
{
	return g_hash_table_size(client_conn_hash);
}

static gboolean look_for_username_callback (gpointer key,
                                             gpointer value,
                                             gpointer user_data)
{
	if(! strcmp(
				((user_session_t*)value)->user_name,
			user_data)){
		return TRUE;
	} else {
		return FALSE;
	}
}

inline user_session_t* look_for_username(const gchar* username)
{
  void * ret;
    g_mutex_lock(client_mutex);
	ret = g_hash_table_find(client_conn_hash,look_for_username_callback,(void*)username);
    g_mutex_unlock(client_mutex);
    return ret;
}

char warn_clients(struct msg_addr_set * global_msg)
{
    GSList* ipsockets=NULL;
#if DEBUG_ENABLE
    if (DEBUG_OR_NOT(DEBUG_LEVEL_VERBOSE_DEBUG,DEBUG_AREA_USER))
    {
        struct in_addr saddress;
        saddress.s_addr=htonl(global_msg->addr);
        g_message("Warn client(s) on IP %s",inet_ntoa(saddress));
    }
#endif

    g_mutex_lock(client_mutex);
    /* look for GSList of tls sessions linked with IP */
    ipsockets=g_hash_table_lookup(client_ip_hash,GINT_TO_POINTER(ntohl(global_msg->addr)));
    if (ipsockets) {
	    global_msg->found=TRUE;
	    while (ipsockets) {

		    g_mutex_lock( ((user_session_t *)(ipsockets->data))->tls_lock);
		    gnutls_record_send(*(gnutls_session*)(((user_session_t *)ipsockets->data)->tls),
				    global_msg->msg,
				    ntohs(global_msg->msg->length)
				    );

		    g_mutex_unlock( ((user_session_t *)ipsockets->data)->tls_lock);
		    ipsockets=ipsockets->next;
        }
        g_mutex_unlock(client_mutex);
        return 1;
    } else {
        g_mutex_unlock(client_mutex);
        return 0;
    }
}

gboolean hash_delete_client(gpointer key, gpointer value, gpointer userdata)
{
    g_slist_free(value);
    return TRUE;
}

void close_clients(int signal)
{
    if (client_conn_hash != NULL)
        g_hash_table_destroy(client_conn_hash);
    if (client_ip_hash != NULL) {
        g_hash_table_foreach_remove(client_ip_hash, hash_delete_client, NULL);
        g_hash_table_destroy(client_ip_hash);
    }
}

gboolean   is_expired_client (gpointer key,
                             gpointer value,
                             gpointer user_data)
{
        if ( ((user_session_t*)value)->expire < *((time_t*)user_data) ){
                return TRUE;
        } else {
                return FALSE;
        }
}

void kill_expired_clients_session()
{
        time_t current_time=time(NULL);
        g_hash_table_foreach_remove (
                        client_conn_hash,
                        is_expired_client,
                        &current_time
                        );
}

/** @} */
