#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Copyright(C) 2007-2008 INL
Written by Romain Bignon <romain AT inl.fr>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, version 3 of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

$Id: infopage.py 12119 2008-01-10 15:24:21Z gryzor $
"""

from nevow import inevow
from nevow import rend, loaders, tags
from nevow.i18n import _, render as i18nrender
from frags import BaseFragment
from tools import trans
import re
import os
import nulog


mypath = os.path.dirname(__file__) + os.sep

class PacketInfo(BaseFragment):
    """ Fragment to show a packet info """

    docFactory = loaders.xmlfile(mypath + 'xml/packetinfo.xml', ignoreDocType=True)

    render_i18n = i18nrender()

    def __init__(self, urlbase, args, function='PacketInfo'):
        """ @param function [string] table object to call
            @param multitable [bool] True if there is multiple BaseFragment herited objects on the same page
        """

        BaseFragment.__init__(self, function, urlbase, args, None)

    #################
    #    RENDER     #
    #################

    def render_table(self, context, data):

        if hasattr(data, 'table'):
            self.table = data
        elif isinstance(data, list) and hasattr(data[0], 'table'):
            self.table = data[0]
            data = data[0]
        elif hasattr(self, 'error'):
            # If it isn't an instance of a list of the TableBase object,
            # we check if this is an object with an "error" attribute.
            # In this case, it is because there was an error.
            return _('Error: %s') % trans(context, self.error)
        else:
            return _('Error: Unable to get table.')

        tag = context.tag
        headers = tag.allPatterns('header')
        pattern = tag.patternGenerator('item')
        divider = tag.patternGenerator('divider', default=tags.invisible)
        content = [(pattern(), divider())]
        if not content:
            content = tag.allPatterns('empty')
        else:
            ## No divider after the last thing.
            content[-1] = content[-1][0]
        footers = tag.allPatterns('footer')

        return tag.clear()[ headers, content, footers ]

    def render_title(self, context, data):
        if not hasattr(self, 'table'):
            return _('Error: ID not found')
        return _('Details for packet %s') % self.table.info['id']

    def render_user(self, ctx, data):
        if not hasattr(self, 'table'): return ''
        if not self.table.info['user_id']:
            return ''
        else:
            return ctx.tag.clear()[
                tags.h3[tags.img(alt="user",src="img/user.gif"), _('User information')],
                tags.table(_class="packet")[
                    tags.thead[
                        tags.tr(_class="t_sheader")[
                            [tags.td[_('Username')], tags.td[_('UserID')]]
                        ]
                    ],
                    tags.tr(_class="t_data")[
                        tags.td[self.table.info['username'] or ''],
                        tags.td[self.table.info['user_id'] or '']
                    ]
                ]]

    def render_time(self, ctx, data):
        if not hasattr(self, 'table'): return ''
        return ctx.tag.clear()[
                tags.h3[tags.img(alt="user",src="img/time.gif"), _('Incoming time')],
                tags.table(_class="packet")[
                    tags.thead[
                        tags.tr(_class="t_sheader")[
                            [tags.td[_('Time (sec)')], tags.td[_('microsec')]]
                        ]
                    ],
                    tags.tr(_class="t_data")[
                        tags.td[str(self.table.info['timestamp'])],
                        tags.td[self.table.info['oob_time_usec'] or '']
                    ]
                ]]

    def render_route(self, ctx, data):
        if not hasattr(self, 'table'): return ''
        r = [tags.h3[tags.img(alt="user",src="img/route.gif"), _('Routing information')]]
        mac = "(\w+:\w+:\w+:\w+:\w+:\w+):(\w+:\w+:\w+:\w+:\w+:\w+):(\w+:\w+)"
        match = self.table.info['raw_mac'] and re.findall(mac, self.table.info['raw_mac'])
        if match and isinstance(match, list) and isinstance(match[0], tuple) and len(match[0]) == 3:
            r += [tags.table(_class="packet")[
                    tags.thead[
                        tags.tr(_class="t_sheader")[
                            [tags.td[_('Mac dest')],
                             tags.td[_('Mac src')],
                             tags.td[_('MAC Proto')]]
                        ]
                    ],
                    tags.tr(_class="t_data")[
                        [tags.td[match[0][0] or ''],
                        tags.td[match[0][1] or ''],
                        tags.td[match[0][2] or '']]
                    ]
                ]]
        r += [tags.table(_class="packet")[
                    tags.thead[
                        tags.tr(_class="t_sheader")[
                            [tags.td[_('In interface')],
                             tags.td[_('Out interface')],
                             tags.td[_('Log prefix')],
                             tags.td[_('Mark')]]
                        ]
                    ],
                    tags.tr(_class="t_data")[
                        [tags.td[self.table.info['oob_in'] or ''],
                        tags.td[self.table.info['oob_out'] or ''],
                        tags.td[self.table.info['oob_prefix'] or ''],
                        tags.td[self.table.info['oob_mark'] or '']]
                    ]
                ]]

        return ctx.tag.clear()[r]

    def render_accounting(self, ctx, data):
        if not hasattr(self, 'table') or self.table.info['packets_in'] is None: return ''
        r = [tags.h3[tags.img(alt="user",src="img/route.gif"), _('Connection accounting')]]

        r += [tags.table(_class="packet")[
               tags.thead[
                   tags.tr(_class="t_sheader")[
                       [tags.td[_('Packets In')],
                        tags.td[_('Packets Out')],
                        tags.td[_('Bytes In')],
                        tags.td[_('Bytes Out')]]
                   ]
               ],
               tags.tr(_class="t_data")[
                   [tags.td[self.table.info['packets_in'] or ''],
                   tags.td[self.table.info['packets_out'] or ''],
                   tags.td[self.table.info['bytes_in'] or ''],
                   tags.td[self.table.info['bytes_out'] or '']]
               ]
           ]]
        return ctx.tag.clear()[r]

    def render_headers(self, ctx, data):
        if not hasattr(self, 'table'): return ''
        r = [tags.h3[tags.img(alt="user",src="img/headers.gif"), _('Packet Headers')]]
        r += [tags.table(_class="packet")[
                    tags.thead[
                        tags.tr(_class="t_sheader")[
                            tags.td[_('Src host')],
                            tags.td[_('Dest host')]
                        ]
                    ],
                    tags.tr(_class="t_data")[
                        tags.td[self.table.info['ip_saddr'] or ''],
                        tags.td[self.table.info['ip_daddr'] or '']
                    ]
                ]]

        if self.table.info['ip_id']:

            r += [tags.table(_class="packet")[
                    tags.thead[
                        tags.tr(_class="t_sheader")[
                            tags.td[_('TOS')],
                            tags.td[_('TTL')],
                            tags.td[_('Total length')],
                            tags.td[_('Header length')],
                            tags.td[_('Checksum')],
                            tags.td[_('Packet Id')]
                        ]
                    ],
                    tags.tr(_class="t_data")[
                        tags.td[self.table.info['ip_tos'] or ''],
                        tags.td[self.table.info['ip_ttl'] or ''],
                        tags.td[self.table.info['ip_totlen'] or ''],
                        tags.td[self.table.info['ip_ihl'] or ''],
                        tags.td[self.table.info['ip_csum'] or ''],
                        tags.td[self.table.info['ip_id'] or ''],
                    ]
                ]]

        return ctx.tag.clear()[r]

    def render_port(self, ctx, data):
        if not hasattr(self, 'table'): return ''
        r = [tags.h3[tags.img(alt="user",src="img/proto.gif"), _('Protocol %s') % self.table.info['proto']]]
        r += [tags.table(_class="packet")[
                    tags.thead[
                        tags.tr(_class="t_sheader")[
                            tags.td[_('Src port')],
                            tags.td[_('Dest port')]
                        ]
                    ],
                    tags.tr(_class="t_data")[
                        tags.td[self.table.info['sport'] or ''],
                        tags.td[self.table.info['dport'] or '']
                    ]
                ]]
        # If protocol is ICMP, we show specific information.
        if self.table.info['proto'] == 'icmp' and self.table.info['icmp_type']:
            r += [tags.table(_class="packet")[
                    tags.thead[
                        tags.tr(_class="t_sheader")[
                            tags.td[_('ICMP Type')],
                            tags.td[_('ICMP Code')]
                        ]
                    ],
                    tags.tr(_class="t_data")[
                        tags.td[self.table.info['icmp_type'] or ''],
                        tags.td[self.table.info['icmp_code'] or '']
                    ]
                ]]
        # If protocol is tcp, we show specific information.
        if self.table.info['proto'] == 'tcp' and self.table.info['tcp_seq']:
            r += [tags.table(_class="packet")[
                    tags.thead[
                        tags.tr(_class="t_sheader")[
                            tags.td[_('Seq Number')],
                            tags.td[_('ACK Number')],
                            tags.td[_('TCP Window')]
                        ]
                    ],
                    tags.tr(_class="t_data")[
                        tags.td[self.table.info['tcp_seq'] or ''],
                        tags.td[self.table.info['tcp_ackseq'] or ''],
                        tags.td[self.table.info['tcp_window'] or '']
                    ]
                ]]
            r += [tags.table(_class="packet")[
                    tags.thead[
                        tags.tr(_class="t_sheader")[
                            tags.td[_('URG')],
                            tags.td[_('URGP')],
                            tags.td[_('ACK')],
                            tags.td[_('PSH')],
                            tags.td[_('RST')],
                            tags.td[_('SYN')],
                            tags.td[_('FIN')]
                        ]
                    ],
                    tags.tr(_class="t_data")[
                        tags.td[self.table.info['tcp_urg'] or ''],
                        tags.td[self.table.info['tcp_urgp'] or ''],
                        tags.td[self.table.info['tcp_ack'] or ''],
                        tags.td[self.table.info['tcp_psh'] or ''],
                        tags.td[self.table.info['tcp_rst'] or ''],
                        tags.td[self.table.info['tcp_syn'] or ''],
                        tags.td[self.table.info['tcp_fin'] or '']
                    ]
                ]]
        return ctx.tag.clear()[r]

    def render_other(self, ctx, data):
        if not hasattr(self, 'table'): return ''
        if not self.table.info['user_id']:
            return ''
        else:
            return ctx.tag.clear()[
                tags.h3[tags.img(alt="user",src="img/info.gif"), _('Additional information')],
                tags.table(_class="packet")[
                    tags.thead[
                        tags.tr(_class="t_sheader")[
                            tags.td[_('Operating system')],
                            tags.td[_('Application')]
                        ]
                    ],
                    tags.tr(_class="t_data")[
                        tags.td[self.table.info['client_os'] or ''],
                        tags.td[self.table.info['client_app'] or '']
                    ]
                ]]

class InfoPage(rend.Fragment):
    """ Page to show information of an entry in a table """

    docFactory = loaders.xmlfile(mypath + 'xml/infopage.xml', ignoreDocType=True)

    def __init__(self, urlbase, table):

        self.function = table
        self.urlbase = urlbase

    def render_info(self, ctx, data):
        """ Get content fragment """

        infos = {'PacketInfo': PacketInfo,
                }

        args = ctx.locate(inevow.IRequest).args
        args = dict((i, args[i][0]) for i in args)

        if infos.has_key(self.function):
            return ctx.tag[infos[self.function](self.urlbase, args)]
        else:
            return tags.h2[_("Page not found")]
