#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Copyright(C) 2007 INL
Written by Romain Bignon <romain AT inl.fr>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, version 3 of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

$Id: triggers.py 12055 2008-01-08 13:54:56Z eric $
"""

#########################
# HOW TO USE THIS SCRIPT
###
# When you create some netfilters rules, the only
# thing to know what is an ACCEPTED or a DROPPED
# packet is the log prefix.
#
# You can tell to this script what word used in
# your log prefix can be used to switch the state
# of packet to 0 (drop).
#
# For NuFW packets, state is correctly setted.
###
# Usage:
#          ./triggers [--drop <drop prefix>] | mysql -ulogin -ppass table
#
###

from optparse import OptionParser

def parseOptions():
    parser = OptionParser(usage="%prog [--drop <drop prefix>]")
    parser.add_option("--drop", "-d", help="Log prefix used to know which packet is dropped", action="store", type="str", default="DROP")

    options, args = parser.parse_args()

    return options


def main():

    # Options parsing
    options = parseOptions()

    print """
        DROP TRIGGER IF EXISTS update_cache;

        DELIMITER |

        CREATE TRIGGER update_cache BEFORE INSERT ON ulog
            FOR EACH ROW BEGIN

                /* If state is null, it is because this is a netfilter log.
                 * We change packet's state to 0 if state is NULL and if
                 * prefix begins with "%(drop)s"
                 */
                IF NEW.state IS NULL AND NEW.oob_prefix LIKE '%%%(drop)s%%'
                THEN
                    SET NEW.state = 0;
                END IF;

                /* Dropped packets */
                IF NEW.state=0
                THEN

                    /* Offenders */
                    IF EXISTS(SELECT * FROM offenders WHERE ip_addr = NEW.ip_saddr)
                    THEN
                        UPDATE offenders SET count = count + 1, last_time = UNIX_TIMESTAMP(NOW()) WHERE ip_addr = NEW.ip_saddr;
                    ELSE
                        INSERT INTO offenders VALUES (NEW.ip_saddr, UNIX_TIMESTAMP(NOW()), UNIX_TIMESTAMP(NOW()), 1);
                    END IF;

                    /* TCP_PORTS */
                    IF NEW.tcp_dport IS NOT NULL
                    THEN
                        IF EXISTS(SELECT * FROM tcp_ports WHERE tcp_dport = NEW.tcp_dport)
                        THEN
                            UPDATE tcp_ports SET count = count + 1, last_time = UNIX_TIMESTAMP(NOW()) WHERE tcp_dport = NEW.tcp_dport;
                        ELSE
                            INSERT INTO tcp_ports VALUES (NEW.tcp_dport, UNIX_TIMESTAMP(NOW()), UNIX_TIMESTAMP(NOW()), 1);
                        END IF;
                    END IF;

                    /* UDP_PORTS */
                    IF NEW.udp_dport IS NOT NULL
                    THEN
                        IF EXISTS(SELECT * FROM udp_ports WHERE udp_dport = NEW.udp_dport)
                        THEN
                            UPDATE udp_ports SET count = count + 1, last_time = UNIX_TIMESTAMP(NOW()) WHERE udp_dport = NEW.udp_dport;
                        ELSE
                            INSERT INTO udp_ports VALUES (NEW.udp_dport, UNIX_TIMESTAMP(NOW()), UNIX_TIMESTAMP(NOW()), 1);
                        END IF;
                    END IF;

                    /* USERSSTATS (dropped) */
                    IF NEW.user_id IS NOT NULL
                    THEN
                        IF EXISTS(SELECT * FROM usersstats WHERE user_id = NEW.user_id)
                        THEN
                            UPDATE usersstats SET bad_conns = bad_conns + 1, last_time = NOW() WHERE user_id = NEW.user_id;
                        ELSE /* bad_conns=1, good_conns=0 */
                            INSERT INTO usersstats VALUES (NEW.user_id, NEW.username, 1, 0,  NOW(), NOW());
                        END IF;
                    END IF;

                /* Allowed packets */
                ELSE
                    /* USERSSTATS (allowed) */
                    IF NEW.user_id IS NOT NULL
                    THEN
                        /* We don't add entry in usersstats if there isn't already a line
                         * for this user_id.
                         * Ask to Regit why :)
                         */
                        IF EXISTS(SELECT * FROM usersstats WHERE user_id = NEW.user_id)
                        THEN
                            /* Do not update timestamp, because we use it only for bad packets.. */
                            UPDATE usersstats SET good_conns = good_conns + 1 WHERE user_id = NEW.user_id;
                        END IF;
                    END IF;
                END IF;
            END;
        |

        DELIMITER ;
        """ % {'drop': options.drop}



if __name__ == '__main__':
    main()
