/* upsrw - simple client for read/write variable access (formerly upsct2)

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "common.h"

#include <netdb.h>
#include <netinet/in.h>
#include <sys/socket.h>

#include "upsclient.h"

struct list_t {
	char	*name;
	struct	list_t	*next;
};

static int old_do_enum(UPSCONN *ups, const char *varname)
{
	char	out[SMALLBUF], temp[SMALLBUF];

	snprintf(out, sizeof(out), "ENUM %s\n", varname);

	if (upscli_sendline(ups, out, strlen(out)) < 0) {
		fprintf(stderr, "Enumerate %s failed: %s\n", 
			varname, upscli_strerror(ups));

		return -1;	/* failed */
	}

	if (upscli_readline(ups, temp, sizeof(temp)) < 0) {
		fprintf(stderr, "Enumerate %s failed: %s\n",
			varname, upscli_strerror(ups));

		return -1;
	}

	if (strncmp(temp, "ENUM", 4) != 0) {
		fprintf(stderr, "Bogus reply from server for %s\n", varname);
		return -1;
	}

	if (upscli_readline(ups, temp, sizeof(temp)) < 0) {
		fprintf(stderr, "Can't get next ENUM value: %s\n",
			upscli_strerror(ups));

		return -1;
	}

	while (strcmp(temp, "END") != 0) {
		printf("Option: %s\n", strstr(temp, "\""));

		if (upscli_readline(ups, temp, sizeof(temp)) < 0) {
			fprintf(stderr, "Can't get next ENUM value: %s\n",
				upscli_strerror(ups));
	
			return -1;
		}
	}

	return 0;
}

static void usage(char *prog)
{
	printf("Network UPS Tools upsrw %s\n\n", UPS_VERSION);
        printf("usage: %s [-h]\n", prog);
	printf("       %s [-s <variable>] [-u <username>] [-p <password>] <ups>\n\n", prog);
	printf("Demo program to set variables within UPS hardware.\n");
        printf("\n");
        printf("  -h            display this help text\n");
	printf("  -s <variable>	specify variable to be changed\n");
	printf("		use -s VAR=VALUE to avoid prompting for value\n");
        printf("  -u <username> set username for command authentication\n");
        printf("  -p <password> set password for command authentication\n");
        printf("\n");
        printf("  <ups>         UPS identifier - myups@localhost, etc.\n");
	printf("\n");
	printf("Call without -s to show all possible read/write variables.\n");

	exit(0);
}

static void clean_exit(UPSCONN *ups, char *upsname, char *hostname, int code)
{
	if (upsname)
		free(upsname);

	if (hostname)
		free(hostname);

	upscli_disconnect(ups);

	exit(code);
}

static int old_set(UPSCONN *ups, const char *upsname, const char *varname, 
	const char *newval)
{
	char	temp[SMALLBUF];

	if (upsname)
		snprintf(temp, sizeof(temp), "SET %s@%s %s\n", 
			varname, upsname, newval);
	else
		snprintf(temp, sizeof(temp), "SET %s %s\n", varname, newval);

	if (upscli_sendline(ups, temp, strlen(temp)) < 0) {
		fprintf(stderr, "Can't set variable: %s\n", 
			upscli_strerror(ups));

		return 1;
	}

	if (upscli_readline(ups, temp, sizeof(temp)) < 0) {
		fprintf(stderr, "Set variable failed: %s\n", 
			upscli_strerror(ups));

		return 1;
	}

	return 0;
}

static int new_set(UPSCONN *ups, const char *upsname, const char *varname, 
	const char *newval)
{
	char	buf[SMALLBUF], enc[SMALLBUF];

	snprintf(buf, sizeof(buf), "SET VAR %s %s \"%s\"\n",
		upsname, varname, pconf_encode(newval, enc, sizeof(enc)));

	if (upscli_sendline(ups, buf, strlen(buf)) < 0) {
		fprintf(stderr, "Can't set variable: %s\n", 
			upscli_strerror(ups));

		return 1;
	}

	if (upscli_readline(ups, buf, sizeof(buf)) < 0) {
		fprintf(stderr, "Set variable failed: %s\n", 
			upscli_strerror(ups));

		return 1;
	}

	/* FUTURE: status cookies will tie in here */
	if (strncmp(buf, "OK", 2) != 0) {
		printf("Unexpected response from upsd: %s\n", buf);
		return 1;
	}

	return 0;
}

static int do_setvar(UPSCONN *ups, const char *varname, char *uin,
		const char *pass, char *upsname, char *hostname)
{
	char	newval[SMALLBUF], temp[SMALLBUF], user[SMALLBUF], *ptr;

	if (uin) {
		snprintf(user, sizeof(user), "%s", uin);
	} else {

		printf("Username: ");
		fgets(user, sizeof(user), stdin);

		/* deal with that pesky newline */
		if (strlen(user) > 1)
			user[strlen(user) - 1] = '\0';
	}

	/* leaks - use -p when running in valgrind */
	if (!pass) {
		pass = GETPASS("Password: " );

		if (!pass) {
			fprintf(stderr, "getpass failed: %s\n", 
				strerror(errno));
			return 1;
		}
	}

	/* Check if varname is in VAR=VALUE form */
	if ((ptr = strchr(varname, '=')) != NULL) {
		*ptr++ = 0;
		snprintf(newval, sizeof(newval), "%s", ptr);
	} else {
		printf("Enter new value for %s: ", varname);
		fflush(stdout);
		fgets(newval, sizeof(newval), stdin);
		newval[strlen(newval) - 1] = '\0';
	}

	snprintf(temp, sizeof(temp), "USERNAME %s\n", user);

	if (upscli_sendline(ups, temp, strlen(temp)) < 0) {
		fprintf(stderr, "Can't set username: %s\n", 
			upscli_strerror(ups));

		return 1;
	}

	if (upscli_readline(ups, temp, sizeof(temp)) < 0) {

		if (upscli_upserror(ups) == UPSCLI_ERR_UNKCOMMAND) {
			fprintf(stderr, "Set username failed due to an "
				"unknown command.\n");

			fprintf(stderr, "You probably need to upgrade upsd.\n");

			clean_exit(ups, upsname, hostname, 1);
		}

		fprintf(stderr, "Set username failed: %s\n",
			upscli_strerror(ups));

		return 1;
	}

	snprintf(temp, sizeof(temp), "PASSWORD %s\n", pass);

	if (upscli_sendline(ups, temp, strlen(temp)) < 0) {
		fprintf(stderr, "Can't set password: %s\n", 
			upscli_strerror(ups));

		return 1;
	}

	if (upscli_readline(ups, temp, sizeof(temp)) < 0) {
		fprintf(stderr, "Set password failed: %s\n", 
			upscli_strerror(ups));

		return 1;
	}

	/* no upsname = fallback until 2.0 */
	if (!upsname)
		return old_set(ups, upsname, varname, newval);

	/* also fallback for old variable names */
	if (!strchr(varname, '.'))
		return old_set(ups, upsname, varname, newval);

	return new_set(ups, upsname, varname, newval);
}	

static int old_print_rwlist(UPSCONN *ups, char *upsname, char *hostname)
{
	int	typelen;
	char	*v, *sp, *ptr, vars[SMALLBUF], out[SMALLBUF], temp[SMALLBUF],
		*type;

	printf("host: %s\n", hostname);

	if (upscli_getlist(ups, upsname, UPSCLI_LIST_RW, vars, 
		sizeof(vars)) < 0) {
		fprintf(stderr, "Unable to get variable list - %s\n",
		       upscli_strerror(ups));

		return 1;
	}

	if (strlen(vars) == 0) {
		fprintf(stderr, "Error: no read/write variables available\n");
		return 1;
	}

	v = vars;
	ptr = strchr(v, ' ');
	if (!ptr) {
		fprintf(stderr, "Broken string from server: %s\n", vars);
		return 1;
	}

	while (v != NULL) {
		ptr = strchr(v, ' ');
		if (ptr)
			*ptr++ = '\0';

		/* get description */
		snprintf(out, sizeof(out), "VARDESC %s\n", v);

		if (upscli_sendline(ups, out, strlen(out)) < 0) {
			fprintf(stderr, "Can't get description of %s: %s\n",
				v, upscli_strerror(ups));
			return 1;
		}

		if (upscli_readline(ups, temp, sizeof(temp)) < 0) {
			fprintf(stderr, "Can't get description of %s: %s\n",
				v, upscli_strerror(ups));
			return 1;
		}

		printf("[%s] %s ", v, strstr(temp, "\""));

		/* now get type */
		snprintf(out, sizeof(out),
			(upsname ? "VARTYPE %s@%s\n" : "VARTYPE %s\n"), v, upsname);

		if (upscli_sendline(ups, out, strlen(out)) < 0) {
			fprintf(stderr, "Can't get type of %s: %s\n",
				v, upscli_strerror(ups));
			return 1;
		}

		if (upscli_readline(ups, temp, sizeof(temp)) < 0) {
			fprintf(stderr, "Can't get type of %s: %s\n",
				v, upscli_strerror(ups));

			return 1;
		}

		/* TYPE (ENUM|STRING) <num> */

		sp = strchr(temp, ' ');
		if ((!sp) || (strncmp(temp, "TYPE ", 5) != 0)) {
			fprintf(stderr, "Unexpected response from server [%s]\n",
				temp);

			return 1;
		}

		*sp++ = '\0';

		type = sp;

		/* now make sure we can handle this vartype */

		sp = strchr(type, ' ');

		if (!sp) {
			fprintf(stderr, "Unexpected vartype from server [%s]\n",
				ptr);

			return 1;
		}
		
		*sp++ = '\0';

		typelen = strtol(sp, (char **) NULL, 10);

		printf("(%s:%d)\n", type, typelen);

		if (!strcmp(type, "ENUM"))
			if (old_do_enum(ups, v) != 0)
				return 1;

		if (!strcmp(type, "STRING")) {
			if (upscli_getvar(ups, upsname, v, temp, 
				sizeof(temp)) < 0) {

				fprintf(stderr, "Can't get value of %s: %s\n",
					temp, upscli_strerror(ups));

				return 1;
			}
				
			printf("Value: %s\n", temp);
		}
		
		v = ptr;

	}

	return 0;
}

static const char *get_data(const char *type, UPSCONN *ups, 
	const char *upsname, const char *varname)
{
	int	ret, numq, numa;
	char	**answer;
	const	char	*query[4];

	query[0] = type;
	query[1] = upsname;
	query[2] = varname;
	numq = 3;

	ret = upscli_get(ups, numq, query, &numa, &answer);

	if ((ret < 0) || (numa < numq))
		return NULL;

	/* <type> <upsname> <varname> <desc> */
	return answer[3];
}

static void do_string(UPSCONN *ups, const char *upsname, const char *varname)
{
	const	char	*val;

	val = get_data("VAR", ups, upsname, varname);

	if (!val) {
		fprintf(stderr, "do_string: can't get current value of %s\n",
			varname);
		return;
	}

	printf("Type: STRING\n");
	printf("Value: %s\n", val);
}

static void new_do_enum(UPSCONN *ups, const char *upsname, const char *varname)
{
	int	ret, numq, numa;
	char	**answer, *val;
	const	char	*query[4], *tmp;

	/* get current value */
	tmp = get_data("VAR", ups, upsname, varname);

	if (!tmp) {
		fprintf(stderr, "do_enum: can't get current value of %s\n",
			varname);
		return;
	}

	/* tmp is a pointer into answer - have to save it somewhere else */
	val = xstrdup(tmp);

	query[0] = "ENUM";
	query[1] = upsname;
	query[2] = varname;
	numq = 3;

	ret = upscli_list_start(ups, numq, query);

	if (ret < 0) {
		fprintf(stderr, "Error: %s\n", upscli_strerror(ups));
		return;
	}

	ret = upscli_list_next(ups, numq, query, &numa, &answer);

	printf("Type: ENUM\n");

	while (ret == 1) {

		/* ENUM <upsname> <varname> <value> */

		if (numa < 4) {
			fprintf(stderr, "Error: insufficient data "
				"(got %d args, need at least 4)\n", numa);

			free(val);
			return;
		}

		printf("Option: \"%s\"", answer[3]);

		if (!strcmp(answer[3], val))
			printf(" SELECTED");

		printf("\n");

		ret = upscli_list_next(ups, numq, query, &numa, &answer);
	}

	free(val);
}

static void do_type(UPSCONN *ups, const char *upsname, const char *varname)
{
	int	i, ret, numq, numa;
	char	**answer;
	const	char	*query[4];

	query[0] = "TYPE";
	query[1] = upsname;
	query[2] = varname;
	numq = 3;

	ret = upscli_get(ups, numq, query, &numa, &answer);

	if ((ret < 0) || (numa < numq)) {
		printf("Unknown type\n");	
		return;
	}

	/* TYPE <upsname> <varname> <type>... */
	for (i = 3; i < numa; i++) {

		if (!strcasecmp(answer[i], "ENUM")) {
			new_do_enum(ups, upsname, varname);
			return;
		}

		if (!strncasecmp(answer[i], "STRING:", 7)) {
			do_string(ups, upsname, varname);
			return;
		}

		/* ignore this one */
		if (!strcasecmp(answer[i], "RW"))
			continue;

		printf("Type: %s (unrecognized)\n", answer[i]);
	}

}

static void print_rw(UPSCONN *ups, const char *upsname, const char *varname)
{
	const	char	*tmp;

	printf("[%s]\n", varname);

	tmp = get_data("DESC", ups, upsname, varname);

	if (tmp)
		printf("%s\n", tmp);
	else
		printf("Description unavailable\n");

	do_type(ups, upsname, varname);

	printf("\n");
}	

static int new_print_rwlist(UPSCONN *ups, char *upsname, char *hostname)
{
	int	ret, numq, numa;
	const	char	*query[2];
	char	**answer;
	struct	list_t	*lhead, *llast, *ltmp, *lnext;

	llast = lhead = NULL;

	query[0] = "RW";
	query[1] = upsname;
	numq = 2;

	ret = upscli_list_start(ups, numq, query);

	if (ret < 0) {

		/* old upsd --> fall back on old LISTRW technique */
		if (upscli_upserror(ups) == UPSCLI_ERR_UNKCOMMAND)
			return old_print_rwlist(ups, upsname, hostname);

		fprintf(stderr, "Error: %s\n", upscli_strerror(ups));
		return 1;
	}

	ret = upscli_list_next(ups, numq, query, &numa, &answer);

	while (ret == 1) {

		/* RW <upsname> <varname> <value> */
		if (numa < 4) {
			fprintf(stderr, "Error: insufficient data "
				"(got %d args, need at least 4)\n", numa);
			return 1;
		}

		/* sock this entry away for later */

		ltmp = xmalloc(sizeof(struct list_t));
		ltmp->name = xstrdup(answer[2]);
		ltmp->next = NULL;

		if (llast)
			llast->next = ltmp;
		else
			lhead = ltmp;

		llast = ltmp;

		ret = upscli_list_next(ups, numq, query, &numa, &answer);
	}

	/* use the list to get descriptions and types */

	ltmp = lhead;

	while (ltmp) {
		lnext = ltmp->next;

		print_rw(ups, upsname, ltmp->name);

		free(ltmp->name);
		free(ltmp);
		ltmp = lnext;
	}

	return 0;
}

static int print_rwlist(UPSCONN *ups, char *upsname, char *hostname)
{
	/* no upsname = fall back on old methods until 2.0 */
	if (!upsname)
		return old_print_rwlist(ups, upsname, hostname);

	return new_print_rwlist(ups, upsname, hostname);
}

int main(int argc, char **argv)
{
	int	i, port, ret;
	char	*upsname, *hostname, *setvar, *prog;
	char	*password = NULL, *username = NULL;
	UPSCONN	ups;

	setvar = username = NULL;
	prog = argv[0];

	while ((i = getopt(argc, argv, "+s:p:u:V")) != EOF) {
		switch (i) {
		case 's':
			setvar = optarg;
			break;
		case 'p':
			password = optarg;
			break;
		case 'u':
			username = optarg;
			break;
		case 'V':
			printf("Network UPS Tools upsrw %s\n", UPS_VERSION);
			exit(0);
		default:
			usage(prog);
			break;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc < 1)
		usage(prog);

	upscli_splitname(argv[0], &upsname, &hostname, &port);

	if (upscli_connect(&ups, hostname, port, 0) < 0) {
		fprintf(stderr, "Can't connect: %s\n", upscli_strerror(&ups));
		clean_exit(&ups, upsname, hostname, 1);
	}

	/* setting a variable? */
	if (setvar) {
		ret = do_setvar(&ups, setvar, username, password, upsname,
			hostname);

		clean_exit(&ups, upsname, hostname, ret);
	}

	/* if not, get the list of supported read/write variables */
	ret = print_rwlist(&ups, upsname, hostname);

	clean_exit(&ups, upsname, hostname, ret);

	/* NOTREACHED */
	return 0;
}
