/*
   bestfortress.c - model specific routines for (very) old Best Power Fortress

   Copyright (C) 2002  Russell Kroll <rkroll@exploits.org> (skeleton)
             (C) 2002  Holger Dietze <holger.dietze@advis.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

/*
	anything commented is optional
	anything else is mandatory
*/

#include "main.h"

#define ENDCHAR		'\r'
#define IGNCHARS	" \n"

#if defined(__sgi) && ! defined(__GNUC__)
#define        inline  __inline
#endif

void instcmd (int auxcmd, int dlen, char *data);
void upsdrv_setvar (int, int, char *);

void upsdrv_initinfo(void)
{
	dstate_setinfo("ups.mfr", "Best Power");
	dstate_setinfo("ups.model", "Fortress");
	dstate_setinfo("battery.voltage.nominal", "24");

	/*addinfo (INFO_ALRM_OVERLOAD, "", 0, 0);*/ /* Flag */
	/*addinfo (INFO_ALRM_TEMP, "", 0, 0);*/ /* Flag */

	dstate_setinfo("ups.delay.shutdown", "10");	/* write only */	
	
	/* tunable via front panel: (european voltage level)
	   parameter		factory default  range
	   INFO_LOWXFER	196 V   p7=nnn   160-210
	   INFO_HIGHXFER	254 V   p8=nnn   215-274
	   INFO_LOBATTIME	2 min   p2=n     1-5
	   
	   comm mode    p6=0 dumb DONT USE (will lose access to parameter setting!)
	   		p6=1 B1200
			p6=2 B2400
			P6=3 B4800
			p6=4 B9600
	   maybe cycle through speeds to autodetect?

	   echo off     e0
	   echo on      e1
	*/
	dstate_setinfo("input.transfer.low", "");
	dstate_setflags("input.transfer.low", ST_FLAG_STRING | ST_FLAG_RW);
	dstate_setaux("input.transfer.low", 3);

	dstate_setinfo("input.transfer.high", "");
	dstate_setflags("input.transfer.high", ST_FLAG_STRING | ST_FLAG_RW);
	dstate_setaux("input.transfer.high", 3);

	dstate_setinfo("battery.runtime.low", "");
	dstate_setflags("battery.runtime.low", ST_FLAG_STRING | ST_FLAG_RW);
	dstate_setaux("battery.runtime.low", 3);

	upsh.instcmd = instcmd; 
	upsh.setvar = upsdrv_setvar;

	dstate_addcmd("shutdown.return");
	dstate_addcmd("load.off");
}

/* convert hex digit to int */
static inline int fromhex (char c)
{
	return (c >= '0' && c <= '9') ? c - '0'
		: (c >= 'A' && c <= 'F') ? c - 'A' + 10
		: (c >= 'a' && c <= 'f') ? c - 'a' + 10
		: 0;
}

/* do checksumming on UPS response */
static int checksum (char * s)
{
	int i;
	int sum;
	for (i = 40, sum = 0; s[0] && s[1] && i > 0; i--, s += 2) {
		sum += (fromhex (s[0]) << 4) + fromhex (s[1]);
	}
	return sum;
}

/* set info to integer value */
static inline void setinfo_int (int id, char * s, int len)
{
	char buf[10];
	
	if (len > sizeof(buf)) len = sizeof(buf)-1;
	strncpy (buf, s, len);
	buf[len] = 0;
	setinfo (id, "%d", atoi (buf));
}

/* set info to integer value (for runtime remaining)
   value is expressed in minutes, but desired in seconds
 */
static inline void setinfo_int_minutes (int id, char * s, int len)
{
	char buf[10];
	
	if (len > sizeof(buf)) len = sizeof(buf)-1;
	strncpy (buf, s, len);
	buf[len] = 0;
	setinfo (id, "%d", 60*atoi (buf));
}

/* set info to float value */
static inline void setinfo_float (int id, char * fmt, char * s, int len, double factor)
{
	char buf[10];
	if (len > sizeof(buf)) len = sizeof(buf)-1;
	strncpy (buf, s, len);
	buf[len] = 0;
	setinfo (id, fmt, factor * (double)atoi (buf));
}

/* read out UPS and store info */
void upsdrv_updateinfo(void)
{
	char temp[256];
	char *p;
	
	int checksum_ok, is_online=1, is_off, low_batt, trimming, boosting;
	
	upssend ("f\r");
	
	do {
		if (upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS) <= 0) {
			upsflushin (0, 0, "\r ");
			upssend ("f\r");
		}
	} while (temp[0] == 0);
	/* setinfo (INFO_, ""); */
	p = temp;
	
	/*syslog (LOG_DAEMON | LOG_NOTICE,"ups: got '%s'\n", p);*/
	/* status example:
	   |Vi||Vo|    |Io||Psou|    |Vb||f| |tr||Ti|            CS
	   000000000001000000000000023802370000000200004700000267500000990030000000000301BD
	   1    1    2    2    3    3    4    4    5    5    6    6    7    7   78
	   0    5    0    5    0    5    0    5    0    5    0    5    0    5    0    5   90
	*/

	/* last bytes are a checksum:
	   interpret response as hex string, sum of all bytes must be zero
	*/
	checksum_ok = (checksum (p) & 0xff) == 0;

	if (!checksum_ok) {
		dstate_datastale();
		return;
	}
	
	setinfo_int (INFO_UTILITY, p+24,4);
	setinfo_int (INFO_OUTVOLT, p+28,4);
	setinfo_float (INFO_BATTVOLT, "%.1f", p+50,4, 0.1);
	setinfo_float (INFO_CURRENT, "%.1f", p+36,4, 0.1);
	setinfo_int (INFO_OUT_VA, p+40,6);
	setinfo_float (INFO_ACFREQ, "%.1f", p+54,3, 0.1);
	setinfo_int_minutes (INFO_BATT_RUNTIME, p+58,4);
	setinfo_int (INFO_UPSTEMP, p+62,4);
	
	is_online = p[17] == '0';
	low_batt = fromhex(p[21]) & 8 || fromhex(p[20]) & 1;
	is_off = p[11] == '0';
	trimming = p[33] == '1';
	boosting = 0; /* FIXME, don't know which bit gets set
			 (brownouts are very rare here and I can't
			 simulate one) */

	setinfo (INFO_STATUS, "%s%s%s",
		 is_online ? (is_off ? "OFF " : "OL ") : "OB ",
		 low_batt ? "LB " : "",
		 trimming ? "TRIM" : boosting ? "BOOST" : "");

	/* setinfo(INFO_STATUS, "%s%s",
	 *	(util < lownorm) ? "BOOST ", "",
	 *	(util > highnorm) ? "TRIM ", "",
	 *	((flags & TIOCM_CD) == 0) ? "" : "LB ",
	 *	((flags & TIOCM_CTS) == TIOCM_CTS) ? "OB" : "OL");
	 */

	dstate_dataok();
}


/* Parameter setting */

/* all UPS tunable parameters are set with command
   'p%d=%s'
*/
int setparam (int parameter, int dlen, char * data)
{
	char reply[80];
	upssend ("p%d=%*s\r", parameter, dlen, data);
	if (upsrecv (reply, sizeof(reply), ENDCHAR, "") < 0) return 0;
	return strncmp (reply, "OK", 2) == 0;
}

/* ups_setsuper: set super-user access
   (allows setting variables)
*/
static void ups_setsuper (int super)
{
	setparam (999, super ? 4 : 0, super ? "2639" : "");
}

/* sets whether UPS will reapply power after it has shut down and line
 * power returns.
 */
static void autorestart (int restart)
{
	ups_setsuper (1);
	setparam (1, 1, restart ? "1" : "0");
	ups_setsuper (0);
}

/* set UPS parameters */
void upsdrv_setvar (int info, int len, char * data)
{
	int parameter;
	switch (info) {
	case INFO_LOWXFER:
		parameter = 7;
		break;
	case INFO_HIGHXFER:
		parameter = 8;
		break;
	case INFO_LOBATTIME:
		parameter = 2;
		break;
	default:
		return;
	}
	ups_setsuper (1);
	if (setparam (parameter, len, data)) {
		setinfo (info, "%*s", len, data);
	}
	ups_setsuper (0);
}

void upsdrv_shutdown(void)
{
	const	char	*grace;

	grace = getdata(INFO_PDNGRACE);

	if (!grace)
		grace = "1"; /* apparently, OFF0 does not work */

	printf ("shutdown in %s seconds\n", grace);
	/* make power return when utility power returns */
	autorestart (1);
	upssend ("OFF%s\r", grace);
	/* I'm nearly dead, Jim */
	/* OFF will powercycle when line power is available again */
}

void instcmd (int auxcmd, int dlen, char *data)
{
	const char *p;
	
	switch (auxcmd) {
	case CMD_OFF:
		printf ("powering off\n");
		autorestart (0);
		upssend ("OFF1\r");
		break;
	case CMD_SHUTDOWN:
		p = getdata (INFO_PDNGRACE);
		if (!p) p = "1";
		printf ("shutdown in %s seconds\n", p);
		autorestart (1);
		upssend ("OFF%s\r", p);
		break;
	default:
		upslogx(LOG_INFO, "instcmd: unknown type 0x%04x", auxcmd);
	}
}

void upsdrv_help(void)
{
}

/* list flags and values that you want to receive via -x */
void upsdrv_makevartable(void)
{
	/* allow '-x xyzzy' */
	/* addvar(VAR_FLAG, "xyzzy", "Enable xyzzy mode"); */
	
	/* allow '-x foo=<some value>' */
	/* addvar(VAR_VALUE, "foo", "Override foo setting"); */
	addvar (VAR_VALUE, "speed", "serial line speed");
}

void upsdrv_banner(void)
{
	printf("Network UPS Tools - Best Fortress UPS driver 0.01 (%s)\n\n", UPS_VERSION);
}

struct {
	char * val;
	speed_t speed;
} speed_table[] = {
	{"1200", B1200},
	{"2400", B2400},
	{"4800", B4800},
	{"9600", B9600},
	{NULL, B1200},
};

void upsdrv_initups(void)
{
	speed_t speed = B1200;

	char * speed_val = getval ("speed");

	if (speed_val) {
		int i;
		for (i=0; speed_table[i].val; i++) {
			if (strcmp (speed_val, speed_table[i].val) == 0)
				break;
		}
		speed = speed_table[i].speed;
	}
	
	open_serial(device_path, speed);
	/* TODO: probe ups type */
	
	/* the upsh handlers can't be done here, as they get initialized
	 * shortly after upsdrv_initups returns to main.
	 */
}

void upsdrv_cleanup(void)
{
}
