/* bestups.c - model specific routines for Best-UPS Fortress models

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "main.h"

#define ENDCHAR  13	/* replies end with CR */
#define UPSDELAY  5
#define MAXTRIES 10

#define DRV_VERSION "0.60"

static	char	*modelname, *va;
static	float	lowvolt = 0, highvolt = 0, voltrange = 0;
static	int	linenorm = 0, poll_failures = 0;
static	int	inverted_bypass_bit = 0;

static void setmodel(const char *abbr, const char *va)
{
	if (!strcmp(abbr, "FOR")) {
		dstate_setinfo("ups.mfr", "%s", "Best Power");
		dstate_setinfo("ups.model", "Fortress %s", va);
		return;
	}

	if (!strcmp(abbr, "FTC")) {
		dstate_setinfo("ups.mfr", "%s", "Best Power");
		dstate_setinfo("ups.model", "Fortress Telecom %s", va);
		return;
	}

	if (!strcmp(abbr, "PRO")) {
		dstate_setinfo("ups.mfr", "%s", "Best Power");
		dstate_setinfo("ups.model", "Patriot Pro %s", va);
		inverted_bypass_bit = 1;
		return;
	}

	if (!strcmp(abbr, "PR2")) {
		dstate_setinfo("ups.mfr", "%s", "Best Power");
		dstate_setinfo("ups.model", "Patriot Pro II %s", va);
		inverted_bypass_bit = 1;
		return;
	}

	if (!strcmp(abbr, "325")) {
		dstate_setinfo("ups.mfr", "%s", "Sola Australia");
		dstate_setinfo("ups.model", "Sola 325 %s", va);
		return;
	}

	if (!strcmp(abbr, "520")) {
		dstate_setinfo("ups.mfr", "%s", "Sola Australia");
		dstate_setinfo("ups.model", "Sola 520 %s", va);
		return;
	}

	if (!strcmp(abbr, "620")) {
		dstate_setinfo("ups.mfr", "%s", "Sola Australia");
		dstate_setinfo("ups.model", "Sola 620 %s", va);
		return;
	}

	if (!strcmp(abbr, "AX1")) {
		dstate_setinfo("ups.mfr", "%s", "Best Power");
		dstate_setinfo("ups.model", "Axxium Rackmount %s", va);
		return;
	}

	dstate_setinfo("ups.mfr", "%s", "Unknown");
	dstate_setinfo("ups.model", "Unknown %s (%s)", abbr, va);

	printf("Unknown model detected - please report this ID: '%s'\n", abbr);
}

int instcmd(const char *cmdname, const char *extra)
{
	if (!strcasecmp(cmdname, "test.battery.stop")) {
		upssendchar('C');
		upssendchar('T');
		upssendchar(13);

		return STAT_INSTCMD_HANDLED;
	}

	if (!strcasecmp(cmdname, "test.battery.start")) {
		upssendchar('T');
		upssendchar(13);

		return STAT_INSTCMD_HANDLED;
	}

	upslogx(LOG_NOTICE, "instcmd: unknown command [%s]", cmdname);
	return STAT_INSTCMD_UNKNOWN;
}

void upsdrv_initinfo(void)
{
	dstate_setinfo("driver.version.internal", "%s", DRV_VERSION);
	dstate_addcmd("test.battery.start");
	dstate_addcmd("test.battery.stop");

	setmodel(modelname, va);
	printf("Detected %s %s on %s\n", dstate_getinfo("ups.mfr"), 
		dstate_getinfo("ups.model"), device_path);

	upsh.new_instcmd = instcmd;

	/* paranoia - cancel any shutdown that might already be running */
	upssendchar('C');
	upssendchar(13);
}

static void setup_serial(void)
{	
	struct	termios	tio;

	if (tcgetattr(upsfd, &tio) == -1)
		fatal("tcgetattr");

	tio.c_iflag = IXON | IXOFF;
	tio.c_oflag = 0;
	tio.c_cflag = (CS8 | CREAD | HUPCL | CLOCAL);
	tio.c_lflag = 0;
	tio.c_cc[VMIN] = 1;
	tio.c_cc[VTIME] = 0; 

#ifdef HAVE_CFSETISPEED
	cfsetispeed(&tio, B2400);
	cfsetospeed(&tio, B2400);
#else
#error This system lacks cfsetispeed() and has no other means to set the speed
#endif

	if (tcsetattr(upsfd, TCSANOW, &tio) == -1)
		fatal("tcsetattr");
}

static void ups_sync(void)
{
	char	buf[256];
	int	i, ret;

	printf("Syncing with UPS: ");
	fflush(stdout);

	for (i = 0; i < MAXTRIES; i++) {
		printf(".");
		fflush(stdout);
		upssend("%s", "\rQ1\r");
		printf(".");
		fflush(stdout);
		sleep(UPSDELAY);
		printf(".");
		fflush(stdout);

		ret = upsrecv(buf, sizeof(buf), ENDCHAR, "");

		/* return once we get something that looks usable */
		if ((ret > 0) && (buf[0] == '(')) {
			printf(" done\n");
			return;
		}
	}

	fatalx("\nFailed - giving up...");
}

static int ups_on_line(void)
{
	int	i, ret;
	char	temp[256], stat[32];

	printf("Checking line status: ");
	fflush(stdout);

	for (i = 0; i < MAXTRIES; i++) {
		upssend("\rQ1\r");
		printf(".");
		fflush(stdout);
		sleep(UPSDELAY);
		printf(".");
		fflush(stdout);

		ret = upsrecv(temp, sizeof(temp), ENDCHAR, "");

		/* Q1 must return 46 bytes starting with a ( */
		if ((ret > 0) && (temp[0] == '(') && (strlen(temp) == 46)) {

			sscanf(temp, "%*s %*s %*s %*s %*s %*s %*s %s", stat);

			if (stat[0] == '0')
				return 1;	/* on line */

			return 0;	/* on battery */
		}
	}

	upslogx(LOG_ERR, "Status read failed: assuming on battery");

	return 0;	/* on battery */
}

void upsdrv_shutdown(void)
{
	printf("The UPS will shut down in approximately one minute.\n");

	if (ups_on_line())
		printf("The UPS will restart in about one minute.\n");
	else
		printf("The UPS will restart when power returns.\n");

	upssend("%s", "S01R0001\r");
}

static int get_ident(char *buf, size_t bufsize)
{
	int	i, ret;

	printf("Identifying UPS: ");
	fflush(stdout);

	for (i = 0; i < MAXTRIES; i++) {
		printf(".");
		fflush(stdout);

		upssend("%s", "\r\rID\r");
		printf(".");

		fflush(stdout);
		sleep(UPSDELAY);
		printf(".");
		fflush(stdout);

		ret = upsrecv(buf, bufsize, ENDCHAR, "");

		/* buf must start with ( and be in the range [25-27] */
		if ((ret > 0) && (buf[0] != '(') && (strlen(buf) >= 25) &&
			(strlen(buf) <= 27)) {

			printf(" done\n");
			return 1;
		}
	}

	upslogx(LOG_INFO, "Giving up on hardware detection after %d tries",
		MAXTRIES);

	return 0;
}

static void ups_ident(void)
{
	char	buf[256], *ptr, *com;
	int	i;

	if (!get_ident(buf, sizeof(buf)))
		fatalx("Unable to get initial hardware info string");

	modelname = va = NULL;

	/* FOR,750,120,120,20.0,27.6 */
	ptr = buf;

	for (i = 0; i < 6; i++) {
		com = strchr(ptr, ',');

		if (com)
			*com = '\0';

		switch (i) {
			case 0: modelname = xstrdup(ptr);
				break;
			case 1: va = xstrdup(ptr);
				break;
			case 2: linenorm = atoi(ptr);
				break;
			case 4: lowvolt = atof(ptr);
				break;
			case 5: highvolt = atof(ptr);
				voltrange = highvolt - lowvolt;
				break;
			default:
				break;
		}

		if (com)
			ptr = com + 1;
	}
}

static void pollfail(char *why)
{
	poll_failures++;

	/* ignore the first few since these UPSes tend to drop characters */
	if (poll_failures == 3)
		upslogx(LOG_ERR, why);

	return;
}

void upsdrv_updateinfo(void)
{
	char	involt[16], outvolt[16], loadpct[16], acfreq[16], 
		battvolt[16], upstemp[16], stat[16], buf[256];
	float	bvoltp;
	int	ret;
	static	int	pass = 0;

	/* this UPS needs a huge delay on queries, so chop up the updates */

	/* half the time, send the Q1, the other half, read it */

	if (pass == 0) {
		upssend("%s", "\rQ1\r");
		pass = 1;
		return;
	}

	pass = 0;

	ret = upsrecv(buf, sizeof(buf), ENDCHAR, "");

	if (strlen(buf) < 46) {
		pollfail("Poll failed: short read from UPS");
		dstate_datastale();
		return;
	}

	if (strlen(buf) > 46) {
		pollfail("Poll failed: oversized read from UPS");
		dstate_datastale();
		return;
	}

	if (buf[0] != '(') {
		pollfail("Poll failed: invalid start character");
		dstate_datastale();
		return;
	}

	/* only say this if it got high enough to log a failure note */
	if (poll_failures >= 3)
		upslogx(LOG_NOTICE, "UPS poll succeeded");

	poll_failures = 0;

	sscanf(buf, "%*c%s %*s %s %s %s %s %s %s", involt, outvolt, 
		loadpct, acfreq, battvolt, upstemp, stat);
	
	bvoltp = ((atof (battvolt) - lowvolt) / voltrange) * 100.0;

	if (bvoltp > 100.0)
		bvoltp = 100.0;

	dstate_setinfo("battery.voltage", "%s", battvolt);
	dstate_setinfo("input.voltage", "%s", involt);
	dstate_setinfo("output.voltage", "%s", outvolt);
	dstate_setinfo("ups.load", "%s", loadpct);
	dstate_setinfo("input.frequency", "%s", acfreq);
	dstate_setinfo("ups.temperature", "%s", upstemp);
	dstate_setinfo("battery.charge", "%02.1f", bvoltp);

	status_init();

	if (stat[0] == '0') {
		status_set("OL");		/* on line */

		/* only allow these when OL since they're bogus when OB */

		if (stat[2] == (inverted_bypass_bit ? '0' : '1')) {
			/* boost or trim in effect */
			if (atof(involt) < atof(outvolt))
				status_set("BOOST");

			if (atof(involt) > atof(outvolt))
				status_set("TRIM");
		}

	} else {
		status_set("OB");		/* on battery */
	}

	if (stat[1] == '1')
		status_set("LB");		/* low battery */

	status_commit();
	dstate_dataok();
}

void upsdrv_help(void)
{
}

void upsdrv_makevartable(void)
{
}

void upsdrv_banner(void)
{
	printf("Network UPS Tools - Best UPS driver %s (%s)\n", 
		DRV_VERSION, UPS_VERSION);
}

void upsdrv_initups(void)
{
	upssend_delay = 100000;
	open_serial(device_path, B2400);
	setup_serial();

	/* don't let upscommon warn about it since it happens way too often */
	/* this driver does its own checking with better handling */
        flag_timeoutfailure = -1;

	ups_sync();
	ups_ident();
}

void upsdrv_cleanup(void)
{
}
