/*                               -*- Mode: C -*- 
 * sec.c -- new style SEC UPS Protocol driver
 * 
 * Copyright (C) 2001 John Marley
 * The author can be contacted at: John.Marley@alcatel.com.au
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Author          : John Marley
 * Created On      : Wed Mar 28 14:48:26 2001
 * Last Modified By: Jules Taplin
 * Last Modified On: Sunday August 11, 2002  19:30 GMT
 * Update Count    : 198
 * Status          : Unknown, Use with caution!
 * $Locker:  $
 * $Log: sec.c,v $
 *
 * Revision 1.6  11 August 2002  Jules Taplin.  jules@netsitepro.co.uk
 * increased size of buffer that holds data returned from UPS during 
 * probing of baud rate in function setup_serial().
 *
 * Move function call nolongertimeout() in sec_upsrecv outside of loop
 * so the timeout is reset only when a valid message start character
 * from the UPS is received.
 *
 * Revision 1.5   10 August 2002 Eric Lawson.  elawson@inficad.com
 * Fix setup_serial so that any valid response from the UPS is accepted
 * as having found the correct serial port baud rate.  During serial port
 * baud rate probing, some UPS units get confused by commands sent at
 * the wrong baud rate and always reports a command failed the first time
 * the command is sent at the correct baud rate.
 *
 * Revision 1.4  17 July 2002 Eric Lawson.  elawson@inficad.com
 * added shutdown feature
 *
 * Revison 1.3   15 July 2002 Eric Lawson.  elawson@inficad.com
 * 
 * Use actual sec protocol command instead of bogus command when
 * probing for a ups on the serial port in the setup_serial function.
 *
 * Zero msglen before calling sec_cmd the 2nd time in upsdrv_initinfo
 * function.  This fixes driver's use of sec protocol parameters 
 * numbered 47 and up.
 *
 * Revision 1.2  2001/05/08 03:05:21  marleyj
 * Added synthetic variables INFO_STATUS, INFO_ACFREQ, INFO_UTILITY,
 * INFO_CURRENT, INFO_LOADPCT, INFO_LOADPWR and INFO_OUTVOLT.
 *
 * Revision 1.1  2001/05/02 04:54:19  marleyj
 * Initial revision
 */

#define SEC_DRIVER_VERSION	"$Revision: 1.4 $"

#include "main.h"
#include <sys/file.h>

extern const char *device_path;
extern char upssend_endchar;

#include "sec.h"

/* #define ENDCHAR	'\r' */
/* #define IGNCHARS	""   */

/*
 * upsread(buf, count)
 *
 * Attempts to read up to count chars from the UPS in the buffer buf.
 *
 * Returns count or -1 if there was an error.
 */
int upsread(char *buf, int count)
{
    int	 ret,c = 0;
    char *b,in;

    b = buf;
    while (c < count) { 
	ret = read(upsfd, &in, 1);
	if (ret < 1) return (-1);
	*b++ = in;
	c++;
    }

    return(c);
}


/*
 * sec_upsrecv(buf)
 *
 * Read a SEC format message (^<cmdchar><len><data>) from the UPS
 * and store in buf.
 *
 * Return -2 if failed to read valid response
 * Return -1 if command failed (^0)
 * Return 0 if command succeeded (^1)
 * Return len if data returned
 *
 * Need to read one char at a time...
 */
int sec_upsrecv (char *buf)
{
    char	recvbuf[512], lenbuf[4], in;
    int		ret, len;
    struct 	sigaction sa;
    sigset_t 	sec_sigmask;

    memset(recvbuf, '\0', sizeof(recvbuf));
 
    sa.sa_handler = timeout;
    sigemptyset (&sec_sigmask);
    sa.sa_mask = sec_sigmask;
    sa.sa_flags = 0;
    sigaction (SIGALRM, &sa, NULL);
 
    alarm (3);

    upsdebugx(1, "sec_upsrecv...");

    /* look for the startchar */
    for (;;) {
	ret = read (upsfd, &in, 1);
	if (ret > 0) {
	    if (in == SEC_MSG_STARTCHAR) break;
	}
	else {
	    alarm (0);
	    signal (SIGALRM, SIG_IGN);
	    upsdebugx(1, " FAILED to find start char %c",SEC_MSG_STARTCHAR);
	    return (-2);
	}
    }

	nolongertimeout();

    /* we found a msg, which one? */
    upsdebugx(1, " found start char...");
    ret = read (upsfd, &in, 1);
	if (ret > 0) {
	if (in == SEC_DATAMSG) {
	    upsdebugx(1, " got a %c!",SEC_DATAMSG);
	    /* data being returned - get the length */
	    ret = upsread(lenbuf, 3);
	    if (ret < 3) {
		alarm (0);
		signal (SIGALRM, SIG_IGN);
		upsdebugx(1, " OOPS only %d of 3 chars read (%c)",ret,*lenbuf);
		return (-2);
	    }
	    len = atoi(lenbuf);
	    upsdebugx(1, " UPS returning %d bytes of data...",len);
	    ret = upsread (recvbuf, len);
	    alarm (0);
	    signal (SIGALRM, SIG_IGN);
	    if (ret == len) {
		strncpy(buf, recvbuf, len);
		upsdebugx(1, " OK, read %d bytes of data",len);
		return (len);
	    } else {
		upsdebugx(1, " OOPS less than %d chars read",len);
		return (-2);
	    }
	}
    }
    alarm (0);
    signal (SIGALRM, SIG_IGN);
    if (ret > 0) {
	switch (in) {
	case '0':
	    upsdebugx(1, "UPS returned: command failed");
	    return(-1);
	case '1':
	    upsdebugx(1, "UPS returned: command succeeded!");
	    return(0);
	default:
	    return(-2);
	}
    }
 
    return (-2);
}


/*
 * sec_cmd(mode, command, buffer, length)
 *
 * Sends mode command to the UPS with the given data.
 *
 * Returns -1 if command fails
 *          0 if command succeeds, but returns no data
 *          length of returned data if data is returned
 */
int sec_cmd(const char mode, const char *command, char *msgbuf, int *buflen)
{
    char msg[256];
    int ret;

    memset(msg, 0, sizeof(msg));

    /* create the message string */
    if (*buflen > 0) {
	sprintf(msg, "%c%c%03d%s%s", SEC_MSG_STARTCHAR,
		mode, (*buflen)+3, command, msgbuf);
    }
    else {
	sprintf(msg, "%c%c003%s", SEC_MSG_STARTCHAR,
		mode, command);
    }	
    upsdebugx(1, "PC-->UPS: \"%s\"",msg);
    ret = upssend(msg);
    upsdebugx(1, " send returned: %d",ret);

    if (ret == -1) return -1;

    ret = sec_upsrecv(msg);
    upsdebugx(1, " receive returned: %d",ret);

    if (ret < 0) return -1;

    if (ret > 0) {
	strncpy(msgbuf, msg, ret);
	upsdebugx(1, "UPS<--PC: \"%s\"",msg);
    }
/*    *(msgbuf+ret) = '\0';*/

    *buflen = ret;
    return ret;
}

/* set up the serial connection */
void setup_serial(const char *port)
{
    char temp[512];
    int i,ret;

    /* The SEC protocol alows for different baud rates.  My daddy told me
       "Never assume ...", so we try the different rates to see which works. */

    for (i=0; i<5; i++) {
	upsdebugx(1, "Trying to connect at %d baud",baud_rates[i].name);
	open_serial(port, baud_rates[i].rate);
	upsdebugx(2, " sending probing command...");
	upssend("^P003MAN");
	upsdebugx(2, " reading reply...");
	ret = sec_upsrecv(temp);
	if (ret >= -1) break;
	upsdebugx(2, " no connection.");
	unlockport(upsfd, port);
    }
    if (i == 5) {
	printf("Can't talk to UPS on port %s!\n",port);
	printf("Check the cabling and portname and try again\n");
	exit (1);
    }
}


/*
 * addquery(cmd, field, varnum)
 *
 * Records that sec variable <varnum> is supported by this UPS and should be
 * queried as part of the regular update.  We need to record which command
 * (cmd) to send, and which <field> corresponds to the variable.
 */

void addquery(char *cmd, int field, int varnum)
{
    int q;

    for (q=0; q<SEC_QUERYLIST_LEN; q++) {
	if (sec_querylist[q].command == NULL) {
	    /* command has not been recorded yet */
	    sec_querylist[q].command = cmd;
	    upsdebugx(1, " Query %d is %s",q,cmd);
	}
	if (sec_querylist[q].command == cmd) {
	    sec_querylist[q].varnum[field-1] = varnum;
	    upsdebugx(1, " Querying varnum %d",varnum);
	    break;
	}
    }
}


/*
 * sec_setinfo(varnum, value)
 *
 * Update variable number <varnum> to value <value> in the info array.
 */

void sec_setinfo(int varnum, char *value)
{
	upsdebugx(1, "Updating variable %d (%s), new value is \"",
		varnum, sec_varlist[varnum].name);

    if (sec_varlist[varnum].flags & FLAG_ENUM) {
	upsdebugx(1, "%s\" (ENUM)",
		    sec_enumdata[sec_varlist[varnum].edi + atoi(value)].value);
	setinfo(sec_varlist[varnum].infotag,
		"%s",
		sec_enumdata[sec_varlist[varnum].edi + atoi(value)].value);
    }
    else if (sec_varlist[varnum].flags & FLAG_STRING) {
	upsdebugx(1, "%s\" (STRING)", value);
	setinfo(sec_varlist[varnum].infotag,
		"%s",
		value);
    }
    else {
	if (sec_varlist[varnum].unit != 1) {
		upsdebugx(1, "%.1f\" (Float)",
			atof(value) / sec_varlist[varnum].unit);
	    setinfo(sec_varlist[varnum].infotag,
		    "%.1f",
		    atof(value) / sec_varlist[varnum].unit);
	}
	else {
		upsdebugx(1, "%d\" (NUM)", atoi(value));
	    setinfo(sec_varlist[varnum].infotag,
		    "%d",
		    atoi(value));
	}	    
    }
}


/*
 * update_pseudovars
 *
 * There are a number of non-SEC variables that are functions of the real
 * variables.  We update them here.
 *
 * OFF   - UPS is off          - INFO_ALRM_OUTPUTOFF is "Output off"
 * OL    - UPS is online       - INFO_OUT_SOURCE is "Normal"
 * OB    - UPS is on battery   - INFO_OUT_SOURCE is "On Battery"
 * BY    - UPS is on bypass    - INFO_OUT_SOURCE is "On Bypass"
 *
 * OVER  - UPS is overloaded   - INFO_ALRM_OVERLOAD is "UPS Overloaded"
 * LB    - UPS battery is low  - INFO_BATT_STATUS is "Battery Low"
 * RB    - UPS replace battery - INFO_BATT_COND is "Replace"
 *
 * TRIM  - UPS is trimming     - INFO_OUT_SOURCE is "Reducing"
 * BOOST - UPS is boosting     - INFO_OUT_SOURCE is "Boosting"
 *
 * FSD   - UPS is shutdown     - INFO_ALRM_SYSOFF is "System off"
 */

void update_pseudovars( void )
{
	const char  *v;
	int   n;
	float fsum;

	upsdebugx(1, "Synthesizing INFO_STATUS...");
	/* the STATUS variable */

	status_init();

	if (supported(INFO_ALRM_OUTPUTOFF) &&
		strcmp(getdata(INFO_ALRM_OUTPUTOFF), "Output off") == 0) {
		status_set("OFF");
	}

	if (supported(INFO_OUT_SOURCE)) {
		if (strcmp(getdata(INFO_OUT_SOURCE), "Normal") == 0)
			status_set("OL");

		if (strcmp(getdata(INFO_OUT_SOURCE), "On Battery") == 0)
			status_set("OB");

		if (strcmp(getdata(INFO_OUT_SOURCE), "On Bypass") == 0)
			status_set("BYPASS");

		if (strcmp(getdata(INFO_OUT_SOURCE), "Reducing") == 0)
			status_set("TRIM");

		if (strcmp(getdata(INFO_OUT_SOURCE), "Boosting") == 0)
			status_set("BOOST");
	}

	if (supported(INFO_ALRM_OVERLOAD) &&
		strcmp(getdata(INFO_ALRM_OVERLOAD), "UPS Overloaded") == 0)
		status_set("OVER");

	if (supported(INFO_BATT_STATUS) &&
		strcmp(getdata(INFO_BATT_STATUS), "Battery Low") == 0)
		status_set("LB");

	if (supported(INFO_BATT_COND) &&
		strcmp(getdata(INFO_BATT_COND), "Replace") == 0)
		status_set("RB");

	if (supported(INFO_ALRM_SYSOFF) &&
		strcmp(getdata(INFO_ALRM_SYSOFF), "System off") == 0)
		status_set("RB");

	status_commit();

    upsdebugx(1, "Synthesizing averages...");
    /* Average stats */
    if (supported(INFO_ACFREQ)) {
	fsum = 0.0; n = 0;
	v = getdata(INFO_IN_ACFREQ1); if (v != NULL) fsum += atof(v), n++;
	v = getdata(INFO_IN_ACFREQ2); if (v != NULL) fsum += atof(v), n++;
	v = getdata(INFO_IN_ACFREQ3); if (v != NULL) fsum += atof(v), n++;
	setinfo(INFO_ACFREQ, "%.1f", fsum / n);
    }
    if (supported(INFO_UTILITY)) {
	fsum = 0.0; n = 0;
	v = getdata(INFO_IN_VOLT1); if (v != NULL) fsum += atof(v), n++;
	v = getdata(INFO_IN_VOLT2); if (v != NULL) fsum += atof(v), n++;
	v = getdata(INFO_IN_VOLT3); if (v != NULL) fsum += atof(v), n++;
	setinfo(INFO_UTILITY, "%.1f", fsum / n);
    }
    if (supported(INFO_CURRENT)) {
	fsum = 0.0; n = 0;
	v = getdata(INFO_OUT_CURRENT1); if (v != NULL) fsum += atof(v), n++;
	v = getdata(INFO_OUT_CURRENT2); if (v != NULL) fsum += atof(v), n++;
	v = getdata(INFO_OUT_CURRENT3); if (v != NULL) fsum += atof(v), n++;
	setinfo(INFO_CURRENT, "%.1f", fsum / n);
    }
    if (supported(INFO_LOADPCT)) {
	fsum = 0.0; n = 0;
	v = getdata(INFO_OUT_LOADPCT1); if (v != NULL) fsum += atof(v), n++;
	v = getdata(INFO_OUT_LOADPCT2); if (v != NULL) fsum += atof(v), n++;
	v = getdata(INFO_OUT_LOADPCT3); if (v != NULL) fsum += atof(v), n++;
	setinfo(INFO_LOADPCT, "%.1f", fsum / n);
    }
    if (supported(INFO_LOADPWR)) {
	fsum = 0.0; n = 0;
	v = getdata(INFO_OUT_LOADPWR1); if (v != NULL) fsum += atof(v), n++;
	v = getdata(INFO_OUT_LOADPWR2); if (v != NULL) fsum += atof(v), n++;
	v = getdata(INFO_OUT_LOADPWR3); if (v != NULL) fsum += atof(v), n++;
	setinfo(INFO_LOADPWR, "%.1f", fsum / n);
    }
    if (supported(INFO_OUTVOLT)) {
	fsum = 0.0; n = 0;
	v = getdata(INFO_OUT_VOLT1); if (v != NULL) fsum += atof(v), n++;
	v = getdata(INFO_OUT_VOLT2); if (v != NULL) fsum += atof(v), n++;
	v = getdata(INFO_OUT_VOLT3); if (v != NULL) fsum += atof(v), n++;
	setinfo(INFO_OUTVOLT, "%.1f", fsum / n);
    }
}


/*-------------------------------------------------------------------------
 *
 * Here are the mandatory functions called from the shared main.c
 */

void upsdrv_initinfo(void)
{
    int msglen, e, v;
    char avail_list[300],*a,*p;

    /* find out which variables/commands this UPS supports */
    msglen = 0;
    sec_cmd(SEC_POLLCMD, SEC_AVAILP1, avail_list, &msglen);
    p = avail_list + msglen;
    if (p != avail_list) *p++ = ',';
    msglen = 0;
    sec_cmd(SEC_POLLCMD, SEC_AVAILP2, p, &msglen);
    *(p+msglen) = '\0';
    
    if (strlen(avail_list) == 0) fatalx("No available variables found!");
	
    upsdebugx(1, "List of available vars: %s",avail_list);

    /* scan list adding variables to info array */
    a = avail_list;
    e = 0;			/* index into enumdata array */
    while ((p = strtok(a, ",")) != NULL) {
	a = NULL;
	v = atoi(p);		/* variable number of supported variable */

	/* don't bother adding a write-only variable */
	if (sec_varlist[v].flags & FLAG_WONLY) continue;
	
	upsdebugx(1, "Adding variable %d (%s)",v,sec_varlist[v].name);
	addinfo(sec_varlist[v].infotag,
		"",
		sec_varlist[v].flags,
		0);

	if (sec_varlist[v].flags & FLAG_ENUM) {
	    /* find entries in enumdata for current variable */
	    while (sec_enumdata[e].type != sec_varlist[v].infotag) e++;
	    /* add entries for enumerated variable */
	    while (sec_enumdata[e].type == sec_varlist[v].infotag) {
		upsdebugx(1, " adding enumval \"%s\" (%d)",sec_enumdata[e].value,e);
		addenum(sec_enumdata[e].type, sec_enumdata[e].value);
		e++;
	    }
	}

	/* record this variable and its command in the query list */
	addquery(sec_varlist[v].cmd, sec_varlist[v].field, v);
    }

    /* add some composite non-SEC variables as well */
    addinfo(INFO_STATUS,  "", 0, 0);
    /* these are averages of the 3 different lines */
    if (supported(INFO_IN_ACFREQ1) ||
	supported(INFO_IN_ACFREQ2) ||
	supported(INFO_IN_ACFREQ3)) addinfo(INFO_ACFREQ,  "", 0, 0);
    if (supported(INFO_IN_VOLT1) ||
	supported(INFO_IN_VOLT2) ||
	supported(INFO_IN_VOLT3)) addinfo(INFO_UTILITY, "", 0, 0);
    if (supported(INFO_OUT_CURRENT1) ||
	supported(INFO_OUT_CURRENT2) ||
	supported(INFO_OUT_CURRENT3)) addinfo(INFO_CURRENT, "", 0, 0);
    if (supported(INFO_OUT_LOADPCT1) ||
	supported(INFO_OUT_LOADPCT2) ||
	supported(INFO_OUT_LOADPCT3)) addinfo(INFO_LOADPCT, "", 0, 0);
    if (supported(INFO_OUT_LOADPWR1) ||
	supported(INFO_OUT_LOADPWR2) ||
	supported(INFO_OUT_LOADPWR3)) addinfo(INFO_LOADPWR, "", 0, 0);
    if (supported(INFO_OUT_VOLT1) ||
	supported(INFO_OUT_VOLT2) ||
	supported(INFO_OUT_VOLT3)) addinfo(INFO_OUTVOLT, "", 0, 0);
}


/*
 * upsdrv_updateinfo()
 *
 * For a SEC protocol UPS, we only need to query those variables that are 
 * supported.
 */

void upsdrv_updateinfo(void)
{
    char retbuf[128],*r,*n;
    int ret, q, retlen, f;

    upsdebugx(1, "--------Updating--------------------------");

    /* loop over each query (a single SEC poll command)*/
    for (q=0; q<SEC_QUERYLIST_LEN; q++) {
	if (sec_querylist[q].command == NULL) break;
	
	upsdebugx(1, "Polling %s...", sec_querylist[q].command);
	
	retlen = 0;
	ret = sec_cmd(SEC_POLLCMD, sec_querylist[q].command, retbuf, &retlen);
	if (ret <=0) {
	    upslog(LOG_WARNING, "Warning sending poll cmd \"%s\" to UPS (%d)",
		   sec_querylist[q].command, ret);
	    continue;
	}
	
	r = retbuf;
	*(r+retlen) = '\0';
	for (f=0; f<SEC_MAXFIELDS; f++) {
	    n = strchr(r, ',');
	    if (n != NULL) *n = '\0';
	    /* is field/variable supported? */
	    if (sqv(q,f) > 0) {
		/* only update the value if it's changed */
		if (strcmp(sec_varlist[sqv(q,f)].value, r) != 0) {

		    snprintf(sec_varlist[sqv(q,f)].value, 
			sizeof(sec_varlist[sqv(q,f)].value), "%s", r);

		    sec_setinfo(sqv(q,f), r);
		}
	    }
	    if (n == NULL) break;
	    r = n+1;
	}
    }

    /* update the non-sec variables */
    update_pseudovars();
    
}

void upsdrv_shutdown(void)
{
   int msg_len;
/*supposedly the serial port stuff is already set                         */
/*since a serial port was found a moment ago, I won't bother seeing if    */
/*the UPS is found here, except for final shutdown command                */

/* so cancel any running shutdowns, set auto restart when AC returns,     */
/* set to shutdown entire UPS and actually shutdown the UPS.              */
/* This UPS momentarily shuts off power to computer even when commercial  */
/* power is available.                                                    */

   msg_len = 2;
   sec_cmd (SEC_SETCMD,SEC_SHUTDOWN,"-1",&msg_len);
   msg_len = 1 ;
   sec_cmd (SEC_SETCMD,SEC_AUTORESTART,"1",&msg_len);    
   msg_len = 1;
   sec_cmd (SEC_SETCMD, SEC_SHUTTYPE,"2",&msg_len);
   msg_len = 1;
   sec_cmd (SEC_SETCMD,SEC_SHUTDOWN,"5",&msg_len);
}

#if 0
void instcmd (int auxcmd, int dlen, char *data)
{
	  /* TODO: reply to upsd? */

	  switch (auxcmd) {
		  case CMD_BTEST0:	/* stop battery test */
		  	upssend("???");
		  	break;
		  case CMD_BTEST1:	/* start battery test */
		  	upssend("???");
		 	break;
		  default:
			  upslogx(LOG_INFO, "instcmd: unknown type 0x%04x", auxcmd);
	  }
}
#endif

void upsdrv_help(void)
{
}

void upsdrv_makevartable(void)
{
}

void upsdrv_banner(void)
{
    printf("Network UPS Tools (%s) - SEC protocol UPS driver \n", UPS_VERSION);
    printf ("\tDriver version %s\n", SEC_DRIVER_VERSION);

}

void upsdrv_initups(void)
{
	/* this driver needs lots of variable conversion work */
	broken_driver = 1;
	return;

/*    upssend_delay = 100000; */

    setup_serial(device_path);
    /* probe ups type */

    /* to get variables and flags from the command line, use this:
     *
     *                   set flag foo : /bin/driver -x foo
     * set variable 'cable' to '1234' : /bin/driver -x cable=1234
     *
     * to test flag foo in your code:
     *
     * 	if (testvar("foo"))
     * 		do_something();
     *
     * to show the value of cable:
     *
     *	printf("cable is set to %s\n", getval("cable"));
     */

}

void upsdrv_cleanup(void)
{
}
