#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include <stdarg.h>

short card_number = 0;
short safety = 0;
float memclk = 0;
float nvclk = 0;

typedef struct
{
    int timeout_id;
    int time;
    GtkDialog *dialog;
    GtkWidget *label; /* The label containing the time */
} Timeout;

GtkObject *value_mem;
GtkObject *value_core;

/*
    This function should be in the "client"
    We receive an error_type code describing the seriousness of the problem:
    -Error: fatal error
    -INFO: unimplemented stuff (like nv30 overclocking..)
    -Warning: non-fatal error
*/
void error(error_type code, const char *format, ...)
{
    va_list arg;
    char* message;
    va_start(arg, format);

    message = g_strdup_vprintf(format, arg);
    
    GtkWidget *dialog;
    switch(code)
    {
	case ERROR:
	    dialog = gtk_message_dialog_new(NULL, 
					    GTK_DIALOG_MODAL, 
					    GTK_MESSAGE_ERROR,
					    GTK_BUTTONS_CLOSE,
					    "%s", message);

	    gtk_dialog_run(GTK_DIALOG(dialog));
	    g_signal_connect_swapped(GTK_OBJECT(dialog), "response",
		G_CALLBACK(gtk_widget_destroy), GTK_OBJECT(dialog));
	    exit(1);	    	    
	    break;
	case INFO:
	    dialog = gtk_message_dialog_new(NULL, 
					    GTK_DIALOG_MODAL, 
					    GTK_MESSAGE_INFO,
					    GTK_BUTTONS_CLOSE,
					    "%s", message);

	    gtk_dialog_run(GTK_DIALOG(dialog));
	    g_signal_connect_swapped(GTK_OBJECT(dialog), "response",
		G_CALLBACK(gtk_widget_destroy), GTK_OBJECT(dialog));
	    gtk_widget_destroy(dialog);
	    break;
	case WARNING:
	    dialog = gtk_message_dialog_new(NULL, 
					    GTK_DIALOG_MODAL, 
					    GTK_MESSAGE_WARNING,
					    GTK_BUTTONS_CLOSE,
					    "%s", message);

	    gtk_dialog_run(GTK_DIALOG(dialog));
	    g_signal_connect_swapped(GTK_OBJECT(dialog), "response",
		G_CALLBACK(gtk_widget_destroy), GTK_OBJECT(dialog));
	    gtk_widget_destroy(dialog);
	    break;
    }
    va_end(arg);
    g_free(message);
}

gboolean timeout_callback(gpointer data)
{
    Timeout *timeout = (Timeout*)data;
    GtkLabel *label = GTK_LABEL(timeout->label);

    timeout->time--;

    /* When we ran out of time simulate "No" */
    if(timeout->time == 0)
    {
	gtk_dialog_response(timeout->dialog, GTK_RESPONSE_REJECT);
	return FALSE;
    }

    /* Refresh the time */
    gtk_label_set_text(label, g_strdup_printf("Timeout in: %d second(s)", timeout->time));
	
    return TRUE;
}

void timer()
{
    GtkWidget *label;
    Timeout *timeout;
    timeout = g_malloc(sizeof(Timeout));

    timeout->dialog = GTK_DIALOG(gtk_dialog_new_with_buttons("Testing the new speeds", 
	GTK_WINDOW(window_nvclock), GTK_DIALOG_MODAL, GTK_STOCK_YES, GTK_RESPONSE_ACCEPT,
	GTK_STOCK_NO, GTK_RESPONSE_REJECT, NULL));

    label = gtk_label_new("Are the new speeds working correctly?");
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(timeout->dialog)->vbox), label);
    gtk_widget_show(label);

    timeout->time = 5;
    timeout->label = gtk_label_new("Timeout in: 5 second(s)");
    gtk_container_add(GTK_CONTAINER(timeout->dialog->vbox), timeout->label);
    gtk_widget_show(GTK_WIDGET(timeout->label));

    /* Create the real timeout */
    timeout->timeout_id = g_timeout_add(1000, timeout_callback, timeout);

    gint result = gtk_dialog_run(GTK_DIALOG(timeout->dialog));

    /* Stop the timer because we got an answer back */
    g_source_remove(timeout->timeout_id);

    /* We receive ACCEPT when the Yes button was pressed. In all other cases it is REJECT */
    switch(result)
    {
	case GTK_RESPONSE_DELETE_EVENT:
	case GTK_RESPONSE_REJECT:
	    /* Restore the default speeds */
	    default_speeds();
	    break;
    }
    gtk_widget_destroy(GTK_WIDGET(timeout->dialog));
    g_free(timeout);
}

void update_core()
{
    gchar *value;
    GtkWidget *entry_core = lookup_widget(window_nvclock, "entry_core");

    value = g_strdup_printf("%d",  (gint)GTK_ADJUSTMENT(value_core)->value);
    gtk_entry_set_text(GTK_ENTRY(entry_core), value);
    nvclk = GTK_ADJUSTMENT(value_core)->value;
    g_free(value);

}

void update_mem()
{
    gchar *value;
    GtkWidget *entry_mem = lookup_widget(window_nvclock, "entry_mem");
    value = g_strdup_printf("%d",  (gint)GTK_ADJUSTMENT(value_mem)->value);
    gtk_entry_set_text(GTK_ENTRY(entry_mem), value);
    memclk = GTK_ADJUSTMENT(value_mem)->value;
    g_free(value);
}

void fill_card_combo()
{
    int i;
    const struct card_info *nv_info;
    GList *items = NULL;
    GtkWidget *combo_card  = lookup_widget(window_nvclock, "combo_card");
    

    for(i = 0; i <= total_cards; i++)
    {
	gchar *temp = g_strdup_printf("%d: %s", i+1, card[i].card_name);
	items = g_list_append(items, temp);
    }

    gtk_combo_set_popdown_strings(GTK_COMBO(combo_card), items);
    
    g_list_free(items);
}

/* This function updates the min/max values of the sliders, the speeds and more on the gui. */
void update_info()
{
    gchar *text;

    GtkWidget *entry_core = lookup_widget(window_nvclock, "entry_core");
    GtkWidget *entry_mem = lookup_widget(window_nvclock, "entry_mem");

    memclk = nv_card.get_memory_speed();
    nvclk = nv_card.get_gpu_speed();

    text = g_strdup_printf("%0.3f", memclk);
    (gint)GTK_ADJUSTMENT(value_mem)->lower = nv_card.memclk_min;
    (gint)GTK_ADJUSTMENT(value_mem)->upper = nv_card.memclk_max;
    (gint)GTK_ADJUSTMENT(value_mem)->value = (memclk);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(value_mem), memclk);
    gtk_entry_set_text(GTK_ENTRY(entry_mem), text);
    g_free(text);
	
    text = g_strdup_printf("%0.3f", nvclk);
    (gint)GTK_ADJUSTMENT(value_core)->lower = nv_card.nvclk_min;
    (gint)GTK_ADJUSTMENT(value_core)->upper = nv_card.nvclk_max;
    (gint)GTK_ADJUSTMENT(value_core)->value = (nvclk);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(value_core), nvclk);
    gtk_entry_set_text(GTK_ENTRY(entry_core), text);


    g_free(text);
}

/* This function sets all card info on the gui. */
void update_card_info()
{
    gchar *memsize, *memtype, *fwstatus, *sbastatus, *agprate;
    int size;

    GtkWidget *label_gpu_text = lookup_widget(window_nvclock, "label_gpu_text");
    GtkWidget *label_bustype_text = lookup_widget(window_nvclock, "label_bustype_text");
    GtkWidget *label_memsize_text = lookup_widget(window_nvclock, "label_memsize_text");
    GtkWidget *label_memtype_text = lookup_widget(window_nvclock, "label_memtype_text");
    GtkWidget *label_agpstatus_text = lookup_widget(window_nvclock, "label_agpstatus_text");
    GtkWidget *label_fwstatus_text = lookup_widget(window_nvclock, "label_fwstatus_text");
    GtkWidget *label_sbastatus_text = lookup_widget(window_nvclock, "label_sbastatus_text");
    GtkWidget *label_agprate_text = lookup_widget(window_nvclock, "label_agprate_text");
    GtkWidget *label_agprates_text = lookup_widget(window_nvclock, "label_agprates_text");
    
    gtk_label_set_text(GTK_LABEL(label_gpu_text), nv_card.card_name);
    gtk_label_set_text(GTK_LABEL(label_bustype_text), nv_card.get_bus_type());

    memsize = g_strdup_printf("%d MB", nv_card.get_memory_size());
    gtk_label_set_text(GTK_LABEL(label_memsize_text), memsize);

    memtype = g_strdup_printf("%d bit %s", nv_card.get_memory_width(), nv_card.get_memory_type());
    gtk_label_set_text(GTK_LABEL(label_memtype_text), memtype);

    gtk_label_set_text(GTK_LABEL(label_agpstatus_text), nv_card.get_agp_status());

//    fwstatus = g_strdup_printf("%s, %s", nv_card.get_fw_support(), nv_card.get_fw_status());
    gtk_label_set_text(GTK_LABEL(label_fwstatus_text), nv_card.get_fw_status());

//    sbastatus = g_strdup_printf("%s, %s", nv_card.get_sba_support(), nv_card.get_sba_status());
    gtk_label_set_text(GTK_LABEL(label_sbastatus_text), nv_card.get_sba_status());
    
    /* Check if the agp_rate isn't 0, if it is 0 the card is a pci card. */
    if(nv_card.get_agp_rate() == 0)
    {
	gtk_label_set_text(GTK_LABEL(label_agprate_text), "-");
    }

    else
    {
	agprate = g_strdup_printf("%dX", nv_card.get_agp_rate());
	gtk_label_set_text(GTK_LABEL(label_agprate_text), agprate);
	g_free(agprate);
    }
    
    gtk_label_set_text(GTK_LABEL(label_agprates_text), nv_card.get_supported_agp_rates());


    g_free(memsize);
    g_free(memtype);
}

/* This function updates the speeds on the gui. */
void update(gfloat memclk, gfloat nvclk)
{
    gchar *text;
    GtkWidget *entry_core = lookup_widget(window_nvclock, "entry_core");
    GtkWidget *entry_mem = lookup_widget(window_nvclock, "entry_mem");
	
    text = g_strdup_printf("%0.3f", memclk);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(value_mem), memclk);
    gtk_entry_set_text(GTK_ENTRY(entry_mem), text);
    g_free(text);
	
    text = g_strdup_printf("%0.3f", nvclk);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(value_core), nvclk);
    gtk_entry_set_text(GTK_ENTRY(entry_core), text);
    g_free(text);
}

/* This function overclocks the card and updates the speeds on the gui. */
void change_speeds(GtkButton *button, gpointer data)
{
    nv_card.set_gpu_speed(nvclk);
    nv_card.set_memory_speed(memclk);

    memclk = nv_card.get_memory_speed();
    nvclk = nv_card.get_gpu_speed();

    if(safety)
	timer();

    update(memclk, nvclk);
}

/* This function resets the card's clocks and updates the speeds on the gui. */
void default_speeds(GtkButton *button, gpointer data)
{
    nv_card.reset_speeds();

    memclk = nv_card.get_memory_speed();
    nvclk = nv_card.get_gpu_speed();

    update(memclk, nvclk);
}

/* This function sets the debug flag. */
void set_debug_info(GtkToggleButton *button, gpointer user_data)
{
    nv_card.debug = gtk_toggle_button_get_active(button);
}

/* This function sets the safety flag. */
void set_safety_check(GtkToggleButton *button, gpointer user_data)
{
    safety = gtk_toggle_button_get_active(button);
}

gboolean nvclock_exit(GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
    gtk_main_quit();
}

/* This function is used when another card is selected using the combo box. */
void change_cards(GtkEditable *editable, gpointer user_data)
{
    char *text;
    GtkWidget *combo = lookup_widget(window_nvclock, "combo_card");
    text = (char*)gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(combo)->entry));

    /*  Use the following line as a workaround. This function is called when the value of the combo_entry changes, but
	but that has one little problem. When selecting something else in the combo box the signal is sent out twice.
	One time to empty the entry and another time to fill it with the new stuff. */

    if(strlen(text) == 0)
	return; /* Do nothing. */

    /* The first char of text is the the card number. Where number is between 1 and 4. This way might be a bit hacky*/
    card_number = atoi(text) - 1;

    /* We don't want to switch when the cards are the same */
    if(strcmp(card[card_number].bus_id, nv_card.bus_id) == 0) 
	return;
	
    /* Set the card object to the new card */
    set_card(card_number);

    /* Update the min/max values of the sliders, the speeds on the gui ... */
    update_info();
    
    /* Update all info about the card: AGP, chipname, bustype ... */
    update_card_info();
}

void slider_core_realize()
{
    GtkWidget *slider_core = lookup_widget(window_nvclock, "slider_core");
    
    value_core = gtk_adjustment_new(0, 1, 1000, 1, 1, 1);
    gtk_signal_connect(GTK_OBJECT(value_core), "value_changed", GTK_SIGNAL_FUNC(update_core), NULL);
    gtk_range_set_adjustment(GTK_RANGE(slider_core), GTK_ADJUSTMENT(value_core));
}

void slider_mem_realize()
{
    GtkWidget *slider_mem = lookup_widget(window_nvclock, "slider_mem");
    
    value_mem = gtk_adjustment_new(0, 1, 1000, 1, 1, 1);
    gtk_signal_connect(GTK_OBJECT(value_mem), "value_changed", GTK_SIGNAL_FUNC(update_mem), NULL);
    gtk_range_set_adjustment(GTK_RANGE(slider_mem), GTK_ADJUSTMENT(value_mem));
}
