/* $Id: messages.h,v 1.19 2004/11/15 20:59:45 graziano Exp $ */

#ifndef MESSAGES_H
#define MESSAGES_H

/*
 * This package defines NWS functions for sending messages between hosts.
 */

#include <sys/types.h>  /* size_t */
#include "formatutil.h" /* DataDescriptor */
#include "protocol.h"   /* Socket */

#ifdef __cplusplus
extern "C" {
#endif

/**
 * This is the four-byte NWS version number.  The first two bytes are
 * major and minor release numbers; the meaning of the other two bytes is
 * undefined.
 */
#define NWS_VERSION 0x02110000


typedef unsigned int MessageType;

/** 
 * this is the message header: it contains the NWS version (if I don't
 * forget to update) so that we can try to be backward compatible, the
 * message type and the data size. Upwn recepit of a message this is
 * what is passed to the listener along with the open socket.
 */
typedef struct {
	unsigned int version;
	MessageType message;
	unsigned int dataSize;
} MessageHeader;

/*
 * A header sent with messages.  #version# is the NWS version.
 * #message# is the actual message.  #dataSize# is the number of bytes
 * that accompany the message.
 */
static const DataDescriptor headerDescriptor[] =
	{SIMPLE_MEMBER(UNSIGNED_INT_TYPE, 1, offsetof(MessageHeader, version)),
	SIMPLE_MEMBER(UNSIGNED_INT_TYPE, 1, offsetof(MessageHeader, message)),
	SIMPLE_MEMBER(UNSIGNED_INT_TYPE, 1, offsetof(MessageHeader, dataSize))};
#define headerDescriptorLength 3

typedef void (*ListenFunction)(Socket *, MessageHeader);


/**
 * Waits up to #timeOut# seconds to see if a message comes in; if so,
 * calls the registered listener for that message (see
 * RegisterListener()). Returns 1 if a message have been served, 0
 * otherwise.
 */
int
ListenForMessages(double timeOut);


/**
 * Indicates a desire that the function #listener# be called whenever a
 * #message# message comes in.  #image# is a printable message name; it's
 * used to log the arrival of the message. If there is already a
 * function registered for this messageType, it would be unregistered
 * before installing the new one.
 */
void
RegisterListener(MessageType message,
                 const char *image,
                 ListenFunction listener);

/**
 * Remove the registered listenere for mesageType #type#. 
 */
void
UnregisterListener(MessageType type);


/**
 * Receives on #sd#, into #data#, the data described by the #howMany#-long
 * array of descriptors #description#.  Returns 1 if successful within
 * #timeOut# seconds, else 0.
 * If #timeOut# is negative adaptive timeouts will be used, if it's 0 no
 * timeouts will be used.
 */
int
RecvData(Socket sd,
         void *data,
         const DataDescriptor *description,
         size_t howMany,
         double timeOut);


/**
 * Waits for a #message# message to come in over #sd#.  If successful within
 * #timeOut# seconds, returns 1 and copies the size of the accompanying data
 * into #dataSize#; otherwise, returns 0.
 * If #timeOut# is negative adaptive timeouts will be used, if it's 0 no
 * timeouts will be used.
 */
int
RecvMessage(Socket sd,
            MessageType message,
            size_t *dataSize,
            double timeOut);


/**
 * Waits for a #message# message to come in over #sd#.  If successful within
 * #timeOut# seconds, returns 1 and copies the accompanying data into #data1#
 * and #data2#, which are described by the #howMany1# and #howMany3#-long
 * arrays of descriptors #description1# and #description2#, respectively;
 * otherwise, returns 0.
 * If #timeOut# is negative adaptive timeouts will be used, if it's 0 no
 * timeouts will be used.
 */
int
RecvMessageAndDatas(Socket sd,
                    MessageType message,
                    void *data1,
                    const DataDescriptor *description1,
                    size_t howMany1,
                    void *data2,
                    const DataDescriptor *description2,
                    size_t howMany2,
                    double timeOut);
#define RecvMessageAndData(sd,message,data,description,howMany,timeOut) \
  RecvMessageAndDatas(sd,message,data,description,howMany,NULL,NULL,0,timeOut)
 

/**
 * Sends #data#, described by the #howMany#-long array of descriptors
 * #description#.  Returns 1 if successful in #timeOut# seconds, else 0.  This
 * fuction allows callers to extend the data that accompanies messages beyond
 * the two items provided for by SendMessageAndDatas().  Note, however, that
 * since the data sent via a call to this function is not packaged with a
 * message, the receiver will not be able to determine the data size directly.
 * The caller must assure that the data size is known to the recipient, either
 * because its length is predetermined, or because its length was enclosed in a
 * previously-transmitted message.
 * If #timeOut# is negative adaptive timeouts will be used, if it's 0 no
 * timeouts will be used.
 */
int
SendData(Socket sd,
         const void *data,
         const DataDescriptor *description,
         size_t howMany,
         double timeOut);


/**
 * Sends a message of type #message# on #sd# followed by #data1# and #data2#,
 * which are described, respectively, by the #howMany1#-long and the
 * #howMany2#-long arrays of descriptors #description1# and #description2#.
 * Each data parameter may be NULL, in which case its description is ignored.
 * Returns 1 if successful within #timeOut# seconds, else 0.
 * If #timeOut# is negative adaptive timeouts will be used, if it's 0 no
 * timeouts will be used.
 */
int
SendMessageAndDatas(Socket sd,
                    MessageType message,
                    const void *data1,
                    const DataDescriptor *description1,
                    size_t howMany1,
                    const void *data2,
                    const DataDescriptor *description2,
                    size_t howMany2,
                    double timeOut);
#define SendMessage(sd,message,timeOut) \
   SendMessageAndDatas(sd,message,NULL,NULL,0,NULL,NULL,0,timeOut)
#define SendMessageAndData(sd,message,data,descriptor,howMany,timeOut) \
   SendMessageAndDatas(sd,message,data,descriptor,howMany,NULL,NULL,0,timeOut)

/**
 * reads the NWS header associated with in incoming message and returns
 * it in #header#.  Returns 0 if the read fails, 1 otherwise. If
 * #timeOut# is negative adaptive timeouts will be used, if it's 0 no
 * timeouts will be used.
 */
int RecvHeader(	Socket sd, 
		MessageHeader *header,
		double timeOut);

#ifdef __cplusplus
}
#endif

#endif /* MESSAGES_H */
