/* $Id: */

#include "config_nws.h"

#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_LIMITS_H
#	include <limits.h>
#endif

#include "diagnostic.h"
#include "osutil.h"
#include "skills.h"
#include "strutil.h"

/* let's be sure we have some resonable definition */
#ifndef INT_MAX
#	define INT_MAX	(2147483647)
#endif
#ifndef SHRT_MAX
#	define SHRT_MAX	(32767)
#endif
#ifndef LONG_MAX
#	define LONG_MAX	(2147483647L)
#endif
#define KB 1024

/*
 * returns the best size to use during memory access (in bytes)
 */
static int
howManyChar() {
	register int x, i, till;
	short best = 4;
	double before, after;
	void *mem;
	long bestTime = INT_MAX;
#define MY_LOOP_SIZE 512

	/* allocate MY_LOOP_SIZE KB */
	mem = MALLOC(KB*MY_LOOP_SIZE, 1);
	memset(mem, '0', KB*MY_LOOP_SIZE);

	till = (MY_LOOP_SIZE*KB)/sizeof(short);
	before = MicroTime();
	for (x=0; x < 50; x++) {
		for (i=0; i < till; i++) {
			((short *)(mem))[i] = SHRT_MAX;
		}
	}
	after = MicroTime();
	if (bestTime > (after - before)) {
		bestTime = after - before;
		best = sizeof(short);
	}

	till = (MY_LOOP_SIZE*KB)/sizeof(int);
	before = MicroTime();
	for (x=0; x < 50; x++) {
		for (i=0; i < till; i++) {
			((int *)(mem))[i] = INT_MAX;
		}
	}
	after = MicroTime();
	if (bestTime > (after - before)) {
		bestTime = after - before;
		best = sizeof(int);
	}
	till = (MY_LOOP_SIZE*KB)/sizeof(long);
	before = MicroTime();
	for (x=0; x < 50; x++) {
		for (i=0; i < till; i++) {
			((long *)(mem))[i] = INT_MAX;
		}
	}
	after = MicroTime();
	if (bestTime > (after - before)) {
		bestTime = after - before;
		best = sizeof(long);
	}
	free(mem);
	DDEBUG1("howManyChar: best word size is %d\n", best);

	return best;
}

/*
 * time memory
 */
static double
timeMemory(	unsigned int size,
		unsigned int loops,
		unsigned int wordSize,
		unsigned int sequential) {
	register unsigned int i, till;	
	unsigned int z, *r, j;
	char *mem;			/* memory we are using */
	double before, after, avg;
	unsigned long randSeed;

	/* let's allocate the buffer */
	mem = MALLOC(size * KB, 0);
	if (mem == NULL) {
		ERROR("timeMemory: out of memory\n");
		return -1;
	}

	/* checking the wordSize */
	if (wordSize != sizeof(short) && wordSize != sizeof(char) 
			&& wordSize != sizeof(int) && wordSize != sizeof(long)
	   ) {
		ERROR1("timeMemory: bogus word size (%d): using int\n", wordSize);
		wordSize = sizeof(int);
	}


	/* this is how many try we have to loop through the memory */
	till = (size * KB)/wordSize;
	r = MALLOC(sizeof(int) * till, 0);
	if (r == NULL) {
		FREE(mem);
		ERROR("timeMemory: out of memory\n");
		return -1;
	}
	randSeed = CurrentTime();

	for (avg=0, z=0; z < loops; z++) { 
		/* reset the indices */
		for (i=0; i < till; i++) {
			r[i] = -1;
		}

		/* and refill them now in the right order */
		for (i=0; i < till; i++) {
			/* this is the sequential part */
			if (sequential) {
				r[i] = i;
			} else { 
				/* this is the random part */
				randSeed = randSeed * 1103515245 + 12345;
				j = (unsigned int)(randSeed) % till;
				while (r[j] != -1) {
					j = (j + 1) % till;
				}
				r[j] = i;
			}
		}


		/* let's try to clean the cache */
		before = MicroTime();
		memset(mem, 0, size * KB);
		after = MicroTime() - before;

		after = ((double)size*1000000)/(after*KB);
		DDEBUG1("timeMemory: memset got %.2f MB/s\n", after);

		/* read (r[i]) and write tests */
		if (wordSize == sizeof(short)) {
			before = MicroTime();
			for (i=0; i < till; i++) {
				((short *)(mem))[r[i]] = (short)i;
			}
			after = MicroTime();
		} else if (wordSize == sizeof(char)) {
			before = MicroTime();
			for (i=0; i < till; i++) {
				((char *)(mem))[r[i]] = (char)i;
			}
			after = MicroTime();
		} else if (wordSize == sizeof(int)) {
			before = MicroTime();
			for (i=0; i < till; i++) {
				((int *)(mem))[r[i]] = (int)i;
			}
			after = MicroTime();
		} else if (wordSize == sizeof(long)) {
			before = MicroTime();
			for (i=0; i < till; i++) {
				((long *)(mem))[r[i]] = (long)i;
			}
			after = MicroTime();
		}

		avg = avg + (after - before)/1000000;
	}
	free(mem);
	free(r);

	return ((2*loops*size)/(avg*KB));
}

int
MemorySpeedAvailable(	const char *opts) {
	/* always available */
	return 1;
}


void
MemorySpeedUseSkill(	const char *opts,
			int *length,
			SkillResult **results) {
	unsigned int size, loops, sequential;
	char name[64+1];
	double res; 
	const char *c;
	char *tmp, o[255 + 1];

	/* let's parse the options*/
	sequential = 0;
	tmp = GetOptionValue(opts, "mode", "random");
	if (strcmp(tmp, "sequential") == 0) {
		sequential = 1;
	}
	FREE(tmp);
	loops = 1;
	tmp = GetOptionValue(opts, "loops", "1");
	loops = strtol(tmp, NULL, 10);
	FREE(tmp);
	if (loops <= 0) {
		loops = 1;
	}

	tmp = GetOptionValue(opts, "size", "8192");
	for (c = tmp; GETTOK(name, c, ",", &c);) {
		size = strtol(name, NULL, 10);
		DDEBUG1("MemorySpeedUseSkill: using %d KB\n", size);

		res = timeMemory(size, loops, howManyChar(), sequential);

		/* let's get the right options */
		snprintf(o, sizeof(o), "mode:%s\tloops:%d\tsize:%d", (sequential ? "sequential" : "random"), loops, size);

		DDEBUG1("MemorySpeedUseSkill: timed memory at %.2f MB/s\n", res);

		if (res < 0) {
			AppendResult(memorySpeed, o, 0, 0, 0, length, results);
		} else {
			AppendResult(memorySpeed, o, 1, res, 0, length, results);
		}
	 }
	FREE(tmp);
}

