#include <getopt.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "flashlogo.h"

/* #define DEBUG */

int main(int argc, char **argv)
{
	FILE *inputfile;
	int fsize = flashsize();
	struct logoblock2 logo2;
	struct logoblock logo1;
	long size;
	int result;
	int c;
	char *buffer;
	int bigimage = 0;
	int logoflags, logox, logoy;

	logoflags = LB_SHOWGIF | LB_SHOWNW;
	logox = logoy = 0;

#ifdef DEBUG
	if (sizeof(struct logoblock) != 65536) {
		fprintf(stderr, "%s: bad logo structure size... recompile? (%i)\n", argv[0], sizeof(struct logoblock)
			);
		exit(1);
	}
#endif

	while (1) {
		static struct option long_options[] = {
			{"no-netwinder", 0, 0, 'w'},
			{"no-usergif", 0, 0, 'u'},
			{"no-sound", 0, 0, 's'},
			{"help", 0, 0, 'h'},
			{"version", 0, 0, 'v'},
			{"full64", 0, 0, 'f'},
			{0, 0, 0, 0}
		};

		c = getopt_long(argc, argv, "x:y:", long_options, NULL);

		if (c == -1)
			break;

		switch (c) {
		case 'x':
			logox = atoi(optarg);
			break;
		case 'y':
			logoy = atoi(optarg);
			break;
		case 'u':
			logoflags &= ~LB_SHOWGIF;
			break;
		case 'w':
			logoflags &= ~LB_SHOWNW;
			break;
		case 's':
			logoflags |= LB_NOSOUND;
			break;
		case 'f':
			bigimage = 1;
			break;
		case 'h':
			printf("Usage: %s [OPTIONS]... [FILE]\n"
				"Write the specified GIF file to flash memory and configure startup graphic\n"
				"behaviour.\n"
				"\n"
				"      --no-netwinder          do not show NetWinder logo on startup.\n"
				"      --no-usergif            do not show user GIF file on startup. If this\n"
				"                               option is given, FILE is ignored.\n"
				"      --no-sound              do not play startup tune.\n"
				"  -x [X-COORD]                set x coordinate for top-left corner of user\n"
				"                               GIF file instead of 0.\n"
				"  -y [Y-COORD]                set y coordinate for top-left corner of user\n"
				"                               GIF file instead of 0.\n"
				"      --full64                use a separate 64K flash segment for the logo.\n"
				"      --help                  display this help and exit.\n"
				"      --version               output version information and exit.\n" "\n", argv[0]);
			exit(1);
		case 'v':
			printf("logowrite 2.00\n");
			exit(1);
		case '?':
			fprintf(stderr, "Try `%s --help' for more information.\n", argv[0]);
			exit(1);
		}
	}

	if (bigimage) {
		logo1.magic = LB_MAGIC;
		logo1.flags = logoflags;
		logo1.x = logox;
		logo1.y = logoy;
	} else {
		logo2.magic = LB_MAGIC;
		logo2.flags = logoflags;
		logo2.x = logox;
		logo2.y = logoy;
	}

	if (logoflags & LB_SHOWGIF) {
		if (optind >= argc) {
			fprintf(stderr, "%s: missing file argument\nTry `%s --help' for more information.\n", argv[0], argv[0]);
			exit(1);
		}

		inputfile = fopen(argv[optind], "rb");

		if (!inputfile) {
			fprintf(stderr, "%s: %s: Error opening input file.\n", argv[0], argv[optind]);
			exit(1);
		}

		printf("Checking image size... ");

		fseek(inputfile, 0, SEEK_END);
		size = ftell(inputfile);
		fseek(inputfile, 0, SEEK_SET);

		printf("%li bytes.\n", size);

		if (!bigimage) {
			if (size > sizeof(logo2.data2)) {
				printf("\n");
				fprintf(stderr, "%s: image is too large; maximum size is %u.\n", argv[0], sizeof(logo2.data2));
				exit(1);
			}
		} else {
			if (size > sizeof(logo1.data)) {
				printf("\n");
				fprintf(stderr, "%s: image is too large; maximum size is %u.\n", argv[0], sizeof(logo1.data));
				exit(1);
			}
		}

		printf("Loading image into buffer... ");
		fflush(stdout);

		if (!bigimage) {
			logo2.size = size;
			fread(&(logo2.data2), logo2.size, 1, inputfile);
		} else {
			logo1.size = size;
			fread(&(logo1.data), logo1.size, 1, inputfile);
		}

		printf("Done.\n");

		fclose(inputfile);
	}

	if (fsize < 0) {
		fprintf(stderr, "%s: unable to access flash device.\n", argv[0]);
		exit(1);
	}

	printf("Flash size: %i bytes.\n", fsize);

	if (bigimage)
		printf("Writing into flash at offset 0x%X... ", fsize - 128 * 1024);
	else
		printf("Writing into flash at offset 0x%X... ", fsize - 62 * 1024);

	fflush(stdout);

	if (bigimage) {
		result = flashwrite((char *) &logo1, fsize - 128 * 1024, 65536);
	} else {
		buffer = (char *) malloc(64 * 1024);
		if (buffer) {
			/* if logo2 - read in the 2K flash prefs strings */
			result = flashread(buffer, fsize - 64 * 1024, 2048);
			memcpy(buffer + 2048, (char *) &logo2, 62 * 1024);
			result = flashwrite(buffer, fsize - 64 * 1024, 65536);
			free(buffer);
		} else
			result = -1;
	}

	if (result < 0)
		printf("failed. (code %i)\n", result);
	else
		printf("%u bytes written.\n", result);

	printf("Done.\n");

	return 0;
}
