/*
 * nzb
 *
 * Copyright (C) 2004-2005 Mattias Nordstrom <matta at ftlight net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *
 * Authors:
 *   Mattias Nordstrom <matta at ftlight net>
 *
 * $Id: downloader.cpp,v 1.4 2005/09/27 17:48:40 mnordstr Exp $
 *   This file provides the NNTP downloader.
 */


#include "downloader.h"
#include "mainwindow.h"

Downloader::Downloader(NzbList *nzblist, int thread_id, QMutex *file_lock, QObject *parent)
{
	this->nzblist = nzblist;
	this->parent = parent;
	this->thread_id = thread_id;
	this->file_lock = file_lock;
	connect(this, SIGNAL(downloadEvent(QString, int, int)), parent, SLOT(downloadEvent(QString, int, int)));
}

void Downloader::run()
{
	qDebug("Downloader running.");
	
	QTimer::singleShot(0, this, SLOT(init()));

	this->exec();
	
	qDebug("Downloader done.");
}

void Downloader::init()
{
	file = 0, seg = 0;
	downloading = false;
	connect(this, SIGNAL(processNext()), this, SLOT(processFiles()));
	
	emit downloadEvent("Connecting", thread_id);
	usenet = new QTcpSocket();
	connect(usenet, SIGNAL(readyRead()), this, SLOT(gotData()));
	initialize(((MainWindow*)parent)->getConfig()->value("server/host").toString(), ((MainWindow*)parent)->getConfig()->value("server/port").toInt());

	last_time = QTime::currentTime();
	last_bytes = 0;
	total_bytes = 0;
	
	QTimer *speedTimer = new QTimer(this);
	connect(speedTimer, SIGNAL(timeout()), this, SLOT(speedMonitor()));
	speedTimer->start(5000);
}

void Downloader::processFiles()
{
	if (downloading) return;
	
	if (getNext()) {
			downloading = true;
			emit downloadEvent("Downloading ["+QString::number(file)+": "+QString::number(seg+1)+"/"+QString::number(nzblist->getFile(file)->getSegments()->size())+"] "+nzblist->getFile(file)->getSegment(seg)->getMsgid(), thread_id);
			getArticle(nzblist->getFile(file)->getSegment(seg)->getMsgid());
	} else {
		emit downloadEvent("Disconnecting", thread_id);
		terminate();
		emit downloadEvent("Disconnected", thread_id);
	}
}

bool Downloader::getNext()
{
	file_lock->lock();
	
	for (;;) {
		if (nzblist->getFile(file)->getSegment(seg)->getStatus() == NZB_NONE) {
			nzblist->getFile(file)->getSegment(seg)->setStatus(NZB_DOWNLOADING);
			file_lock->unlock();
			return true;
		}
		
		if (nzblist->getFile(file)->getSegments()->size() == seg+1) {
			if (nzblist->getList()->size() == file+1) {
				file_lock->unlock();
				return false;
			}
			
			file = file + 1;
			seg = 0;
		} else {
			seg = seg + 1;
		}
	}
	
	return false;
}

void Downloader::stop()
{
	emit downloadEvent("Disconnecting", thread_id);
	terminate();
	emit downloadEvent("Disconnected", thread_id);

	dldata.clear();
	delete usenet;
	this->exit();
}

void Downloader::initialize(QString host, int port)
{
	usenet->connectToHost(host, port);
}

void Downloader::terminate()
{
	usenet->write(((QString)("QUIT\r\n")).toAscii());
	usenet->close();
}

void Downloader::getArticle(QString msgid)
{
	usenet->write(("ARTICLE "+msgid+"\r\n").toAscii());
	usenet->flush();
	dldata.clear();
}

void Downloader::gotData()
{
	if (!usenet->bytesAvailable()) return;

	total_bytes += usenet->bytesAvailable();
	
	QByteArray hdata;
	int pos;
	
	if (dldata.isEmpty()) {
		hdata = usenet->readLine();
		if (hdata.left(3) == "200") {
			if (((MainWindow*)parent)->getConfig()->value("server/auth").toBool()) {
				usenet->write(("AUTHINFO USER "+((MainWindow*)parent)->getConfig()->value("server/username").toString()+"\r\n").toAscii());
				usenet->flush();
				return;
			} else {
				emit downloadEvent("Connected", thread_id, 3);
				emit processNext();
				return;
			}
		} else if (hdata.left(3) == "381") {
			usenet->write(("AUTHINFO PASS "+((MainWindow*)parent)->getConfig()->value("server/password").toString()+"\r\n").toAscii());
			usenet->flush();
			return;
		} else if (hdata.left(3) == "281") {
			qDebug("Connected");
			emit downloadEvent("Connected", thread_id, 3);
			emit processNext();
			return;
		} else if (hdata.left(1) == "4" || hdata.left(1) == "5") {
			emit downloadEvent("NNTP Error: "+hdata, thread_id);
			qDebug() << "Error: " << hdata;
			return;
		} else if (hdata.left(2) == "22") {
			dldata = hdata;
			dldata += usenet->readAll();
		} else {
			/*emit downloadEvent("Connect failure", thread_id);
			qDebug("Connection failed.");
			nzblist->getFile(file)->getSegment(seg)->setStatus(NZB_DOWNLOADED);
			usenet->close();
			qDebug("NNTP Protocol handshake error.");
			qDebug() << "Data: " << hdata;
			return;*/
			dldata = hdata+usenet->readAll();
		}
	} else {
		dldata += usenet->readAll();
	}
	
	if (dldata.right(5) == "\r\n.\r\n") {
		downloading = false;
		dldata = dldata.left(dldata.length() - 3);

		pos = dldata.indexOf("\r\n\r\n", 0);
		dldata = dldata.mid(pos + 4, dldata.length() - (pos + 4));

		if (dldata.left(2) == "..") {
			dldata.remove(0, 1);
		}
		
		pos = 0;
		while ((pos = dldata.indexOf("\r\n..", pos+3)) != -1) {
			dldata.remove(pos + 2, 1);
		}
		
		nzblist->getFile(file)->getSegment(seg)->setData(dldata);
		nzblist->getFile(file)->getSegment(seg)->setStatus(NZB_DOWNLOADED);
		//dldata.clear();
			
		emit downloadEvent("", file, 2);
		emit processNext();
	}
}

void Downloader::speedMonitor()
{
	double speed = ((total_bytes - last_bytes) / last_time.secsTo(QTime::currentTime())) / 1000;
	last_time = QTime::currentTime();
	last_bytes = total_bytes;
	
	emit downloadEvent(QString::number(speed), thread_id, 4);
}
