(* SQLite database interface for OCamlDBI.
 * Copyright (C) 2004 Evan Martin <martine@danga.com>
 * Based on PostgreSQL database interface:
 * Copyright (C) 2003-2004 Merjis Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: dbi_sqlite.ml,v 1.1 2004/04/10 09:16:01 rwmj Exp $
 *)

open Dbi_utils_
open Printf

(* [encode_sql_t v] returns a string suitable for substitution of "?"
   in a SQL query. *)
let encode_sql_t = function
  | `Null -> "NULL"
  | `Int i -> string_of_int i
  | `Float f -> string_of_float f
  | `String s -> Dbi.string_escaped s
  | `Bool b -> if b then "T" else "F" (* XXX *)
      (*  | `Bigint i -> string_of_big_int i *)
  | `Decimal d -> Dbi.Decimal.to_string d
  | `Date d -> sprintf "'%04i-%02i-%02i'" d.Dbi.year d.Dbi.month d.Dbi.day
  | `Time t -> sprintf "'%02i:%02i:%02i'" t.Dbi.hour t.Dbi.min t.Dbi.sec
  | `Timestamp t -> invalid_arg "Dbi_sqlite: timestamp not handled"
  | `Interval i -> invalid_arg "Dbi_sqlite: timestamp not handled"
  | `Blob s -> Dbi.string_escaped s (* XXX *)
  | `Binary s -> invalid_arg "Dbi_sqlite: binary not handled"
  | `Unknown s -> Dbi.string_escaped s

class statement dbh db original_query =
  let query = split_query original_query in
object (self)
  inherit Dbi.statement dbh

  val mutable cur_vm = None
  val mutable next_row = None
  val mutable executed = false
  
  method execute args =
    if dbh#debug then eprintf "Dbi_sqlite: execute: %s\n%!" original_query;

    if dbh#closed then
      failwith "Dbi_sqlite: execute called on a closed database handle.";
    let query =
      make_query "Dbi_sqlite: execute called with wrong number of args."
        encode_sql_t query args in
    if dbh#debug then eprintf "Dbi_sqlite: query: %s\n%!" query;
    (* Finish previous statement, if any. *)
    self#finish ();
    let vm = Sqlite.compile_simple db query in
    cur_vm <- Some vm;
    executed <- true;
    (* we must step immediately, because that's the only way to execute
       immediately. *)
    try next_row <- Some (Sqlite.step_opt vm)
    with Sqlite.Sqlite_done -> cur_vm <- None; next_row <- None

  method fetch1 () =
    if dbh#debug then eprintf "Dbi_sqlite: fetch1\n%!";

    if not executed then failwith "Dbi_sqlite: fetch called without execute.";

    let row = match next_row with 
      None -> self#finish (); raise Not_found
    | Some r ->
        (match cur_vm with
          Some vm ->
            (try next_row <- Some (Sqlite.step_opt vm)
            with Sqlite.Sqlite_done -> cur_vm <- None; next_row <- None);
        | None -> next_row <- None);
        r
    in
    let rec loop acc i =
      if i < 0 then acc else
      let value = match row.(i) with
        None -> `Null
      | Some x -> `String x in
      loop (value :: acc) (i - 1)
    in
    loop [] (Array.length row - 1)
    
  method fetch1int () =
    match self#fetch1 () with
    | [`String i] -> (try int_of_string i
                      with Failure _ -> invalid_arg "fetch1int")
    | _ -> invalid_arg "fetch1int"

  method names = []
  method serial seq = raise Not_found
  method finish () =
    (match cur_vm with
      Some vm -> 
        if dbh#debug then
          eprintf "Dbi_sqlite: dbh %d: finish %s\n%!" dbh#id original_query;
        Sqlite.finalize vm;
        cur_vm <- None
    | None -> ());
    next_row <- None;
    executed <- false
end

and connection ?host ?port ?user ?password database =
  let db = Sqlite.db_open database in
object (self)
  inherit Dbi.connection ?host ?port ?user ?password database as super
  method host = None
  method port = None
  method user = None
  method password = None
  method database = database

  method database_type = "sqlite"

  method prepare query =
    if self#closed then
      failwith "Dbi_sqlite: prepare called on closed database handle.";
    new statement
      (self : #Dbi.connection :> Dbi.connection) db query

  method commit () = ()
  method rollback () = ()
  method close () =
    Sqlite.db_close db;
    super#close ()
end

let () = Dbi.Factory.register "sqlite" (new connection)

(* vim: set ts=2 sw=2 et : *)
