(*
 * Graph: generic graph library
 * Copyright (C) 2004
 * Sylvain Conchon, Jean-Christophe Filliatre and Julien Signoles
 * 
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License version 2, as published by the Free Software Foundation.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 * See the GNU Library General Public License version 2 for more details
 * (enclosed in the file LGPL).
 *)

(* $Id: rand.ml,v 1.4 2004/02/18 09:42:25 filliatr Exp $ *)

module type S = sig
  type graph 
  type vertex
  type edge_label
  val graph : ?loops:bool -> v:int -> e:int -> unit -> graph
  val labeled : 
    (vertex -> vertex -> edge_label) -> 
    ?loops:bool -> v:int -> e:int -> unit -> graph
end

module Generic(B : Builder.INT) = struct

  open B
  type graph = G.t
  type vertex = G.V.t
  type edge_label = G.E.label

  let fold_for i0 i1 f =
    let rec loop i v = if i > i1 then v else loop (i + 1) (f v i) in
    loop i0

  let random add_edge ~loops ~v ~e () =
    let a = Array.init v G.V.create in
    let g = Array.fold_left add_vertex (empty ()) a in
    let rec random_edge g =
      let i = Random.int v in
      let j = Random.int v in
      if (i = j && not loops) || G.mem_edge g a.(i) a.(j) then 
	random_edge g
      else
	add_edge g a.(i) a.(j)
    in
    fold_for 1 e (fun g _ -> random_edge g) g

  let graph ?(loops=false) ~v ~e () = 
    random B.add_edge ~loops ~v ~e ()
  let labeled f ?(loops=false) ~v ~e () = 
    random 
      (fun g v1 v2 -> B.add_edge_e g (G.E.create v1 (f v1 v2) v2)) 
      ~loops ~v ~e ()

end

module P (G : Sig.P with type V.label = int) = Generic(Builder.P(G))

module I (G : Sig.I with type V.label = int) = Generic(Builder.I(G))


(** Random planar graphs *)

module Planar = struct

  module type S = sig
    type graph 
    val graph : 
      ?loops:bool -> xrange:int*int -> yrange:int*int ->
	prob:float -> int -> graph
  end

  module Generic
    (G : Sig.G with type V.label = int * int and type E.label = int)
    (B : Builder.S with module G = G) = 
  struct

    type graph = G.t
    open G

    module Point = struct
      type point = V.t
      let ccw v1 v2 v3 = 
	Delaunay.IntPoints.ccw (V.label v1) (V.label v2) (V.label v3)
      let in_circle v1 v2 v3 v4 =
	Delaunay.IntPoints.in_circle 
	  (V.label v1) (V.label v2) (V.label v3) (V.label v4)
      let distance v1 v2 =
	let x1,y1 = V.label v1 in
	let x2,y2 = V.label v2 in
	let sqr x = let x = float x in x *. x in
	truncate (sqrt (sqr (x1 - x2) +. sqr (y1 - y2)))
    end

    module Triangulation = Delaunay.Make(Point)

    let graph ?(loops=false) ~xrange:(xmin,xmax) ~yrange:(ymin,ymax) ~prob v =
      if not (0.0 <= prob && prob <= 1.0) then invalid_arg "Planar.graph";
      if v < 2 then invalid_arg "Planar.graph";
      (* [v] random points and their Delaunay triangulation *)
      let random_point () =
	xmin + Random.int (1 + xmax - xmin),
	ymin + Random.int (1 + ymax - ymin)
      in
      let vertices = Array.init v (fun _ -> V.create (random_point ())) in
      let t = Triangulation.triangulate vertices in
      (* a graph with [v] vertices and random loops if any *)
      let g = Array.fold_left B.add_vertex (B.empty ()) vertices in
      let g = 
	if loops then 
	  Array.fold_left 
	    (fun g v -> 
	       if Random.float 1.0 < prob then 
		 g
	       else
		 let e = E.create v 0 v in B.add_edge_e g e)
	    g vertices
	else 
	  g
      in
      (* we keep some edges from the triangulation according to [prob] *)
      let add_edge v1 v2 g =
	if Random.float 1.0 < prob then 
	  g
	else
	  let e = E.create v1 (Point.distance v1 v2) v2 in B.add_edge_e g e
      in
      Triangulation.fold
	(fun v1 v2 g -> 
	   let g = add_edge v1 v2 g in
	   if G.is_directed then add_edge v2 v1 g else g)
	t g

  end

  module P (G : Sig.P with type V.label = int * int and type E.label = int) = 
    Generic(G)(Builder.P(G))

  module I (G : Sig.I with type V.label = int * int and type E.label = int) = 
    Generic(G)(Builder.I(G))

end
