(* $Id: netstring_str.mli,v 2.3 2002/01/12 17:58:56 stolpmann Exp $
 * ----------------------------------------------------------------------
 *
 *)

(* This module was written at a time when we had only the Str module
 * for regular expressions. However, Str has an interface that does
 * not work for multi-threaded programs, because the state of the module
 * is visible to the outside. The module Netstring_str is similar to Str,
 * but has a thread-compatible interface.
 * Now we have the much better PCRE library for regular expressions.
 * This implementation of Netstring_str bases on the PCRE engine but retains
 * the old interface as much as possible. This includes that the [regexp]
 * function parses old-style regular expressions and internally translates 
 * them to PCRE-style expressions.
 * See also Netstring_pcre. The difference is that Netstring_pcre expects
 * already PCRE-style regular expressions and does not translate them.
 *)

(* Supported operators:
 *
 * .             matches every character but newline
 * e*            matches e several times
 * e+            matches e several times but at least once
 * e?            matches e optionally
 * e{m,n}        matches e at least m times and at most n times
 * e1\|e2        matches e1 or e2
 * [set]         matches the characters from set
 * [^set]        matches the characters except from set
 * \(...\)       group paranthesis
 * \n            back reference
 * \w            matches word (= alphanumeric characters (and underscore))
 * \W            matches characters except \w
 * \b            matches at word boundary
 * \B            matches everywhere but not at word boundary
 * \<            matches at beginning of word
 * \>            matches at end of word
 * ^             matches at beginning of line
 * $             matches at end of line
 * \`            matches at beginning of buffer
 * \'            matches at end of buffer
 *
 * This is exactly what Str supports with one exception. Character classes
 * are not implemented.
 *)

type regexp = Pcre.regexp;;
  (* CHANGE: This is now Pcre.regexp *)

type split_result = Str.split_result = Text of string | Delim of string;;
  (* Here we keep compatibility with Str *)

type result;;
  (* The type of matching results *)

val regexp: string -> regexp
val regexp_case_fold: string -> regexp
val quote: string -> string
val regexp_string: string -> regexp
val regexp_string_case_fold: string -> regexp

(* Note: the ?groups argument is ignored in the following functions.
 * Once upon a time this argument determined how many groups were 
 * copied to the [result] value.
 * Now all groups are accessible in the [result] value.
 *)

val string_match: 
      ?groups:int -> regexp -> string -> int -> result option
val search_forward: 
      ?groups:int -> regexp -> string -> int -> (int * result)
val search_backward: 
      ?groups:int -> regexp -> string -> int -> (int * result)

(* NOT SUPPORTED:
 *
 *  val string_partial_match: 
 *        ?groups:int -> pat:regexp -> string -> pos:int -> result option
 *
 * The function [string_partial_match] cannot be emulated using PCRE.
 *)

val matched_string : result -> string -> string
val match_beginning : result -> int
val match_end : result -> int
val matched_group : result -> int -> string -> string
val group_beginning : result -> int -> int
val group_end : result -> int -> int

val global_replace: regexp -> (*templ:*) string -> string -> string
val replace_first: regexp -> (*templ:*) string -> string -> string
  (* In [templ] one can refer to matched groups by the backslash notation:
   * \1 refers to the first group, \2 to the second etc.
   * \0 is the whole match. \\ is the backslash character.
   *)

val global_substitute:
       ?groups:int -> 
       regexp -> (result -> string -> string) -> string -> string
val substitute_first:
       ?groups:int -> 
       regexp -> (result -> string -> string) -> string -> string

(* replace_matched: not available *)

val split: regexp -> string -> string list
val bounded_split: regexp -> string -> int -> string list
val split_delim: regexp -> string -> string list
val bounded_split_delim: regexp -> string -> int -> string list
val full_split: regexp -> string -> split_result list
val bounded_full_split: regexp -> string -> int -> split_result list

val string_before: string -> int -> string
val string_after: string -> int -> string
val first_chars: string -> int -> string
val last_chars: string -> int -> string

(* Private: *)

(* NOT NECESSARY:
 * val init_mt : (unit -> unit) -> (unit -> unit) -> unit
 * In previous versions of Netstring_str it was necessary to intialize the
 * module in a special way when used in multi-threaded environment. This
 * is no longer necessary.
 *)

(* ======================================================================
 * History:
 * 
 * $Log: netstring_str.mli,v $
 * Revision 2.3  2002/01/12 17:58:56  stolpmann
 * 	Removed the labels, following the Str update in O'Caml 3.04.
 *
 * Revision 2.2  2001/11/07 00:31:42  stolpmann
 * 	Reimplementation of the module: It bases now on PCRE, and
 * translates Str-style regular expressions to PCRE-style regular expressions.
 *
 * Revision 2.1  2001/09/14 14:22:34  stolpmann
 * 	Initial revision (sourceforge)
 *
 *
 * ======================================================================
 * Revision 1.2  2000/06/25 21:15:48  gerd
 * 	Checked thread-safety.
 *
 * Revision 1.1  2000/06/25 20:48:19  gerd
 * 	Initial revision.
 *
 * 
 *)
