/* OpenCP Module Player
 * copyright (c) '94-'98 Niklas Beisert <nbeisert@physik.tu-muenchen.de>
 *
 * GMIPlay timidity PAT format handler (reads GUS patches etc)
 *
 * revision history: (please note changes here)
 *  -sss050415 Stian Skjelstad <stian@nixia.no>
 *    -first release
 */

#include "config.h"
#include "types.h"
#include "gmipat.h"
#include "gmiplay.h"
#include <ctype.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include "stuff/err.h"

#define DirStackSize 5
#define maxlevel 10

static char DirectoryStack[DirStackSize][PATH_MAX+1];
static int DirectoryStackIndex;

static void parse_config(FILE *input, int level)
{
	const char *home=getenv("HOME");
	char line[1024], *pos, *base;
	int type=0;
	while (fgets(line, sizeof(line), input))
	{
		if ((pos=strchr(line, '\r')))
			*pos=0;
		if ((pos=strchr(line, '\n')))
			*pos=0;
		if ((pos=strchr(line, '#')))
			*pos=0;
		base=line;
		while ((*base)&&(*base==' '))
			base++;
		if (!base)
			continue;
		if (!strncmp(base, "dir ", 4))
		{
			int pos;
			if (DirectoryStackIndex==DirStackSize)
			{
				fprintf(stderr, "[timidity] Too many dir sections in config\n");
				continue;
			}
			pos=0;
			base+=4;
			while (*base)
			{
				if ((*base)=='~')
				{
					if ((pos+strlen(home))>PATH_MAX)
					{
						fprintf(stderr, "[timidity] a dir entry is too long\n");
						goto no_add_dir;
					}
					strcpy(DirectoryStack[DirectoryStackIndex]+pos, home);
					pos+=strlen(home);
				} else
					DirectoryStack[DirectoryStackIndex][pos++]=*base;
				base++;
			}
			DirectoryStack[DirectoryStackIndex++][pos]=0;
no_add_dir:{}
		} else if (!strncmp(base, "source ", 7))
		{
			char path[PATH_MAX+1];
			int i;
			if (level==maxlevel)
			{
				fprintf(stderr, "[timidity] Too high nesting level of config-files");
				continue;
			}
			base+=7;
			for (i=DirectoryStackIndex-1;i>=0;i--)
			{
				fprintf(stderr, "[timidity]: Directorystack %d is %s\n", i, DirectoryStack[i]);
				if ((strlen(base)+strlen(DirectoryStack[i]+1))<=PATH_MAX)
				{
					FILE *file;
					strcpy(path, DirectoryStack[i]);
					strcat(path, "/");
					strcat(path, base);
					if ((file=fopen(path, "r")))
					{
						fprintf(stderr, "[timidity] parsing %s\n", path);
						parse_config(file, level+1);
						fclose(file);
						break;
					}
				}
			}
			if (i<0)
				fprintf(stderr, "[timidity] Failed to find file for source '%s'\n", base);
		} else if (!strncmp(base, "drumset ", 8))
		{
			type=0;
			base+=8;
			while ((*base)&&isspace(*base))
				base++;
			if (*base)
				if (isdigit(*base)&&!atoi(base))
					type=2;
		} else if (!strncmp(base, "bank ", 5))
		{
			type=0;
			base+=5;
			while ((*base)&&isspace(*base))
				base++;
			if (*base)
				if (isdigit(*base)&&!atoi(base))
					type=1;
		} else if (isdigit(*base)&&type)
		{
			unsigned long insnum=strtoul(base, 0, 10)+((type==2)?128:0);
			if (insnum<256)
			{
				while ((*base)&&isdigit(*base))
					base++;
				while ((*base)&&isspace(*base))
					base++;
				if (*base)
				{
					while ((*base)&&isspace(*base))
						base++;
					for(pos=base;!isspace(*pos);pos++)
					{
						if (!*pos)
						{
							pos=0;
							break;
						}
					}
					if (pos)
						*(pos++)=0;
					snprintf(midInstrumentNames[insnum], sizeof(midInstrumentNames[insnum]), "%s", base); /* we ignore parameters for now TODO */
				}
			}
		}
	}
}

static int loadpatchTimidity( struct minstrument *ins,
                              uint8_t             program,
                              uint8_t            *sampused,
                              struct sampleinfo **smps,
                              uint16_t           *samplenum)
{
	int i;
      	int file=-1, retval;
	char path[PATH_MAX+NAME_MAX+1];

	ins->sampnum=0;
	*ins->name=0;

	if (!*midInstrumentNames[program])
	{
		fprintf(stderr, "[timidity] not entry configured for program %d\n", program);
		return errFileMiss;
	}
	for (i=DirectoryStackIndex-1;i>=0;i--)
	{
		snprintf(path, sizeof(path), "%s/%s.pat", DirectoryStack[i], midInstrumentNames[program]);
		if ((file=open(path, O_RDONLY))>=0)
			break;
	}
	if (file<0)
	{
		fprintf(stderr, "[timidity] '%s': failed to open file\n", midInstrumentNames[program]);
		return errFileMiss;
	}
	fprintf(stderr, "[timidity] loading file %s\n", path);
	retval=loadpatchPAT(file, ins, program, sampused, smps, samplenum);
	close(file);
	if (retval)
		fprintf(stderr, "Invalid PAT file\n");
	return retval;
}

static int addpatchTimidity( struct minstrument *ins,
                             uint8_t             program,
                             uint8_t             sn,
                             uint8_t             sampnum,
                             struct sampleinfo  *sip,
                             uint16_t           *samplenum)
{
	int file=-1, retval;
	char path[PATH_MAX+NAME_MAX];
	int i;

	if (!*midInstrumentNames[program])
	{
		fprintf(stderr, "[timidity] not entry configured for program %d\n", program);
		return errFileMiss;
	}
	for (i=DirectoryStackIndex-1;i>=0;i--)
	{
		snprintf(path, sizeof(path), "%s/%s.pat", DirectoryStack[i], midInstrumentNames[program]);
		if ((file=open(path, O_RDONLY))>=0)
			break;
	}
	if (file<0)
	{
		fprintf(stderr, "[timidity] '%s': failed to open file\n", midInstrumentNames[program]);
		return errFileMiss;
	}

	fprintf(stderr, "[timidity] loading file %s\n", path);
	retval=addpatchPAT(file, ins, program, sn, sampnum, sip, samplenum);
	close(file);
	if (retval)
		fprintf(stderr, "Invalid PAT file\n");
	return retval;
}

int __attribute__ ((visibility ("internal"))) midInitTimidity(void)
{
	FILE *inifile;
	int i;

	_midClose=0;

	for (i=0; i<256; i++)
		midInstrumentNames[i][0]=0;

	DirectoryStackIndex=0;

	if ((inifile=fopen("/etc/timidity.cfg", "r")))
	{
		fprintf(stderr, "[timidity] parsing /etc/timitidy.cfg\n");
	} else if ((inifile=fopen("/usr/local/etc/timidity.cfg", "r")))
	{
		fprintf(stderr, "[timidity] parsing /usr/local/etc/timitidy.cfg\n");
	} else if ((inifile=fopen("/usr/share/timidity/timidity.cfg", "r")))
	{
		fprintf(stderr, "[timidity] /usr/share/timidity/timidity.cfg\n");
	} else if ((inifile=fopen("/usr/local/share/timidity/timidity.cfg", "r")))
	{
		fprintf(stderr, "[timidity] /usr/local/share/timidity/timidity.cfg\n");
	} else {
		fprintf(stderr, "[timididy] failed to open /etc/timidity.cfg\n");
		return 0;
	}
	parse_config(inifile, 0);
	fclose(inifile);
	loadpatch = loadpatchTimidity;
	addpatch = addpatchTimidity;
	return 1;
}
