/*  GNU Ocrad - Optical Character Recognition program
    Copyright (C) 2003, 2004, 2005 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <algorithm>
#include <cctype>
#include <cstdio>
#include <stack>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "vrhomboid.h"
#include "track.h"
#include "bitmap.h"


namespace {

void enlarge( std::vector< std::vector< bool > > & data, const int n ) throw()
  {
  if( n < 2 ) return;
  const int height = data.size(), width = data[0].size();
  std::vector< std::vector< bool > > new_data;
  new_data.reserve( n * height );

  for( int row = 0; row < height; ++row )
    {
    new_data.push_back( std::vector< bool >() );
    for( int col = 0; col < width; ++col )
      {
      bool d = data[row][col];
      for( int i = 0; i < n; ++i ) new_data.back().push_back( d );
      }
    for( int i = 1; i < n; ++i ) new_data.push_back( new_data.back() );
    }
  swap( data, new_data );
  }


void enlarge2( std::vector< std::vector< bool > > & data ) throw()
  {
  const int height = data.size(), width = data[0].size();
  std::vector< std::vector< bool > > new_data( 2 * height );

  for( unsigned int row = 0; row < new_data.size(); ++row )
    new_data[row].resize( 2 * width, false );

  for( int row = 0; row < height; ++row )
    {
    const std::vector< bool > & datarow = data[row];
    std::vector< bool > & new_datarow0 = new_data[2*row];
    std::vector< bool > & new_datarow1 = new_data[2*row+1];
    for( int col = 0; col < width; ++col )
      if( datarow[col] )
        {
        if( row > 0 )
          {
          if( col > 0 &&
              ( data[row-1][col] || data[row-1][col-1] || datarow[col-1] ) )
            new_datarow0[2*col] = true;
          if( col < width - 1 &&
              ( data[row-1][col] || data[row-1][col+1] || datarow[col+1] ) )
            new_datarow0[2*col+1] = true;
          }
        if( row < height - 1 )
          {
          if( col > 0 &&
              ( data[row+1][col] || data[row+1][col-1] || datarow[col-1] ) )
            new_datarow1[2*col] = true;
          if( col < width - 1 &&
              ( data[row+1][col] || data[row+1][col+1] || datarow[col+1] ) )
            new_datarow1[2*col+1] = true;
          }
        }
    }
  swap( data, new_data );
  }


void mirror_left_right( std::vector< std::vector< bool > > & data ) throw()
  {
  int height = data.size();
  for( int row = 0; row < height; ++row )
    reverse( data[row].begin(), data[row].end() );
  }


void mirror_top_bottom( std::vector< std::vector< bool > > & data ) throw()
  {
  for( int u = 0, d = data.size() - 1; u < d; ++u, --d )
    swap( data[u], data[d] );
  }


void mirror_diagonal( std::vector< std::vector< bool > > & data,
                      int & _height, int & _width ) throw()
  {
  int size = std::max( _width, _height );

  if( _height < size )
    {
    data.resize( size );
    for( int row = _height; row < size; ++row )
      data[row].resize( size );
    }
  else if( _width < size )
    for( int row = 0; row < _height; ++row )
      data[row].resize( size );

  for( int row = 0; row < size; ++row )
    for( int col = 0; col < row; ++col )
      {
      bool b = data[row][col];
      data[row][col] = data[col][row]; data[col][row] = b;
      }

  int tmp = _width; _width = _height; _height = tmp;
  if( _height < size ) data.resize( _height );
  else if( _width < size )
    for( int row = 0; row < _height; ++row )
      data[row].resize( _width );
  }

} // end namespace


void Bitmap::draw_rectangle( const Rectangle & re ) throw()
  {
  int l = std::max( 0, re.left() );
  int t = std::max( 0, re.top() );
  int r = std::min( _width - 1, re.right() );
  int b = std::min( _height - 1, re.bottom() );
  if( l == re.left() ) for( int row = t; row <= b; ++row ) data[row][l] = true;
  if( t == re.top() ) for( int col = l; col <= r; ++col ) data[t][col] = true;
  if( r == re.right() ) for( int row = t; row <= b; ++row ) data[row][r] = true;
  if( b == re.bottom() ) for( int col = l; col <= r; ++col ) data[b][col] = true;
  }


void Bitmap::draw_track( const Track & tr ) throw()
  {
  int l = std::max( 0, tr.left() );
  int r = std::min( _width - 1, tr.right() );
  if( l == r ) return;
  if( l == tr.left() )
    for( int row = tr.top(l); row <= tr.bottom(l); ++row )
      if( row >= 0 && row < _height ) data[row][l] = true;
  if( r == tr.right() )
    for( int row = tr.top(r); row <= tr.bottom(r); ++row )
      if( row >= 0 && row < _height ) data[row][r] = true;
  for( int col = l; col <= r; ++col )
    {
    int row = tr.top( col );
    if( row >= 0 && row < _height ) data[row][col] = true;
    row = tr.bottom( col );
    if( row >= 0 && row < _height ) data[row][col] = true;
    }
  }


bool Bitmap::scale( const int n, const int th ) throw()
  {
  if( n <= -2 )
    { Bitmap reduced( *this, -n, th ); *this = reduced; return true; }
  if( n >= 2 )
    {
    int n2 = 0, nr = n;
    while( ( nr & 1 ) == 0 ) { ++n2; nr /= 2; }
    enlarge( data, nr );
    while( --n2 >= 0 ) enlarge2( data );
    _height = data.size(); _width = data[0].size();
    return true;
    }
  return false;
  }


void Bitmap::transform( const Transformation & t ) throw()
  {
  switch( t.type() )
    {
    case Transformation::none:
      break;
    case Transformation::rotate90:
      mirror_diagonal( data, _height, _width ); mirror_top_bottom( data ); break;
    case Transformation::rotate180:
      mirror_left_right( data ); mirror_top_bottom( data ); break;
    case Transformation::rotate270:
      mirror_diagonal( data, _height, _width ); mirror_left_right( data ); break;
    case Transformation::mirror_lr:
      mirror_left_right( data ); break;
    case Transformation::mirror_tb:
      mirror_top_bottom( data ); break;
    case Transformation::mirror_d1:
      mirror_diagonal( data, _height, _width ); break;
    case Transformation::mirror_d2:
      mirror_diagonal( data, _height, _width );
      mirror_left_right( data ); mirror_top_bottom( data ); break;
    }
  }
