## Copyright (C) 2004  Dragan Tubic
## 
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2, or (at your option)
## any later version.
## 
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
## General Public License for more details. 
## 
## You should have received a copy of the GNU General Public License
## along with this file.  If not, write to the Free Software Foundation,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

## -*- texinfo -*-
## @deftypefn {Function File} {} vtk_arrows3(@var{x},@var{y},@var{z},@var{nx},@var{ny},@var{nz}, [@var{fmt} | @var{prop},@var{val}])
## Displayes oriented cones at points defined by triplets (@var{x},@var{y},@var{z}).
## Cones are oriented along the vectors defined by triplets (@var{nx},@var{ny},@var{nz})
##
## The optional @var{fmt} takes the form of a "line specification"; e.g.
## "r" would cause the arrows to be red.  The optional
## @var{prop},@var{val} is a property,value pair argument. Valid
## properties are ";Color;Height;Radius;".
## @end deftypefn
## @seealso{vtk_plot3,vtk_get_line_spec}

## Author: Dragan Tubic


function vtk_cone3(varargin)
  
  valid_props = ";Color;Height;Radius;";
  [no_numerical_params, first_prop_index, line_spec_index] = vtk_parse_params(valid_props, all_va_args);
  
  if ( no_numerical_params < 6 )
    error("Syntax is vtk_plot3(x,y,z,nx,ny,nz)");
  end
  
  px = nth (varargin,1);
  py = nth (varargin,2);
  pz = nth (varargin,3);
  pu = nth (varargin,4);
  pv = nth (varargin,5);
  pw = nth (varargin,6);
  
  [nr nc] = size(pz);
  
  x = px(:);
  y = py(:);
  z = pz(:);
  u = pu(:);
  v = pv(:);
  w = pw(:);

  
  ## set default marker type and color
  marker_type = "*";
  color = [0 0 1];   # same as default in vtk_get_line_spec function

  ## use "line specs", if provided
  if ( line_spec_index > 0 )
    line_spec = nth (varargin,line_spec_index);
    [color, marker_type, line_style] = vtk_get_line_spec( line_spec );
  end
  ## use "property specs", if provided
  if ( first_prop_index > 0 )
    properties = struct(varargin{first_prop_index:length(varargin)});
    if ( isfield(properties,"Color") )
      color = properties.Color;
    end
  end
  properties.Color = color;

  
  vtk_init;	
  
  if ( length(x) != length(y) | length(x) != length(z) | length(x) != length(u) )
    error("Lengths of all three coordinates have to be the same.");
  end
  
  pts = [x y z]';
  
  coords = vtkFloatArray; 
  coords.SetNumberOfTuples( length(x) );
  coords.SetNumberOfComponents(3);
  coords.SetArray( pts(:), 3*length(x), 0 );
  points   = vtkPoints();
  points.SetData(coords);
  surface = vtkPolyData();
  surface.SetPoints(points);
  
  [nr nc] = size(pz);
  len = nr*nc*3;		
  vect = zeros(len,1);
  vect(1:3:len) = u;
  vect(2:3:len) = v;
  vect(3:3:len) = w;
  
  vectors = vtkFloatArray;
  vectors.SetArray(vect,nr*nc,0);
  vectors.SetNumberOfComponents(3);
  
  surface.GetPointData().SetVectors(vectors);
  
  arr = vtkConeSource();
  arr.SetResolution(12);
  if struct_contains(properties,"Height")
    arr.SetHeight(properties.Height);
  end
  if struct_contains(properties,"Radius")
    arr.SetRadius(properties.Radius);
  end
  glyph = vtkGlyph3D();
  glyph.SetInput(surface);
  glyph.SetSource(arr.GetOutput());
  glyph.SetVectorModeToUseVector()
  glyph.SetScaleModeToDataScalingOff();
  
  arrowMapper = vtkPolyDataMapper();
  arrowMapper.SetInput(glyph.GetOutput());
  arrowActor = vtkActor();
  arrowActor.SetMapper( arrowMapper );
  if struct_contains(properties,"Color")
    c = properties.Color;
    arrowActor.GetProperty().SetDiffuseColor(c(1),c(2),c(3));
  else
    arrowActor.GetProperty().SetDiffuseColor(1,1,1);
  end
  if struct_contains(properties,"Opacity")
    arrowActor.GetProperty().SetOpacity(properties.Opacity);
  else
    arrowActor.GetProperty().SetOpacity(1);
  end
  
  f = vtk_figure(0);
  f.renderer.AddProp(arrowActor);
  vtk_update(f);
  
endfunction
