## Copyright (C) 2004  Dragan Tubic
## 
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2, or (at your option)
## any later version.
## 
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
## General Public License for more details. 
## 
## You should have received a copy of the GNU General Public License
## along with this file.  If not, write to the Free Software Foundation,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

## -*- texinfo -*-
## @deftypefn {Function File} {} vtk_axis([@var{limits}],[@var{options}])
##
## Set or alter axis limits for plots.
##
## The argument @var{limits} should be a 6 element vector.  The first
## and second elements specify the lower and upper limits for the x
## axis, the third and fourth specify the limits for the y axis, and the
## fifth and sixth specify the limits for the z axis.
##
## The following are valid options:
##
## @table @code
## @item "square"
## Set a square aspect ratio.
## @item "equal"
## Set x, y, and z scaling to be equal.
## @item "auto"
## Set limits so that all the data is visible.
## @item "tight"
## Take up any extra space so that the axes are "tight" around the plot.
## @item "default"
## Set axes limits and scales to vtk default ("auto" and "equal")
## @item "on"
## Turn tic marks and labels on for all axes.
## @item "off"
## Turn tic marks and labels off for all axes.
## @end table
##
## @end deftypefn
## @seealso{vtk_xlim,vtk_xlabel,vtk_scalarbar}

## Author: Jonathan Stickel


function vtk_axis(varargin)

  f = vtk_figure(0);
  
  for i=1:nargin
    arg = varargin{i};

    if isstr(arg)

      if ( strcmp(arg, "on") )
	f.axes_on = 1;

      elseif ( strcmp(arg, "off") )
	f.axes_on = 0;

      elseif ( strcmp(arg, "default") ) ## default of vtk
	vtk_axis("equal")
	vtk_axis("auto")

      elseif ( strcmp(arg, "equal") )
	## reset scaling
	for i = 1:f.renderer.GetActors.GetNumberOfItems-1            # outline actor is i=0
	  f.renderer.GetActors.GetItemAsObject(i).SetScale(1,1,1)
	endfor
	f.axes.UseRangesOff

      elseif ( strcmp(arg, "square") )
	vtk_axis("equal")    # get to a known scale
	dlims = f.outline.GetBounds;
	span = [dlims(2)-dlims(1);dlims(4)-dlims(3);dlims(6)-dlims(5)];
	## set axis limits to data values (not visible scaled values)
	f.axes.UseRangesOn
	f.axes.SetRanges(dlims)
	## scale the plot so that it is square
	for i = 1:f.renderer.GetActors.GetNumberOfItems-1
	  f.renderer.GetActors.GetItemAsObject(i).SetScale(1/span(1),1/span(2),1/span(3))
	endfor

      elseif ( strcmp(arg, "auto") ) ## reset axes bounds to limits of actors
	## first determine if square option has been used
	rescale = f.axes.GetUseRanges;
	## check if there has been clipping (as determined by some actors not being visible),
	clipping = 0;
	for i = 1:f.renderer.GetActors.GetNumberOfItems-1  # vtk counting starts at 0
	  actor = f.renderer.GetActors.GetItemAsObject(i);
	  if (!actor.GetVisibility)
	    clipping = 1;
	    break
	  endif
	endfor
	## remove visible actors (the clipped ones) and make visible the others (unclipped)
	if (clipping)
	  i = 1;
	  while i < f.renderer.GetActors().GetNumberOfItems();
	    actor = f.renderer.GetActors.GetItemAsObject(i);
	    if (actor.GetVisibility)
	      f.renderer.RemoveViewProp(actor);
	      --i;
	    else
	      actor.VisibilityOn
	    endif
	    ++i;
	  endwhile
	endif
	if (rescale)
	  vtk_axis("square")
	endif

      elseif ( strcmp(arg, "tight") )  ## keep cuts, but otherwise bring axes tight
	## find current limits and limits for "auto"
	if (f.axes.GetUseRanges)
	  limits1 = f.axes.GetRanges;
	else
	  limits1 = f.outline.GetBounds;
	endif
	vtk_axis("auto")
	if (f.axes.GetUseRanges)
	  limits2 = f.axes.GetRanges;
	else
	  limits2 = f.outline.GetBounds;
	endif
	## reduce, but do not expand, the limits from what they were
	for i = 1:2:5
	  limits(i) = max( limits1(i),limits2(i) );
	endfor
	for i = 2:2:6
	  limits(i) = min( limits1(i),limits2(i) );
	endfor
	vtk_axis(limits)

      else
	printf("Option '%s' is not implemented;\n", arg)
	disp("If you would like it to be, please contact the authors.")
      endif
      	
    else  # input is not a string; force the specified axes limits
      ## first determine if square option has been used
      rescale = f.axes.GetUseRanges;
      ## return to original state
      vtk_axis("default")
      ## clip everything to desired axes limits; how to do this elegently?
      ## this works, but it is slow and may lead to a memory leak
      limits = arg;
      for i = 1:f.renderer.GetActors.GetNumberOfItems-1
	clipmapper{i} = vtkPolyDataMapper;
	clipactor{i} = vtkActor;
	unclipactor = f.renderer.GetActors.GetItemAsObject(i);
 	clipactor{i}.ShallowCopy(unclipactor);               # copy actor properties (e.g. color)
	clipmapper{i}.ShallowCopy(unclipactor.GetMapper);    # copy mapper properties (e.g. lookup table)
	unclipactor.VisibilityOff;                           # turn unclipped actor off
 	clippd{i} = clipaxes(clipactor{i}.GetMapper.GetInput, limits);
  	clipmapper{i}.SetInput(clippd{i});
  	clipactor{i}.SetMapper(clipmapper{i});
  	f.renderer.AddActor(clipactor{i});
      endfor
      ## add an actor that contains only corner points for the specified limits
      ## this will force axes as wide as specified
      cornerpoints = vtkPoints;
      cornerpoly = vtkPolyData;
      cornermapper = vtkPolyDataMapper;
      corneractor = vtkActor;
      limvals = reshape (limits,2,3)';
      for i = 1:2
	for j = 1:2
	  for k = 1:2
	    cornerpoints.InsertNextPoint(limvals(1,i),limvals(2,j),limvals(3,k));
	  endfor
	endfor
      endfor
      cornerpoly.SetPoints(cornerpoints);
      cornermapper.SetInput(cornerpoly);
      corneractor.SetMapper(cornermapper);
      f.renderer.AddActor(corneractor);
      if (rescale)
	vtk_axis("square")
      endif
    endif
  endfor
  
  vtk_update(f);
  
endfunction


function clippd = clipaxes(polydata,limits)
  plane = vtkPlane();
  clippoly = vtkClipPolyData;
  clippoly.SetClipFunction(plane);
  clippd = vtkPolyData;
  clippoly.GenerateClippedOutputOn();

  ## clip xlow
  plane.SetOrigin(limits(1), 0, 0);
  plane.SetNormal(1, 0, 0);
  clippoly.SetInput(polydata);
  clippoly.Update;
  clippd.DeepCopy(clippoly.GetOutput);
  ## clip xhigh
  plane.SetOrigin(limits(2), 0, 0);
  plane.SetNormal(-1, 0, 0);
  clippoly.SetInput(clippd)
  clippoly.Update;
  clippd.DeepCopy(clippoly.GetOutput);
  ## clip ylow
  plane.SetOrigin(0, limits(3), 0);
  plane.SetNormal(0, 1, 0);
  clippoly.SetInput(clippd)
  clippoly.Update;
  clippd.DeepCopy(clippoly.GetOutput);
  ## clip yhigh
  plane.SetOrigin(0, limits(4), 0);
  plane.SetNormal(0, -1, 0);
  clippoly.SetInput(clippd)
  clippoly.Update;
  clippd.DeepCopy(clippoly.GetOutput);
  ## clip zlow
  plane.SetOrigin(0, 0, limits(5));
  plane.SetNormal(0, 0, 1);
  clippoly.SetInput(clippd)
  clippoly.Update;
  clippd.DeepCopy(clippoly.GetOutput);
  ## clip zhigh
  plane.SetOrigin(0, 0, limits(6));
  plane.SetNormal(0, 0, -1);
  clippoly.SetInput(clippd)
  clippoly.Update;
  clippd.DeepCopy(clippoly.GetOutput);
endfunction
