## Copyright (C) 2004  Dragan Tubic
## 
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2, or (at your option)
## any later version.
## 
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
## General Public License for more details. 
## 
## You should have received a copy of the GNU General Public License
## along with this file.  If not, write to the Free Software Foundation,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

## -*- texinfo -*-
## @deftypefn {Function File} {@var{f} =} vtk_trisurf(@var{t},@var{x},@var{y},@var{z},@var{c},[@var{prop},@var{val}])
## Displays a surface defined by specified triangles.
## @var{t} gives a list of vertex indexes forming the triangles.
## @var{x},@var{y},@var{z} are vectors defining the vertices.  @var{c} is a vector
## of color scalar values for the vertices and may be omitted.
## The vtk_figure object is returned in f, if desired.
## trisurf can be called with 2(3) or 4(5) parameters as
## trisurf(t,[x1 y1 z1; ...; xn, yn, zn],(c)) or
## trisurf(t,[x1; ...; xn],[y1; ...; yn], [z1; ...; zn],(c))
##
## Optional @var{prop},@var{val} arguments can be used to change
## properties of the plot.  Currently, valid properties are
## ";Opacity;Color;Interpolation;".
##
## @end deftypefn
## @seealso{vtk_trimesh, vtk_surf}

## Author: Dragan Tubic

function f = vtk_trisurf( varargin )
  valid_props = ";Opacity;Color;Interpolation;";
  
  [no_numerical_params, first_prop_index, line_spec_index] = vtk_parse_params(valid_props, varargin{:});
  if ( first_prop_index > 0 )
    properties = struct(varargin{first_prop_index:length(varargin)});
  else
    properties.empty = 1;
  end
    
  if ( no_numerical_params < 2 | no_numerical_params > 5 )
    error("Syntax is trimesh( t, x, y, z, c )");
  end
  
  if ( no_numerical_params == 2 )
    t = nth (varargin,1);
    pts = nth (varargin,2);
    [nr nc] = size(pts);
    if ( nr ~= 3 )
      pts = pts';
      if ( nc ~= 3 )
	error ("p has to be Nx3 matrix!");
      end
    end
    c = pts(:,3);
  elseif ( no_numerical_params == 3 )
    t = nth (varargin,1);
    pts = nth (varargin,2);
    [nr nc] = size(pts);
    if ( nr ~= 3 )
      pts = pts';
      if ( nc ~= 3 )
	error ("p has to be Nx3 matrix!");
      end
    end
    %% Check scalars
    c = nth (varargin,2);
    [nr nc] = size(c);
    if ( nr ~= 1 )
      c = c';
      if ( nc ~= 1 )
	error ("c has to be Nx1 matrix!");
      end
    end
  elseif ( no_numerical_params == 4 )
    t = nth (varargin,1);
    x = nth (varargin,2);
    y = nth (varargin,3);
    z = nth (varargin,4);
    if ( length(x) ~= length(y) | length(x) ~= length(z) )
      error("Lengths of all three coordinate arrays have to be the same.");
    end
    [nr, nc] = size(x);
    if ( nc ~= 1 )
      x = x';
      y = y';
      z = z';
      if ( nr ~= 1 )  
	error ("x has to be Nx1 matrix!");
      end
    end
    pts = [x y z];
    c = z;
  elseif ( no_numerical_params == 5 )
    t = nth (varargin,1);
    x = nth (varargin,2);
    y = nth (varargin,3);
    z = nth (varargin,4);
    c = nth (varargin,5);
    if ( length(x) ~= length(y) | length(x) ~= length(z) | length(x) ~= length(c) )
      error("Lengths of all three coordinate arrays have to be the same.");
    end
    [nr, nc] = size(x);
    if ( nc ~= 1 )
      x = x';
      y = y';
      z = z';
      if ( nr ~= 1 )  
	error ("x has to be Nx1 matrix!");
      end
    end
    pts = [x y z];
  end
  
  vtk_init;	
  
  %% We'll create the building blocks of polydata including data attributes.
  surface  = vtkPolyData();
  points   = vtkPoints();
  polys    = vtkCellArray();
  scalars  = vtkFloatArray();
  
  %% Load the point, cell, and data attributes.
  coords = vtkFloatArray; 
  [no_pts, tmp] = size(pts);
  coords.SetNumberOfTuples( no_pts );
  coords.SetNumberOfComponents(3);
  pts = pts';
  coords.SetArray( pts(:), 3*no_pts, 0 );
  points.SetData(coords);
  
  [nr nc] = size(t);
  if ( nc ~= 3 )
    if ( nr == 3 )
      t=t';
      [nr nc] = size(t);
    else
      error("t has to be a Nx3 matrix");
    endif
  endif
  
  no_tris = length(t);
  ## check whether t has invalid indices (prevent segfaults), JJS 1/22/07
  npts = length(x);
  tmin = min(t(:));
  tmax = max(t(:));
  if ( tmin < 0 )
    error ("negative value indices detected")
  elseif ( tmin==0 )
    if ( tmax>npts-1 )
      error ("maximum index value is larger than the number of points")
    endif
    t = [ones(no_tris,1)*3 t]';
    warning ("using zero value indices")
  else
    if ( tmax>npts )
      error ("maximum index value is larger than the number of points")
    endif
    t = [ones(no_tris,1)*3 t-1]';
  endif
  ptids = vtkIdTypeArray;
  ptids.SetArray( t(:), no_tris*4, 0 );
  polys.SetCells( no_tris, ptids );
  
  ##in VTK, by default red = low values, blue = high values; we reverse that here
  ##c=-c;
  ## manually create color map table instead; a must if later add a scalar bar
  lut = vtkLookupTable();
  lut.SetHueRange(0.66667, 0.0);
  lut.SetNumberOfColors(256);
  lut.SetRampToLinear();
  lut.Build();
  scalars.SetArray( c, length(c) , 0 );
  
  %% We now assign the pieces to the vtkPolyData.
  surface.SetPoints(points);
  surface.SetPolys(polys);
  
  if !isfield(properties,"Color")
    surface.GetPointData().SetScalars(scalars);
  end
  
  %% necessary for smooth gouraud shading
  normals = vtkPolyDataNormals();
  normals.SetInput(surface);
  normals.SetFeatureAngle(60);  % limit for sharp edge

  %% Now we'll look at it.
  surfaceMapper = vtkPolyDataMapper();
  ##surfaceMapper.SetInput(surface);
  surfaceMapper.SetInput(normals.GetOutput());  % with normals
  surfaceMapper.SetLookupTable(lut);            % use manually created color map table
  surfaceMapper.SetScalarRange(min(c),max(c));  % relative scaling

  surfaceActor = vtkActor();
  surfaceActor.SetMapper(surfaceMapper);
  prop = surfaceActor.GetProperty();
  
  ##prop.SetInterpolationToGouraud();    %default
  if isfield(properties,"Interpolation")
    if ( properties.Interpolation == "Flat" )
       prop.SetInterpolationToFlat();
	end
  end
  
  ##prop.SetInterpolationToFlat();
  ## prop.SetDiffuse(0);
  ##prop.SetAmbient(1);
  ##prop.SetRepresentationToWireframe();

  if isfield(properties,"Opacity")
    op = properties.Opacity;
    surfaceActor.GetProperty().SetOpacity( op );
  end
  if isfield(properties,"Color")
    c = properties.Color;
    surfaceActor.GetProperty().SetDiffuseColor( c(1), c(2), c(3) );
  end
    
  f = vtk_figure(0);
  f.renderer.AddActor(surfaceActor);
  vtk_update(f);

endfunction
