// Copyright (C) 2003 Shai Ayal <shaiay@users.sourceforge.net>
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  

#include <iostream>
#include <stdio.h>
#include "figure.h"
#include "figurewindow.h"
#include "property.h"
#include "root.h"
#include "../config.h"

#define OBJ_NAME "Root"

Root::Root(ocpl::Handle Parent) : Object(Parent)
{
  _last_fig = 0;
  
  //  const char* clipping[] = {"on","off"};
  char onoff[] = "|{on}|off|";
  char offon[] = "|on|{off}|";

  SET_TYPE;
  Properties["Children"]      = new HandleVect;
  Properties["CurrentFigure"] = new HandleVectNoOwn;

  ///////////////////////////////////////////
  // Defaults
  ///////////////////////////////////////////

  // Figure
  Properties["DefaultFigurePosition"] = Real2Matrix(10,10,400,300);
  Properties["DefaultFigureColor"] =  new Color(0.9,0.9,0.9,1.0);
  // default grayscale colormap
  const int cmap_size=64;
  double* gray = new double[3*cmap_size];
  for(int i=0;i<cmap_size;i++) {
    gray[i] = gray[i+cmap_size] = gray[i+2*cmap_size] = 
      static_cast<double>(i)/(cmap_size-1);
  }
  Properties["DefaultFigureColorMap"] = new Matrix(gray,cmap_size,3);

  
  // Axes
  Properties["DefaultAxesColor"]     = new Color('w');
  Properties["DefaultAxesXColor"]    = new Color('k');
  Properties["DefaultAxesYColor"]    = new Color('k');
  Properties["DefaultAxesZColor"]    = new Color('k');
  Properties["DefaultAxesPosition"]  = Real2Matrix(0.1,0.1,0.8,0.8);
  Properties["DefaultAxesLinewidth"] = new Scalar(1.0);
  Properties["DefaultAxesTickLength"]= Real2Matrix(5,5);
  Properties["DefaultAxesGridLineStyle"]  = new Radio("|:|-|--|-.|{none}|");
  char *as="|{linear}|log|";
  Properties["DefaultAxesXScale"] = new Radio(as);
  Properties["DefaultAxesYScale"] = new Radio(as);
  Properties["DefaultAxesZScale"] = new Radio(as);
  const double color_order[] ={
    1,0,0,1,0,1,0,
    0,0,1,1,1,0,0,
    0,1,0,0,1,1,0,
    1,1,1,1,1,1,1
  };
  const int co_size = sizeof(color_order);
  double* co= new double[co_size];
  for(int i=0;i<co_size;i++) {
    co[i] = color_order[i];
  }
  Properties["DefaultAxesColorOrder"] = new Matrix(co,7,4);

  // Text
  Properties["DefaultTextColor"] =  new Color('k');
  Properties["DefaultTextClipping"] = new Radio(offon);
  Properties["DefaultTextHorizontalAlignment"] = 
    new Radio("|{left}|center|right|");
  Properties["DefaultTextVerticalAlignment"] = 
    new Radio("|top|{middle}|bottom|");
  Properties["DefaultTextFontName"] = 
    new Radio("|{sans}|mono|serif|");
  Properties["DefaultTextFontSize"] = new Scalar(10);
  Properties["DefaultTextRotation"] = new Scalar(0);

  // line & marker for types with lines
  const char* linestyle = "|{-}|--|:|-.|none|";
  const char* marker = "|{none}|+|o|*|.|x|s|d|^|v|>||<||p||h|";

  // Line
  Properties["DefaultLineColor"] =  new Color('r');
  Properties["DefaultLineMarkerEdgeColor"] =  new ColorNone('r');
  Properties["DefaultLineMarkerFaceColor"] =  new ColorNone('w');
  Properties["DefaultLineMarkerSize"] =  new Scalar(10);
  Properties["DefaultLineLineWidth"] =  new Scalar(1.0);
  Properties["DefaultLineClipping"] = new Radio(onoff);
  Properties["DefaultLineLineStyle"] = new Radio(linestyle);
  Properties["DefaultLineMarker"]    = new Radio(marker);

  // Patch
  Properties["DefaultPatchFaceColor"] = new ColorNone('r');
  Properties["DefaultPatchEdgeColor"] = new ColorNone('k');
  Properties["DefaultPatchLineStyle"] = new Radio(linestyle);
  Properties["DefaultPatchLineWidth"] =  new Scalar(1.0);
  Properties["DefaultPatchClipping"] = new Radio(onoff);
  Properties["DefaultPatchMarker"]    = new Radio(marker);
  Properties["DefaultPatchMarkerEdgeColor"] =  new ColorNone('r');
  Properties["DefaultPatchMarkerFaceColor"] =  new ColorNone('w');
  Properties["DefaultPatchMarkerSize"] =  new Scalar(10);

  // Surface
  Properties["DefaultSurfaceClipping"] = new Radio(onoff);

  // Legend
  Properties["DefaultLegendColor"]     = new Color('k');
  Properties["DefaultLegendFaceColor"] = new ColorNone('w');
  Properties["DefaultLegendEdgeColor"] = new ColorNone('k');
  Properties["DefaultLegendClipping"]  = new Radio(offon);
  Properties["DefaultLegendFontName"]  = 
    new Radio("|{sans}|mono|serif|");
  Properties["DefaultLegendFontSize"]  = new Scalar(10);
  Properties["DefaultLegendLocation"]  = 
    new Radio("|n|{ne}|e|se|s|sw|w|nw|north|northeast|east|southeast|south|southwest|west|northwest|");

  // FontDir
  Properties["DefaultFontDir"]  = new String(OPATH "/fonts/");
}

//! Addes a new Figure object, taking care of the numbering etc...
void Root::AddFigure()
{
  char Caption[25];
  MAKE_REF(children,HandleVect);
  MAKE_REF(currentfigure,HandleVectNoOwn);
  MAKE_REF(defaultfigureposition,Matrix);
  
  sprintf(Caption,"octplot - Figure %d",++ _last_fig);

  FigureWindow *nf = 
    new FigureWindow(
      (int)defaultfigureposition[0],
      (int)defaultfigureposition[1],
      (int)defaultfigureposition[2],
      (int)defaultfigureposition[3],
      Caption,
      _last_fig,
      20);
  nf->size_range(40, 40);
  nf->AddChildren(GetHandle());
  Figure* Current = nf->fig;
  children.Add(Current->GetHandle());
  Current->AddAxes();
  currentfigure.Clear();
  currentfigure.Add(Current->GetHandle());

  nf->show();
}

void Root::DeleteChild(ocpl::Handle Child)
{
  MAKE_REF(children,HandleVect);
  MAKE_REF(currentfigure,HandleVectNoOwn);
  
  Figure *fig = dynamic_cast<Figure*>(::GetObjectD(Child));
  fig->hide();
  fig->win->hide();
  delete fig->win;
  
  children.Delete(Child);
  if(currentfigure()==Child)
  {
    currentfigure.Clear();
    ocpl::Handle last=0;
    children.First();
    while(!children.IsDone()) {
        last = children.CurrentHandle();
        children.Next();
    }
    if(last) currentfigure.Add(last);
  }
}

void Root::Parse(ocpl::command& command)
{
  MAKE_REF(currentfigure,HandleVectNoOwn);

  if(command.id()==ocpl::figure) {
    command.parsed(true);
    if(command.nargin()!=0) {
      ret_error(command,"figure accepts 0 arguments");
      return;
    }
    AddFigure();
  
    command.dirty(true);
    
    //return handle
    double* hnd = new double(static_cast<double>(currentfigure()));
    command.argout(0,ocpl::real,1,1,reinterpret_cast<char*>(hnd),true);
  }
  else { // All other commands should be passed down
    
    if(!currentfigure.Size()) {
      // redraw should not create a new figure, all others shpould
      if(command.id() == ocpl::redraw) {
        command.init_argout(0);
        return;
      }
      else {
        AddFigure();
      }
    }
      
    ::GetObjectD(currentfigure())->Parse(command);
    if(command.dirty()) dirty=true;
  }
}

bool Root::PreSet(ocpl::command& com)
{
  MAKE_REF(children,HandleVect)
  
  std::string prop(tolower(com.argin(1)->data));
  
  if(prop=="currenfigure") {
    ocpl::Handle new_ca = 
      static_cast<ocpl::Handle>(*(com.argin(2)->real_data()));
    if(!children.exist(new_ca)) {
      ocpl::ret_error(com,"Can't set CurrentFigure to non child");
      return false;
    }
  }

  return true;
}
