## Copyright (C) 2006 Michel D. Schmid
##
## This program is free software; you can redistribute it and/or modify it
## under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2, or (at your option)
## any later version.
##
## OctPlot is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
## General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with OctPlot; see the file COPYING.  If not, write to the Free
## Software Foundation, 59 Temple Place - Suite 330, Boston, MA
## 02111-1307, USA.

## -*- texinfo -*-
## @deftypefn {Function File} {} @var{h}= stem ([@var{x}],@var{y},[@var{lineSpec}])
## Plots a stem graph.
## The default color is "r" (red). The default line style is "-" and the default
## marker is "o".
## Returns in @var{h} the handles of the line and marker objects used to draw the
## stems.
##
## @example
## x = 1:10;
## stem(x);
## @end example
## @noindent
## plots 10 stems with hights from 1 to 10
##
## @example
## x = 1:10;
## y = ones(1,length(x))*2.*x;
## stem(x,y);
## @end example
## @noindent
## plots 10 stems with hights from 2 to 20
##
## @example
## x = 1:10;
## y = ones(1,length(x))*2.*x;
## h = stem(x,y,'b');
## @end example
## @noindent
## plots 10 bars with hights from 2 to 20
## color is blue, 2x10 handles will be returned
## The first row holds the lines handles,
## the second rows holds the handles of the markers.
##
## @example
## x = 1:10;
## y = ones(1,length(x))*2.*x;
## h = stem(x,y,'-.k');
## @end example
## @noindent
## plots 10 stems with hights from 2 to 20
## color is black and line style is "-.", 2x10 handles will be returned
## The first row holds the lines handles,
## the second rows holds the handles of the markers.
##
## @example
## x = 1:10;
## y = ones(1,length(x))*2.*x;
## h = stem(x,y,'-.k.');
## @end example
## @noindent
## plots 10 stems with hights from 2 to 20
## color is black, line style is "-." and the marker style is ".",
## 2x10 handles will be returned
## The first row holds the lines handles,
## the second rows holds the handles of the markers.
##
## For coloring the stems, rgb-tripl isn't a valid option.
## @end deftypefn
## @seealso{bar,barh,plot}

## Author: mds
## $Revision: 1.2 $, $Date: 2006/02/14 04:29:53 $

function h = stem(varargin)

 # check number of input parameters
 error(nargchk(1,4,nargin));

 ca = gca();
 if (isempty(ca))
   cla();
 elseif (strcmp(get(ca,"nextplot"),"replace")),
   cla();
 endif

 # set specifiers to default values
 mc = "r";
 lc = "r";
 ls = "-";
 ms = "o";

 # check input arguments
 if(nargin==1)
   y=varargin{1};
   if any(size(y)==1)
     x=1:length(y);
   else
     x=1:size(y,1);
   endif

 elseif(nargin==2)
        x=varargin{1};
   y=varargin{2};
   if ischar(y)
     error("Second input argument doesn't have to be characters!")
   endif

 elseif(nargin==3)
   x=varargin{1};
   y=varargin{2};

   if ischar(varargin{3})
        # __pltopt__ parse the line specifier string
           cur_props = __pltopt__(varargin{3});
           for i=1:1:length(cur_props)
                           if strcmp(cur_props(i).name,'markercolor');
                                mc = cur_props(i).val;
                        elseif strcmp(cur_props(i).name,'color'); # means line color
                                        lc = cur_props(i).val;
                                elseif strcmp(cur_props(i).name,'linestyle');
                                   ls = cur_props(i).val;
                           elseif strcmp(cur_props(i).name,'marker');
                              ms = cur_props(i).val;
                  endif
           endfor
   else
         error("Wrong argument type, must be a string of specifiers!");
        endif
 endif

 # first, plot the lines.. without marker
 hh = plot([x(:)'; x(:)'],[zeros(size(y(:)')) ; y(:)'],[lc ls]);
 # second, plot the markers..
 hhh = [];
 hold on;
 hhh = plot([x(:)'; x(:)'],[y(:)' ; y(:)'],[mc ms]);
 hold off;

 h = [hh;hhh];

endfunction

## $Log: stem.m,v $
## Revision 1.2  2006/02/14 04:29:53  shaiay
## mds changes to stem
##

