/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "OSCController.h"
#include <list>
#include <cassert>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <unistd.h>
#include "DummyClientHooks.h"
#include "OSCListener.h"
#include "PatchModel.h"
#include "ConnectionModel.h"
#include "MetadataModel.h"
#include "PresetModel.h"
#include "ControlModel.h"
#include "NodeModel.h"
#include "PluginModel.h"

using std::cerr; using std::cout; using std::endl;

namespace LibOmClient {

	
/** Construct a OSCController with a user-provided ClientHooks object for notification
 *  of engine events.
 */
OSCController::OSCController(ClientHooks* const client_hooks)
: m_client_hooks(client_hooks),
  m_osc_listener(NULL),
  m_st(NULL),
  m_engine_addr(NULL),
  m_request_id(0),
  m_blocking(false),
  m_waiting_for_response(false),
  m_wait_response_id(0),
  m_response_received(false),
  m_wait_response_was_affirmative(false)
{
	assert(m_client_hooks != NULL);

	pthread_mutex_init(&m_response_mutex, NULL);
	pthread_cond_init(&m_response_cond, NULL);
}


/** Construct a OSCController with no ClientHooks object.
 *
 * The user will receive no notification of anything that happens in the
 * engine.  (If this seems silly, it's because I don't want to have
 * "if (m_client_hooks != NULL)" before every single call to a ClientHooks
 * method in this code).
 */
OSCController::OSCController()
: m_client_hooks(NULL),
  m_osc_listener(NULL),
  m_st(NULL),
  m_engine_addr(NULL),
  m_request_id(0),
  m_blocking(false),
  m_waiting_for_response(false),
  m_wait_response_id(0),
  m_response_received(false),
  m_wait_response_was_affirmative(false)
{	
	pthread_mutex_init(&m_response_mutex, NULL);
	pthread_cond_init(&m_response_cond, NULL);
}


OSCController::~OSCController()
{
	detach();
	pthread_mutex_destroy(&m_response_mutex);
	pthread_cond_destroy(&m_response_cond);

	delete m_osc_listener;
}


/** Register this client with the engine.
 *
 * A registered client will receive notifications of everything happening with
 * the engine (ie new nodes, etc).
 */
void
OSCController::register_client(int id)
{
	assert(m_engine_addr != NULL);

	lo_send(m_engine_addr, "/om/engine/register_client", "is", id,
		lo_server_thread_get_url(m_st));
}


/** Unregister this client from the engine.
 */
void
OSCController::unregister_client()
{
	assert(m_engine_addr != NULL);

	int id = m_request_id++;
	
	lo_send(m_engine_addr, "/om/engine/unregister_client", "is",
		id, lo_server_thread_get_url(m_st));
}


/** Connect to the Om engine and notify it of our existance.
 *
 * Optionally takes the client's address (to advertise to the engine) as
 * a parameter, useful for working around networking problems sometimes.
 *
 * Passing a client_port of 0 will automatically choose a free port.
 */
void
OSCController::attach(const string& engine_url, int client_port)
{
	m_engine_url = engine_url;
	m_engine_addr = lo_address_new_from_url(engine_url.c_str());

	const int request_id = 1;
	
	start_listen_thread(client_port);

	if (m_engine_addr == NULL) {
		cerr << "Unable to connect, aborting." << endl;
		exit(EXIT_FAILURE);
	}

	char* lo_url = lo_address_get_url(m_engine_addr);
	cout << "[OSCController] Attempting to contact engine at " << lo_url << " ";

	set_wait_response_id(request_id);
	
	while (1) {	
		pthread_mutex_lock(&m_response_mutex);
		if (!m_response_received) {	
			cout << ".";
			cout.flush();
			pthread_mutex_unlock(&m_response_mutex);
			ping(request_id);
			sleep(1);
		} else {
			cout << " connected" << endl;
			m_waiting_for_response = false;
			pthread_mutex_unlock(&m_response_mutex);
			break;
		}
	}

	free(lo_url);
}


void
OSCController::detach()
{
	if (m_st != NULL) {
		unregister_client();
		lo_server_thread_free(m_st);
		m_st = NULL;
	}
}


void
OSCController::start_listen_thread(int client_port)
{
	if (client_port == 0) { 
		m_st = lo_server_thread_new(NULL, error_cb);
	} else {
		char port_str[8];
		snprintf(port_str, 8, "%d", client_port);
		m_st = lo_server_thread_new(port_str, error_cb);
	}

	if (m_st == NULL) {
		cerr << "[OSCController] Could not start OSC listener.  Aborting." << endl;
		exit(EXIT_FAILURE);
	} else {
		cout << "[OSCController] Started OSC listener on port " << lo_server_thread_get_port(m_st) << endl;
	}

	//lo_server_thread_add_method(m_st, NULL, NULL, generic_cb, NULL);

	lo_server_thread_add_method(m_st, "/om/response/ok", "i", om_response_ok_cb, this);
	lo_server_thread_add_method(m_st, "/om/response/error", "is", om_response_error_cb, this);
	
	
	m_osc_listener = new OSCListener(m_st, m_client_hooks);
	m_osc_listener->setup_callbacks();

	// Display any uncaught messages to the console
	lo_server_thread_add_method(m_st, NULL, NULL, generic_cb, NULL);

	lo_server_thread_start(m_st);
}



///// OSC Commands /////



/** Pings the engine.
 */
void
OSCController::ping(int id)
{
	lo_send(m_engine_addr, "/om/ping", "i", id);
}


/** Activate the engine.
 */
void
OSCController::activate()
{
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/engine/activate", "i", id);
}


/** Deactivate the engine.
 */
void
OSCController::deactivate()
{
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/engine/deactivate", "i", id);
}


/** Enable DSP processing.
 */
void
OSCController::enable()
{
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/engine/enable", "i", id);
}


/** Disable DSP processing.
 */
void
OSCController::disable()
{
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/engine/disable", "i", id);
}


/** Kill the engine.
 */
void
OSCController::quit()
{
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/engine/quit", "i", id);
}


/** Load all (LADSPA, etc) plugins into the engine's database.
 */
void
OSCController::load_plugins(int id)
{
	lo_send(m_engine_addr, "/om/engine/load_plugins", "i", id);
}


/** Load a node.
 */
void
OSCController::add_node(const NodeModel* nm, int id)
{
	assert(m_engine_addr != NULL);
	
	lo_send(m_engine_addr, "/om/synth/create_node",  "issssi", id, nm->path().c_str(),
	        nm->plugin_model()->type_string(),
	        nm->plugin_model()->lib_name().c_str(), nm->plugin_model()->plug_label().c_str(),
			(nm->polyphonic() ? 1 : 0));
}


/** Remove a node.
 */
void
OSCController::remove_node(const string& node_path)
{
	int id = m_request_id++;
	assert(m_engine_addr != NULL);
	lo_send(m_engine_addr, "/om/synth/destroy_node", "is", id, node_path.c_str());
}


/** Create a patch.
 */
void
OSCController::create_patch(const PatchModel* pm, int id)
{
	assert(m_engine_addr != NULL);
	lo_send(m_engine_addr, "/om/synth/create_patch", "isi", id, pm->path().c_str(), pm->poly());
}


/** Rename an object
 *
 * old_path is a full path to the object, but new_name is just a name.  It is
 * illegal for the name to contain slashes.
 */
void
OSCController::rename(const string& old_path, const string& new_name)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/synth/rename", "iss", id, old_path.c_str(), new_name.c_str());
}


/** Destroy a patch.
 */
void
OSCController::destroy_patch(const string& patch_path)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/synth/destroy_patch", "is", id, patch_path.c_str());
}


/** Enable a patch.
 */
void
OSCController::enable_patch(const string& patch_path)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/synth/enable_patch", "is", id, patch_path.c_str());
}


/** Disable a patch.
 */
void
OSCController::disable_patch(const string& patch_path)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/synth/disable_patch", "is", id, patch_path.c_str());
}


/** Connect two ports.
 */
void
OSCController::connect(const string& port1_path, const string& port2_path)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/synth/connect", "iss", id, port1_path.c_str(), port2_path.c_str());
}


/** Disconnect two ports.
 */
void
OSCController::disconnect(const string& port1_path, const string& port2_path)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/synth/disconnect", "iss", id, port1_path.c_str(), port2_path.c_str());
}


/** Disconnect all connections from a node.
 */
void
OSCController::disconnect_all(const string& node_path)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/synth/disconnect_all", "is", id, node_path.c_str());
}


/** Set a port's value.
 */
void
OSCController::set_control(const string& port_path, const float val)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/synth/set_port_value", "isf", id, port_path.c_str(), val);
}


/** Set a port's value for a specific voice.
 */
void
OSCController::set_control(const string& port_path, int voice, const float val)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/synth/set_port_value", "isif", id, port_path.c_str(), voice, val);
}


/** Set a port's value.
 */
void
OSCController::set_control_slow(const string& port_path, const float val)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/synth/set_port_value_slow", "isf", id, port_path.c_str(), val);
}


/** Set a preset by setting all relevant controls for a patch.
 */
void
OSCController::set_preset(const string& patch_path, const PresetModel* const pm)
{
	for (list<ControlModel>::const_iterator i = pm->controls().begin(); i != pm->controls().end(); ++i) {
		set_control_slow((*i).port_path(), (*i).value());
		usleep(1000);
	}
}


/** Learn a binding for a Midi control node.
 */
void
OSCController::midi_learn(const string& node_path)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/midi/learn", "is", id, node_path.c_str());
}


/** Set/add a piece of metadata.
 */
void
OSCController::set_metadata(const string& obj_path,
                   const string& key, const string& value)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;

	// Deal with the "special" DSSI metadata strings
	if (key.substr(0, 16) == "dssi-configure--") {
		string path = "/dssi" + obj_path + "/configure";
		string dssi_key = key.substr(16);
		lo_send(m_engine_addr, path.c_str(), "ss", dssi_key.c_str(), value.c_str());
	} else if (key == "dssi-program") {
		string path = "/dssi" + obj_path + "/program";
		string dssi_bank_str = value.substr(0, value.find("/"));
		int dssi_bank = atoi(dssi_bank_str.c_str());
		string dssi_program_str = value.substr(value.find("/")+1);
		int dssi_program = atoi(dssi_program_str.c_str());
		lo_send(m_engine_addr, path.c_str(), "ii", dssi_bank, dssi_program);
	}
	
	// Normal metadata
	lo_send(m_engine_addr, "/om/metadata/set", "isss", id,
		obj_path.c_str(), key.c_str(), value.c_str());
}


/** Set all pieces of metadata in a NodeModel.
 */
void
OSCController::set_all_metadata(const NodeModel* nm)
{
	assert(m_engine_addr != NULL);

	for (map<string, string>::const_iterator i = nm->metadata().begin(); i != nm->metadata().end(); ++i)
		set_metadata(nm->path(), (*i).first, (*i).second.c_str());
}


///// Requests /////



void
OSCController::request_control(const string& port_path)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/request/port_value", "iss",
		id, m_osc_listener->listen_url().c_str(), port_path.c_str());
}


/** Requests all known plugins from the server.
 */
void
OSCController::request_plugins()
{
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/request/plugins", "is", id, m_osc_listener->listen_url().c_str());
}


/** Requests all object from the server (Nodes, Patches, Ports).
 */
void
OSCController::request_all_objects(int id)
{
	lo_send(m_engine_addr, "/om/request/all_objects", "is", id, m_osc_listener->listen_url().c_str());
}


/** Sets the response ID to be waited for on the next call to wait_for_response()
 */
void
OSCController::set_wait_response_id(int id)
{	
	pthread_mutex_lock(&m_response_mutex);

	assert(!m_waiting_for_response);
	m_wait_response_id = id;
	m_response_received = false;
	m_waiting_for_response = true;
	
	pthread_mutex_unlock(&m_response_mutex);
}


/** Waits for the response set by set_wait_response() from the server.
 *
 * Returns whether or not the response was positive (ie a success message)
 * or negative (ie an error)
 */
bool
OSCController::wait_for_response()
{
	cerr << "[OSCController] Waiting for response " << m_wait_response_id << ": ";
	bool ret = true;
	
	pthread_mutex_lock(&m_response_mutex);

	assert(m_waiting_for_response);
	
	if ( ! m_response_received) {
		while (!m_response_received) {
			pthread_cond_wait(&m_response_cond, &m_response_mutex);
			cerr << ".";
		}
	}
	cerr << " received." << endl;

	m_waiting_for_response = false;
	ret = m_wait_response_was_affirmative;
	
	pthread_mutex_unlock(&m_response_mutex);

	return ret;
}


///// Static OSC callbacks //////


void
OSCController::error_cb(int num, const char* msg, const char* path)
{
	cerr << "Got error from server: " << msg << endl;
}


int
OSCController::generic_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* user_data)
{
	printf("[OSCMsg] %s\n", path);
    
	for (int i=0; i < argc; ++i) {
		printf("         '%c'  ", types[i]);
		lo_arg_pp(lo_type(types[i]), argv[i]);
		printf("\n");
    }
    printf("\n");

	return 1;  // not handled
}


int
OSCController::unknown_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* user_data)
{
	string msg = "Received unknown OSC message: ";
	msg += path;

	cerr << msg << endl;

	return 0;
}



//// End static callbacks, member callbacks below ////


int
OSCController::m_om_response_ok_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{	
	assert(argc == 1 && !strcmp(types, "i"));
	
	if (!m_waiting_for_response)
		return 0;

	const int request_id = argv[0]->i;

	pthread_mutex_lock(&m_response_mutex);
	
	if (request_id == m_wait_response_id) {
		m_response_received = true;
		m_wait_response_was_affirmative = true;
	}
	
	pthread_cond_signal(&m_response_cond);
	pthread_mutex_unlock(&m_response_mutex);

	return 0;
}


int
OSCController::m_om_response_error_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	assert(argc == 2 && !strcmp(types, "is"));

	//cerr << argv[0]->i << ",";
	
	const int   request_id =  argv[0]->i;
	const char* msg        = &argv[1]->s;
	
	
	
	if (m_waiting_for_response) {
		pthread_mutex_lock(&m_response_mutex);

		if (request_id == m_wait_response_id) {
			m_response_received = true;
			m_wait_response_was_affirmative = false;
		}
		
		pthread_cond_signal(&m_response_cond);
		pthread_mutex_unlock(&m_response_mutex);
	}
	
	m_client_hooks->error(msg);
	
	return 0;
}


} // namespace LibOmClient
