/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "OSCController.h"
#include "PatchLibrarian.h"
#include "DemolitionClientHooks.h"
#include <iostream>
#include <unistd.h>
#include <stdlib.h>
#include "cmdline.h"  // generated by gengetopt

using std::cout;
using std::endl;

using namespace LibOmClient;

void do_something();

string random_name();

void create_patch(); 
void destroy_patch(); 
void add_node(); 
void remove_node(); 
void connect(); 
void disconnect(); 
void disconnect_all(); 
void set_control(); 
void set_control_slow(); 
void rename_object();


DemolitionModel*       model;
DemolitionClientHooks* client_hooks;
OSCController*         osc_controller;
PatchLibrarian*        librarian;
	

int
main(int argc, char** argv)
{
	const char* engine_url  = NULL;
	int         client_port = 0;

	model          = new DemolitionModel();
	client_hooks   = new DemolitionClientHooks(model);
	osc_controller = new OSCController(client_hooks);
	librarian      = new PatchLibrarian(osc_controller);
	
	/* Parse command line options */
	gengetopt_args_info args_info;
	if (cmdline_parser (argc, argv, &args_info) != 0)
		return 1;

	if (args_info.engine_url_given) {
		engine_url = args_info.engine_url_arg;
	} else {
		cout << "[Main] No engine URL specified.  Attempting to use osc.udp://localhost:16180" << endl;
		engine_url = "osc.udp://localhost:16180";
	}	

	if (args_info.client_port_given)
		client_port = args_info.client_port_arg;
	else
		client_port = 0; // will choose a free port automatically
	
	/* Connect to engine */
	osc_controller->attach(engine_url, client_port);
	osc_controller->register_client();

	osc_controller->activate();
	
	osc_controller->load_plugins();
	osc_controller->request_plugins();

	int id = osc_controller->get_next_request_id();
	osc_controller->request_all_objects(id);
	osc_controller->set_wait_response_id(id);
	osc_controller->wait_for_response();

	// Disable DSP for stress testing
	osc_controller->disable();
	
	while (true) {
		do_something();
		usleep(10000);
	}
	
	sleep(2);
	osc_controller->unregister_client();
	osc_controller->detach();

	delete librarian;
	delete osc_controller;
	delete client_hooks;
	delete model;
	
	return 0;
}


/** Does some random action
 */
void
do_something()
{
	int action = rand() % 10;

	switch(action) {
	case 0:
		create_patch(); break;
	case 1:
		destroy_patch(); break;
	case 2:
		add_node(); break;
	case 3:
		remove_node(); break;
	case 4:
		connect(); break;
	case 5:
		disconnect(); break;
	case 6:
		disconnect_all(); break;
	case 7:
		set_control(); break;
	case 8:
		set_control_slow(); break;
	case 9:
		rename_object(); break;
	default:
		break;
	}
}


string
random_name()
{
	int length = (rand()%10)+1;
	string name(length, '-');

	for (int i=0; i < length; ++i)
		name[i] = 'a' + rand()%26;

	return name;
}


void
create_patch()
{
	// Make the probability of this happening inversely proportionate to the number
	// of patches to keep the # in a sane range
	//if (model->num_patches() > 0 && (rand() % model->num_patches()))
	//	return;

	bool subpatch = rand()%2;
	PatchModel* parent = NULL;
	string name = random_name();
	PatchModel* pm = NULL;
	
	if (subpatch)
		parent = model->random_patch();

	if (parent != NULL)
		pm = new PatchModel(parent->path() +"/"+ name, (rand()%8)+1);
	else
		pm = new PatchModel(string("/") + name, (rand()%8)+1);

	cout << "Creating patch " << pm->path() << endl;

	osc_controller->create_patch(pm);
	delete pm;
}

 
void
destroy_patch()
{
	// Make the probability of this happening proportionate to the number
	// of patches to keep the # in a sane range
	if (model->num_patches() == 0 || !(rand() % model->num_patches()))
		return;

	PatchModel* pm = model->random_patch();
	if (pm != NULL) {
		cout << "Destroying patch " << pm->path() << endl;
		osc_controller->destroy_patch(pm->path());
	}
}

 
void
add_node()
{
	PatchModel*  parent = model->random_patch();
	PluginModel* plugin = model->random_plugin();
	
	if (parent != NULL && plugin != NULL) {
		NodeModel* nm = new NodeModel(parent->path() +"/"+ random_name(), 1);
		nm->plugin_model(plugin);
		cout << "Adding node " << nm->path() << endl;
		osc_controller->add_node(nm);
	}
}

 
void
remove_node()
{
	NodeModel* nm = model->random_node();

	if (nm != NULL) {
		cout << "Removing node " << nm->path() << endl;
		osc_controller->remove_node(nm->path());
	}
}

 
void
connect()
{
	if (!(rand() % 10)) {  // Attempt a connection between two nodes in the same patch
		PatchModel* parent = model->random_patch();
		NodeModel* n1 = model->random_node_in_patch(parent);
		NodeModel* n2 = model->random_node_in_patch(parent);
		PortModel* p1 = model->random_port_in_node(n1);
		PortModel* p2 = model->random_port_in_node(n2);
		
		if (p1 != NULL && p2 != NULL) {
			cout << "Connecting " << p1->path() << " -> " << p2->path() << endl;
			osc_controller->connect(p1->path(), p2->path());
		}

	} else {  // Attempt a connection between two truly random nodes
		PortModel* p1 = model->random_port();
		PortModel* p2 = model->random_port();
	
		if (p1 != NULL && p2 != NULL) {
			cout << "Connecting " << p1->path() << " -> " << p2->path() << endl;
			osc_controller->connect(p1->path(), p2->path());
		}
	}	
}

 
void
disconnect()
{
	PortModel* p1 = model->random_port();
	PortModel* p2 = model->random_port();

	if (p1 != NULL && p2 != NULL) {
		cout << "Disconnecting " << p1->path() << " -> " << p2->path() << endl;
		osc_controller->disconnect(p1->path(), p2->path());
	}
}

 
void
disconnect_all()
{
	PortModel* p = model->random_port();

	if (p != NULL) {
		cout << "Disconnecting all from" << p->path() << endl;
		osc_controller->disconnect_all(p->path());
	}
}

 
void
set_control()
{
	PortModel* pm = model->random_port();
	float val = (float)rand() / (float)RAND_MAX;
	
	if (pm != NULL) {
		cout << "Setting control for port " << pm->path() << " to " << val << endl;
		osc_controller->set_control(pm->path(), val);
	}
}

 
void
set_control_slow()
{
	PortModel* pm = model->random_port();
	float val = (float)rand() / (float)RAND_MAX;
	
	if (pm != NULL) {
		cout << "Setting control (slow) for port " << pm->path() << " to " << val << endl;
		osc_controller->set_control_slow(pm->path(), val);
	}
}


void
rename_object()
{
	// 1/6th chance of it being a patch
	/*int type = rand()%6;

	if (type == 0) {
		NodeModel* n = model->random_node();
		if (n != NULL)
			osc_controller->rename(n->path(), random_name());
	} else {
		PatchModel* p = model->random_patch();
		if (p != NULL)
			osc_controller->rename(p->path(), random_name());
	}*/
}

 
