/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef CONTROLPANEL_H
#define CONTROLPANEL_H

#include <gtkmm.h>
#include <sigc++/sigc++.h>
#include <libglademm/xml.h>
#include <libglademm.h>
#include <vector>
#include <string>
#include <iostream>
#include <utility> // for pair<>
#include "ControlGroups.h"
using std::vector; using std::string; using std::pair;
using std::cerr; using std::cout; using std::endl;

namespace LibOmClient {
class PortModel;
class NodeModel;
}
using namespace LibOmClient;

namespace OmGtk {

class PatchController;


/** A group of controls for a node (or patch).
 *
 * Used by both NodeControlWindow and the main window (for patch controls).
 */
class ControlPanel : public Gtk::VBox {
public:
	ControlPanel(BaseObjectType* cobject, const Glib::RefPtr<Gnome::Glade::Xml>& glade_xml);
	virtual ~ControlPanel();
	
	void init(PatchController* patch_controller, NodeModel* node_model, bool voice_controls);

	ControlGroup* find_port(const string& path);

	void add_port(PortModel* pm);
	void remove_port(const string& path);

	void rename_port(const string& old_path, const string& new_path);

	void enable_port(const string& path);
	void disable_port(const string& path);
	
	void mirror(ControlPanel* cp) { m_mirror = cp; }
	ControlPanel* mirror()        { return m_mirror; }
	
	uint num_controls() { return m_controls.size(); }

	pair<int,int> ideal_size() { return m_ideal_size; }
	
	// Callbacks for ControlGroup
	void value_changed(const string& port_path, float val);
	void range_changed(const string& port_path, float min, float max);

	inline void set_control(const string& port_path, float value);
	void set_range_min(const string& port_path, float value);
	void set_range_max(const string& port_path, float value);

private:
	void all_voices_selected();
	void specific_voice_selected();
	void voice_selected();

	PatchController* m_patch_controller;
	NodeModel*       m_node_model;
	ControlPanel*    m_mirror;
	bool             m_callback_enabled;
	
	pair<int,int> m_ideal_size;

	vector<ControlGroup*>    m_controls;
	Gtk::VBox*               m_control_box;
	Gtk::Box*                m_voice_control_box;
	Gtk::RadioButton*        m_all_voices_radio;
	Gtk::RadioButton*        m_specific_voice_radio;
	Gtk::SpinButton*         m_voice_spinbutton;
};


/** Set a port on this panel to a certain value.
 *
 * Profiling has shown this is performance critical.  Needs to be made
 * faster.
 */
inline void
ControlPanel::set_control(const string& port_path, const float val)
{
	m_callback_enabled = false;
	ControlGroup* cg   = NULL;
	
	for (vector<ControlGroup*>::iterator i = m_controls.begin(); i != m_controls.end(); ++i) {
		cg = (*i);
		if (cg->port_model()->path() == port_path) {
			cg->set_value(val);
			m_callback_enabled = true;
			return;
		}
	}
	
	cerr << "[ControlPanel::set_control] Unable to find control " << port_path << endl;
	m_callback_enabled = true;
}


} // namespace OmGtk

#endif // CONTROLPANEL_H
